import { cookie } from 'cookie_js';

import { getI18N } from 'lang/i18n';

export const ACTION_SET_LANGUAGE = 'set language';

/**
 * Sets the language translation to use for display to the user.
 *
 * @param {I18n} lang
 * @return {Object}
 */
export function setLanguage(lang) {
    return {
        type: ACTION_SET_LANGUAGE,
        lang,
    };
}

/**
 * Loads the language indicated by the given code.
 *
 * @param {String} langCode
 * @return {Function}
 */
export function loadLanguage(langCode) {
    return function(dispatch, getState) {
        const { lang } = getState();

        if (lang.shortCode === langCode) {
            return Promise.resolve();
        }

        return getI18N(langCode).then(setLanguage).then(dispatch);
    };
}

/**
 * Attempts to load a default language for the current user, based on a number
 * of fallback options in order of decreasing priority:
 *   - the cookie set on the page by web-client
 *   - the language configured by modern browsers
 *   - the language configured by slightly older browsers
 *   - the language configured by Internet Explorer
 *   - the language defined as a fallback language
 *
 * @param {?Object} navigator
 * @param {?String} fallbackLang
 * @return {Function}
 */
export function loadDefaultLang(navigator, fallbackLang) {
    const langToLoad = (
        cookie.get('language', null) ||
        (Array.isArray(navigator.languages) && navigator.languages[0]) ||
        navigator.language ||
        navigator.userLanguage ||
        fallbackLang
    );

    return loadLanguage(langToLoad);
}
