import { localStore } from 'util/storage';
import { WAITING, PAUSE, PLAYING, ENDED } from '../backend/events/media-event';
import { clearCollection } from '../actions/collection';
import { PLAYER_SITE, PLAYER_POPOUT, PLAYER_TWITCH_EVERYWHERE } from 'util/player-type';
import { parse } from 'util/params';
import extend from 'lodash/extend';
import { stringify } from 'query-string';
import { setStream, TYPE_VIDEO } from './stream';
import { emitTransitionToCollection, emitTransitionToRecommendedVOD } from 'actions/event-emitter';

export const ACTION_CLEAR_QUALITY_RESTRICTED_ERROR = 'clear error quality restricted';
export const ACTION_ENDED = 'set player ended';
export const ACTION_INITIALIZE_PLAYBACK_SETTINGS = 'initialize playback settings';
export const ACTION_PAUSE = 'set player pause';
export const ACTION_PLAYING = 'set player playing';
export const ACTION_QUALITY_RESTRICTED_ERROR = 'error quality restricted';
export const ACTION_SET_AUTOPLAY_STATE = 'set autoplay state';
export const ACTION_SET_LOADING = 'set loading state';
export const ACTION_PLAYER_MUTED = 'player muted';
export const ACTION_VOLUME_CHANGED = 'volume changed';
export const ACTION_UPDATE_BUFFER_VALUES = 'update buffer values';
export const ACTION_UPDATE_CURRENT_TIME = 'update current time';
export const ACTION_UPDATE_PLAYBACK_DURATION = 'update playback duration';
export const ACTION_WAITING = 'set player waiting';
export const ACTION_CONTENT_IS_SHOWING = 'content is showing';
export const ACTION_SET_TRANSITION_FUNCTION = 'set VOD transition function';
export const ACTION_PLAYBACK_RATE_CHANGED = 'playback rate changed';
export const ACTION_INCREMENT_BUFFER_EMPTIES = 'increment bufferEmpties count';
export const ACTION_PLAYER_SEEKING = 'player seeking';
export const ACTION_PLAYER_SEEKED = 'player seeked';
export const ACTION_SET_START_TIME = 'set start time';
export const INVALID_PLAYBACK_ACTION = 'invalid playback action';

/**
 * sets the initial playback state
 *
 * @return {Action} action
 */
export function initializePlaybackSettings() {
    return function(dispatch, getState) {
        const { playback } = getState();
        dispatch({
            type: ACTION_INITIALIZE_PLAYBACK_SETTINGS,
            playback: {
                muted: localStore.get('muted', playback.muted),
                volume: localStore.get('volume', playback.volume),
            },
        });
    };
}

/**
 * Clears a restricted quality error.
 *
 * @return {Object}
 */
export function clearQualityRestrictedError() {
    return {
        type: ACTION_CLEAR_QUALITY_RESTRICTED_ERROR,
    };
}

/**
 * Sets a restricted quality error.
 *
 * @return {Object}
 */
export function qualityIsRestricted() {
    return {
        type: ACTION_QUALITY_RESTRICTED_ERROR,
    };
}

/**
 * Updates the duration of the current stream
 *
 * @param {Number} duration Duration of the current stream, in seconds
 * @return {Object}
 */
export function updateDuration(duration) {
    return {
        type: ACTION_UPDATE_PLAYBACK_DURATION,
        playback: {
            duration,
        },
    };
}

/**
 * Updates the current playback state
 *
 * @param {String} state State must be a media-event state, one of:
 * PLAYING, WAITING, PAUSE or ENDED
 */
export function updatePlaybackState(state) {
    switch (state) {
    case PLAYING:
        return {
            type: ACTION_PLAYING,
        };
    case WAITING:
        return {
            type: ACTION_WAITING,
        };
    case PAUSE:
        return {
            type: ACTION_PAUSE,
        };
    case ENDED:
        return {
            type: ACTION_ENDED,
        };
    }
}

/**
 * Updates the autoplay state of the player
 *
 * @param {Boolean} whether autoplay is enabled or not
 * @return {Object}
 */
export function setAutoplay(autoplay) {
    return {
        type: ACTION_SET_AUTOPLAY_STATE,
        playback: {
            autoplay,
        },
    };
}

/**
 * Sets the muted state of the player.
 *
 * @param {Boolean} muted
 * @return {Action}
 */
export function playerMuted(muted) {
    return {
        type: ACTION_PLAYER_MUTED,
        muted,
    };
}

/**
 * Sets the volume level for the player.
 *
 * @param {Number} volume Volume, as a float between 0 and 1.
 * @return {Action}
 */
export function volumeChanged(volume) {
    return {
        type: ACTION_VOLUME_CHANGED,
        // Clamp the volume to the range [0, 1]
        volume: Math.max(0, Math.min(1, volume)),
    };
}

/**
 * Displays Loading overlay based on player's readyState
 *
 * @param {Boolean} isLoading
 * @return {Action}
 */
export function setLoading(isLoading) {
    return {
        type: ACTION_SET_LOADING,
        isLoading,
    };
}

/**
 * Updates the currentTime property on playback
 *
 * @param {Integer} current time in seconds
 * @return {Action}
 */
export function updateCurrentTime(currentTime) {
    return {
        type: ACTION_UPDATE_CURRENT_TIME,
        currentTime,
    };
}

/**
 * Updates the buffer values
 *
 * @param {Float} start time of the buffer
 * @param {Float} end time of the buffer
 * @return {Action}
 */
export function updateBufferValues(start, end) {
    return {
        type: ACTION_UPDATE_BUFFER_VALUES,
        start,
        end,
    };
}

/**
 * Increments bufferEmpties property on playback
 *
 * @return {Action}
 */
export function incrementBufferEmpties() {
    return {
        type: ACTION_INCREMENT_BUFFER_EMPTIES,
    };
}

/*
 * Sets contentShowing state to true
 *
 * @return {Object}
 */
export function contentIsShowing() {
    return {
        type: ACTION_CONTENT_IS_SHOWING,
    };
}

/**
 * Dispatched to notify that the playback rate has changed
 *
 * @param {number} playbackRate - the new playback rate
 * @return {Object}
 */
export function playbackRateChanged(playbackRate) {
    return  {
        type: ACTION_PLAYBACK_RATE_CHANGED,
        playbackRate,
    };
}

/**
 * Dispatched to set the initial start time of a vod
 * after it has loaded. Will only accept a number > 0.
 *
 * @param {number} playbackRate - the new playback rate
 * @return {Object}
 */
export function setStartTime(startTime) {
    let action = { type: INVALID_PLAYBACK_ACTION };

    if (Number.isFinite(startTime) && startTime >= 0) {
        action = {
            type: ACTION_SET_START_TIME,
            startTime,
        };
    }

    return action;
}

/**
 * Displays to notify that we are seeking
 *
 * @return {Action}
 */
export function playerSeeking() {
    return {
        type: ACTION_PLAYER_SEEKING,
    };
}

/**
 * Dispatched to notify that the player has seeked
 *
 * @param {number} currentTime - time of the player after the seek
 * @return {Object}
 */
export function playerSeeked(currentTime) {
    return {
        type: ACTION_PLAYER_SEEKED,
        currentTime,
    };
}

/**
 * Transition to a VOD selected from in-player recommendations, dependent on player environment
 *
 * @param {string} vodId - video id of VOD to transition to
 */
export function selectRecommendedVideo(vodId) {
    return function(dispatch, getState) {
        const { env } = getState();

        switch (env.playerType) {
        case PLAYER_SITE:
            dispatch(emitTransitionToRecommendedVOD(vodId));
            break;
        case PLAYER_POPOUT: // eslint-disable-line no-case-declarations
            const params = getRecommendedVideoParams(vodId, getState);
            navigateToVideo(params, getState);
            break;
        case PLAYER_TWITCH_EVERYWHERE:
            dispatch(emitTransitionToRecommendedVOD(vodId));
            dispatch(clearCollection());
            dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: vodId,
            }));
            break;
        default:
            dispatch(clearCollection());
            dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: vodId,
            }));
        }
    };
}

/**
 * Transition to a VOD selected from a collection, dependent on player environment
 *
 * @param {string} vodId - video id of VOD to transition to
 * @param {string} collectionId - collection id of the collection associated with the video
 */
export function selectCollectionVideo(vodId, collectionId) {
    return function(dispatch, getState) {
        const { env, screenMode } = getState();

        if (env.playerType === PLAYER_SITE) {
            // VP-2065 player sometimes does not persist when transitioning between VODs,
            // but we want to keep the fullscreen experience. For now, allow users to have
            // seamless fullscreen experience and do not do Ember transition.
            if (screenMode.isFullScreen) {
                dispatch(setStream({
                    contentType: TYPE_VIDEO,
                    contentId: vodId,
                }));
            } else {
                dispatch(emitTransitionToCollection(collectionId, vodId));
            }
        } else if (env.playerType === PLAYER_POPOUT) {
            const params = getCollectionVideoParams(vodId, getState);
            navigateToVideo(params, getState);
        } else {
            dispatch(setStream({
                contentType: TYPE_VIDEO,
                contentId: vodId,
            }));
        }
    };
}

function navigateToVideo(params, getState) {
    const { window: windowObj } = getState();
    windowObj.location.search = `?${stringify(params)}`;
}

/**
 * Builds player URL when an in-player recommendation is selected
 */
function getRecommendedVideoParams(videoId, getState) {
    const { window: windowObj } = getState();
    const urlParams = _getBaseParameters(windowObj);
    return extend({}, { video: videoId }, urlParams);
}

/**
 * Builds player URL when a collection item is selected or being transitioned to
 */
function getCollectionVideoParams(videoId, getState) {
    const { window: windowObj, collection } = getState();
    const urlParams = _getBaseParameters(windowObj);
    return extend({}, {
        video: videoId,
        collection: collection.pendingRequest.collectionId || collection.id,
    }, urlParams);
}

function _getBaseParameters(windowObj) {
    const urlParams = parse(windowObj.location.search.slice(1));
    delete urlParams.channel;
    delete urlParams.video;
    delete urlParams.collection;
    delete urlParams.time;
    return urlParams;
}
