import includes from 'lodash/includes';
import { ACTION_QUALITY_RESTRICTED_ERROR } from './playback';
import { localStore } from 'util/storage';
import find from 'lodash/find';

export const ACTION_SELECT_QUALITY = 'select quality';
export const ACTION_SET_CURRENT_QUALITY = 'set current quality';
export const ACTION_SET_QUALITIES = 'set qualities';
export const ACTION_SET_ABS_AVAILABILITY = 'abs is availabile';

export const KEY_AUTO_QUALITY_FORCED = 'auto-quality-forced-v2';
export const KEY_SEEN_AUTO_QUALITY_NOTIFICATION = 'has-seen-abs-notification';
export const QUALITY_AUTO = 'auto';
export const QUALITY_AUTO_OBJECT = Object.freeze({
    bitrate: 0,
    resolution: '0x0',
    group: 'auto',
    name: 'Auto',
});
export const DEFAULT_STREAM_FORMAT = 'medium';
export const DEFAULT_STREAM_BITRATE_IN_BPS = 992000;
export const CLIP_RESOLUTION_MAP = Object.freeze({
    1080: '1920x1080',
    720: '1280x720',
    480: '854x480',
    360: '640x360',
});

/**
 * setABSAvailablity lets UI know if ABS is available
 *
 * @param {Boolean} absAvailable
 * @return {Object}
 */
export function setABSAvailability(absAvailable) {
    return {
        type: ACTION_SET_ABS_AVAILABILITY,
        absAvailable,
    };
}

/**
 * Initialize preferred quality settings
 *
 * @return {Action}
 */
export function initializeQuality() {
    return function(dispatch, getState) {
        const { quality } = getState();

        dispatch({
            type: ACTION_SELECT_QUALITY,
            quality: localStore.get('quality', quality.selected),
            bitrate: localStore.get('quality-bitrate', quality.bitrate),
        });
    };
}

/**
 * Selects a new quality for the player
 * calling this method indicates that the quality of the stream will be changing soon.
 *
 * @param {String} value new quality group id
 * @param {Number} bitraet new quality bitrate
 * @return {Action} action
 */
export function selectQuality(value, bitrate = DEFAULT_STREAM_BITRATE_IN_BPS) {
    return function(dispatch, getState) {
        const { analyticsTracker, manifestInfo, quality, stream } = getState();

        if (stream.restrictedBitrates.indexOf(value) !== -1) {
            dispatch({
                type: ACTION_QUALITY_RESTRICTED_ERROR,
            });
            return;
        }
        /* eslint-disable camelcase */
        analyticsTracker.trackEvent('quality_change_request', {
            prev_quality: quality.current,
            new_quality: value,
            serving_id: manifestInfo.serving_id,
        });
        /* eslint-enable camelcase */

        dispatch({
            type: ACTION_SELECT_QUALITY,
            quality: value,
            bitrate: bitrate,
        });
    };
}

/**
 * Sets the user's preferred quality to the given value.
 *
 * @param {Object} quality the object representing the quality preferred by the user
 * @return {Action}
 */
export function setPreferredQuality(quality) {
    return function() {
        localStore.set('quality', quality.group);
        localStore.set('quality-bitrate', quality.bandwidth);
    };
}

/**
 * Sets the current quality of the player
 * calling this method indicates that the quality of the stream has recently changed.
 * (this field should always represent the actual quality of the stream.)
 *
 * @param {String} quality quality group ID
 * @param {Boolean} isAuto - true if quality is set via ABS
 * @return {Action} action
 */
export function setCurrentQuality(quality, isAuto = false) {
    return {
        type: ACTION_SET_CURRENT_QUALITY,
        quality,
        isAuto,
    };
}

/**
 * Matches experimental transcode quality with the selected quality
 * Remove with VP-1957
 *
 * @param {String} selectedQuality
 * @param {Array<Object>} qualities
 * @return {Object | Undefined}
 */
export function _getMatchingTranscodeQuality(selectedQuality, qualities) {
    return find(qualities, quality => {
        return `${quality.group}2` === selectedQuality ||
            `${selectedQuality}2` === quality.group;
    });
}

/**
 * Sets the list of qualities available to the player
 * NOTE: This may include qualities the user can't view, eg. because they are restricted.
 *
 * @param {Array<Object>} the list of qualities
 *
 * @return {Action}
 */
export function setQualities(qualities) {
    return function(dispatch, getState) {
        const { quality, stream } = getState();
        const allowedQualities = qualities.filter(q => !includes(stream.restrictedBitrates, q.group));
        if (
            allowedQualities.length > 0 &&
            !allowedQualities.some(q => q.group === quality.selected) &&
            quality.selected !== QUALITY_AUTO
        ) {
            // Instead of choosing 0th default, choose a max bandwidth lesser than or equal to default.
            const sortedQualities = allowedQualities.sort((a, b) => b.bandwidth - a.bandwidth);
            // If user is switching between transcode groups, match on groupname
            const matchedExperimentalQuality = _getMatchingTranscodeQuality(quality.selected, sortedQualities);
            // If user supplied completely invalid quality, match on bitrate
            const nearestQuality = find(sortedQualities, currentQuality => {
                return currentQuality.bandwidth <= quality.bitrate;
            });
            // If no quality found choose the last quality in the sorted list
            const { group, bandwidth } = matchedExperimentalQuality ||
                nearestQuality ||
                sortedQualities[sortedQualities.length - 1];

            dispatch({
                type: ACTION_SELECT_QUALITY,
                quality: group,
                bitrate: bandwidth,
            });
        }
        dispatch({
            type: ACTION_SET_QUALITIES,
            qualities,
        });
    };
}

/**
 * Sets the list of qualities available to the player based on clip status
 *
 * @param {Array<Object>} the list of qualities retrieved from clip endpoint
 *
 * @return {Action}
 */
export function setClipQualities(clipQualities) {
    const qualities = clipQualities.map(qualityInfo => {
        return {
            name: `${qualityInfo.quality}p`,
            group: `${qualityInfo.quality}p${qualityInfo.frame_rate}`,
            bitrate: '', // not used for clips
            resolution: CLIP_RESOLUTION_MAP[qualityInfo.quality],
            source: qualityInfo.source,
        };
    });

    return {
        type: ACTION_SET_QUALITIES,
        qualities,
    };
}
