import { krakenRequest } from 'api';
import { PINEAPPLE } from './../experiments';
import assign from 'lodash/assign';
import { stringify } from 'query-string';

export const ACTION_SET_RECOMMENDED_VODS = 'set VOD recommendations';
export const ACTION_CLEAR_RECOMMENDED_VODS = 'clear VOD recommendations';
export const ACTION_SET_RECOMMENDED_VODS_WATCHED = 'set VOD recommendations watched';
export const ACTION_SET_NUM_VODS_VISIBLE = 'set number of VOD recommendations visible';
export const ACTION_SET_FETCHING_STATUS = 'set fetching recommendations status';

export const UNFETCHED = 'unfetched'; // state before fetching recommendations
export const FETCHING = 'fetching'; // state while recommendations are being fetched
export const FETCHED = 'fetched'; // state after recommendations have been fetched

// maximum number of recommendations to be shown to the user
export const MAX_RECOMMENDED_VODS_VISIBLE = 32;

// Minimum length of a VOD watched to consider being a watched recommendation
export const MINIMUM_WATCHED_LENGTH = 300; // 5 minutes

// Amount of time before the end of a VOD when we should fetch recommendations, in ms.
export const FETCH_VODS_THRESHOLD = 12000;

export const TEN_PERCENT = 0.1;

// Maximum amount of time to delay fetching VOD recommendations
export const MAX_JITTER_DELAY = 5000;

// Recommended VOD types
export const SIMILAR_VODS = 'similar';
export const CHANNEL_VODS = 'channel';

// Recommendation Types
export const POST_VOD_RECOMMENDATIONS_TYPE = 'post vod recommendations';
export const OFFLINE_RECOMMENDATIONS_TYPE = 'offline recommendations';

/**
 * Attempts to fetch VOD recommendations for current stream. Sets found VOD recommendations in store.
 * VOD recommendations are in descending order of most recommended.
 *
 * @param {Number} number of VOD recommendations to fetch
 * @return {Action} function to run to get recommendations
 */
export function fetchRecommendedVODs(recommendationsType = POST_VOD_RECOMMENDATIONS_TYPE) {
    return function(dispatch, getState) {
        const channelName = getState().streamMetadata.channel.name;
        if (
            getState().recommendations.status !== UNFETCHED ||
            channelName === ''
        ) {
            return Promise.resolve();
        }

        // Add random delay for this user
        var jitter = Math.floor(Math.random() * MAX_JITTER_DELAY);

        dispatch({
            type: ACTION_SET_FETCHING_STATUS,
            status: FETCHING,
        });
        return new Promise(function(resolve) {
            getState().window.setTimeout(resolve, jitter);
        }).then(() => {
            return getState().experiments.get(PINEAPPLE);
        }).then(getSimilarVideos => {
            if (getSimilarVideos === 'yes') {
                return fetchSimilarVideos(getState().stream.videoId);
            }
            return Promise.reject();
        }).
        catch(() => {
            return fetchChannelVideos(channelName, MAX_RECOMMENDED_VODS_VISIBLE + 1);
        }).
        then(videos => {
            const { resumeWatch, streamMetadata } = getState();

            const normalizedRecommendedVODs = videos.
                filter(video => video.broadcast_id !== streamMetadata.broadcastID).
                sort((video1, video2) => {
                    const video1Watched = isWatched(video1, resumeWatch.times);
                    const video2Watched = isWatched(video2, resumeWatch.times);

                    if (!video1Watched && video2Watched) {
                        return -1;
                    } else if (video1Watched && !video2Watched) {
                        return 1;
                    }

                    // Both VODs have same watched state
                    return (new Date(video2.created_at).getTime()) - (new Date(video1.created_at).getTime());
                }).
                map(normalizeV3Video);

            dispatch(setRecommendedVODs(normalizedRecommendedVODs, recommendationsType));
        });
    };
}

function fetchChannelVideos(channelName, numberOfVideos) {
    const args = {
        limit: numberOfVideos,
        // eslint-disable-next-line camelcase
        broadcast_type: 'archive',
    };

    return krakenRequest(`channels/${channelName}/videos?${stringify(args)}`).
        then(({ videos }) => {
            return videos.map(v => assign(v, { recommendationType: CHANNEL_VODS }));
        });
}

function fetchSimilarVideos(videoID) {
    /* eslint-disable camelcase */
    return krakenRequest(`videos/similar/${videoID}`).
        then(({ similar_videos }) => {
            if (similar_videos.length === 0) {
                return Promise.reject();
            }

            return similar_videos.map(v => assign(v, { recommendationType: SIMILAR_VODS }));
        });
    /* eslint-enable camelcase */
}

function normalizeV3Video(video) {
    return {
        recommendationType: video.recommendationType,
        creationDate: video.created_at,
        thumbnailURL: (video.thumbnails.length > 0) ? video.thumbnails[0].url : '',
        title: video.title,
        channelName: video.channel.display_name,
        duration: video.length,
        id: video._id,
        game: video.game,
    };
}

/**
 * Based off of resumetime on a VOD, determine if a VOD has been watched.
 *
 * @param {Number} length of VOD
 * @param {Number} the resume time defined on the VOD in resume watch
 */
export function isWatched(video, resumeTimes) {
    if (!resumeTimes.hasOwnProperty(video._id)) {
        return false;
    }

    const resumeTime = resumeTimes[video._id];
    if (resumeTime === 0) { // Resumewatch marks fully watched VODs with t = 0
        return true;
    }

    const timeThreshold = Math.max(MINIMUM_WATCHED_LENGTH, TEN_PERCENT * video.length);

    return video.length > MINIMUM_WATCHED_LENGTH && resumeTime > timeThreshold;
}

/**
 * Updates recommended VODs with provided information.
 *
 * @param {Array<Object>} recommended VODs
 * @return {Action} action
 */
export function setRecommendedVODs(videos, recommendationsType) {
    return {
        type: ACTION_SET_RECOMMENDED_VODS,
        recommendationsType,
        videos,
    };
}

/**
 * Clears recommended VODs
 * (we've loaded a new video and maybe they dont't make sense anymore)
 *
 * @return {Action} action
 */
export function clearRecommendedVODs() {
    return {
        type: ACTION_CLEAR_RECOMMENDED_VODS,
    };
}

/*
 * Sets the number of recommendations that should be visible to the viewer.
 * this depends on the size of the player
 *
 * @return {Action} Action
 */
export function setVODVisibility(numVids) {
    return {
        type: ACTION_SET_NUM_VODS_VISIBLE,
        numVids,
    };
}
