import { stringify } from 'query-string';
import { localStore } from 'util/storage';
import { resumewatchingHost } from 'settings';
import { CONTENT_MODE_VOD } from '../stream/twitch-vod';
import { CONTENT_MODE_LIVE } from '../stream/twitch-live';

export const ACTION_SET_RESUME_TIMES = 'set resume times';
export const ACTION_VOD_INIT_RESUME = 'initialize vod resume';
export const ACTION_VOD_CANCEL_RESUME = 'cancel vod resume';
export const ACTION_VOD_SET_RESUME_TIME = 'set vod resume time';
export const ACTION_LIVESTREAM_CANCEL_RESUME = 'cancel livestream resume';
export const ACTION_LIVESTREAM_SET_RESUME_TIME = 'set livestream resume time';
export const ACTION_VOD_POST_BACKEND_TIME = 'post vod backend time';
export const ACTION_VOD_SET_USER = 'set user';
export const ACTION_VOD_SET_IS_SEEKED = 'set isSeeked';

const KEY_VOD_RESUME_TIMES = 'vodResumeTimes';
const KEY_VOD_RESUME_WATCHEDS = 'vodResumeWatcheds';
const KEY_LIVESTREAM_RESUME_TIMES = 'livestreamResumeTimes';

/**
 * Initializes VOD resume watching state from persisted values in localStorage and initialization time.
 *
 * @return {Action}
 */
export function initVodResume() {
    return {
        type: ACTION_VOD_INIT_RESUME,
        times: localStore.get(KEY_VOD_RESUME_TIMES, {}),
        watch: localStore.get(KEY_VOD_RESUME_WATCHEDS, {}),
        streamTimes: localStore.get(KEY_LIVESTREAM_RESUME_TIMES, {}),
        lastTimeStamp: 0,
        userId: null,
        isSeeked: false,
    };
}

/**
* Sets the user id
*
* @param {String} userId
* @return {Action}
**/
export function setUser(userId) {
    return {
        type: ACTION_VOD_SET_USER,
        userId,
    };
}

/**
* Sets the user id
*
* @param {String} userId
* @return {Action}
**/
export function setIsSeeked(isSeeked) {
    return {
        type: ACTION_VOD_SET_IS_SEEKED,
        isSeeked,
    };
}

/**
*
*  Sets the last timestamp on the resumewatching service through an AJAX PUT request.
*  @param {String} userId
*  @param {String} videoId
*  @param {Int} channelUserID
*  @param {Number} time
*  @param {String} type
*  @return {Promise}
*/
function putBackendResumeTime(userId, videoId, channelUserID, time, type) {
    const data = {
        video_id: videoId, // eslint-disable-line camelcase
        channel_user_id: channelUserID, // eslint-disable-line camelcase
        position: Math.floor(time),
        type,
    };

    return fetch(
        `${resumewatchingHost}user-video?id=${userId}`,
        {
            method: 'PUT',
            credentials: 'include',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: stringify(data),
        }
    );
}

/**
*
*  Retrieves the timestamps on the resumewatching service through an AJAX GET request.
*  @param {String} userId
*  @return {Promise}
*/
export function getResumeTimes(userId) {
    if (!userId) {
        return Promise.reject();
    }

    return fetch(
        `${resumewatchingHost}user?id=${userId}`,
        {
            credentials: 'include',
        }
    ).then(response => response.json());
}

export function setResumeTimes(resumeTimes) {
    return {
        type: ACTION_SET_RESUME_TIMES,
        resumeTimes,
    };
}

/**
 * Checks the update time offset has passed to make the initial update
 * @param {Number} initTime
 * @param {Number} initUpdateOffset
 * @return {Boolean}
 */
function initOffsetPassed(initTime, currentTime, initUpdateOffset) {
    return currentTime - initTime > initUpdateOffset;
}

/**
 * Checks the update time interval has passed to make the next update
 * @param {Number} currentTimeStamp
 * @param {Number} lastTimeStamp
 * @param {Number} updateInterval
 * @return {Boolean}
 */
function timeIntervalPassed(currentTimeStamp, lastTimeStamp, updateInterval) {
    if (lastTimeStamp === 0 && currentTimeStamp !== 0) {
        // always update on the first update
        return true;
    }
    return Math.abs(currentTimeStamp - lastTimeStamp) > updateInterval;
}

/**
 *
 * Sets the lastTimeStamp on the resumewatching service when a VOD is playing.
 * @param {String} videoId
 * @param {Int} channelUserID
 * @param {Number} time
 * @return {Action}
 */
function setBackendVodResumeTime(videoId, channelUserID, time) {
    return function(dispatch, getState) {
        const { resumeWatch, window, analytics } = getState();
        if (resumeWatch.userId &&
            initOffsetPassed(analytics.playSessionStartTime, window.Date.now(), resumeWatch.initUpdateOffset) &&
            timeIntervalPassed(time, resumeWatch.lastTimeStamp, resumeWatch.updateInterval)) {
            putBackendResumeTime(resumeWatch.userId, videoId, channelUserID, time, CONTENT_MODE_VOD);
            dispatch({
                type: ACTION_VOD_POST_BACKEND_TIME,
                lastTimeStamp: Math.floor(time),
            });
        }
    };
}

/**
 *
 * Sets the lastTimeStamp on the resumewatching service when a livestream is playing.
 * @param {String} broadcastID
 * @param {Int} channelUserID
 * @param {Number} time
 * @return {Action}
 */
function setBackendLivestreamResumeTime(broadcastID, channelUserID, time) {
    return function(dispatch, getState) {
        const { resumeWatch, window, analytics } = getState();
        if (resumeWatch.userId &&
            initOffsetPassed(analytics.playSessionStartTime, window.Date.now(), resumeWatch.initUpdateOffset) &&
            timeIntervalPassed(time, resumeWatch.lastTimeStamp, resumeWatch.updateInterval)) {
            putBackendResumeTime(resumeWatch.userId, broadcastID, channelUserID, time, CONTENT_MODE_LIVE);
            dispatch({
                type: ACTION_VOD_POST_BACKEND_TIME,
                lastTimeStamp: Math.floor(time),
            });
        }
    };
}

/**
 * Set the time to which a VOD should automatically seek when watched again.
 *
 * @param {String} videoID
 * @param {Int} channelUserID
 * @param {Number} time
 * @return {Action}
 */
export function setVodResumeTime(videoID, channelUserID, time) {
    return function(dispatch, getState) {
        const { resumeWatch } = getState();
        if (resumeWatch.userId) {
            dispatch(setBackendVodResumeTime(videoID, channelUserID, time));
        } else {
            dispatch({
                type: ACTION_VOD_SET_RESUME_TIME,
                videoID: videoID,
                time: time,
            });

            // TODO serializing the reducer state after a dispatch seems like the
            // wrong solution; find a more robust way to update a store and persist
            // via localStorage.
            persist(getState());
        }
    };
}

/**
 * Removes the resume time for a given VOD.
 *
 * @param {String} videoID
 * @param {Int} channelUserID
 * @return {Action}
 */
export function cancelVodResumeTime(videoID, channelUserID) {
    return function(dispatch, getState) {
        const { resumeWatch } = getState();
        if (resumeWatch.userId) {
            dispatch(setBackendVodResumeTime(videoID, channelUserID, 0));
        } else {
            dispatch({
                type: ACTION_VOD_CANCEL_RESUME,
                videoID,
            });

            // TODO serializing the reducer state after a dispatch seems like the
            // wrong solution; find a more robust way to update a store and persist
            // via localStorage.
            persist(getState());
        }
    };
}

/**
 * Set the time to which a livestream should automatically seek when watched again.
 *
 * @param {String} broadcastID
 * @param {Int} channelUserID
 * @param {Number} time
 * @return {Action}
 */
export function setLivestreamResumeTime(broadcastID, channelUserID, time) {
    return function(dispatch, getState) {
        const { resumeWatch } = getState();
        if (resumeWatch.userId) {
            dispatch(setBackendLivestreamResumeTime(broadcastID, channelUserID, time));
        } else {
            dispatch({
                type: ACTION_LIVESTREAM_SET_RESUME_TIME,
                broadcastID,
                time,
            });

            // TODO serializing the reducer state after a dispatch seems like the
            // wrong solution; find a more robust way to update a store and persist
            // via localStorage.
            persist(getState());
        }
    };
}

/**
 * Removes the resume time for a given livestream.
 *
 * @param {String} broadcastID
 * @param {Int} channelUserID
 * @return {Action}
 */
export function cancelLivestreamResumeTime(broadcastID, channelUserID) {
    return function(dispatch, getState) {
        const { resumeWatch } = getState();
        if (resumeWatch.userId) {
            dispatch(setBackendLivestreamResumeTime(broadcastID, channelUserID, 0));
        } else {
            dispatch({
                type: ACTION_LIVESTREAM_CANCEL_RESUME,
                broadcastID,
            });

            // TODO serializing the reducer state after a dispatch seems like the
            // wrong solution; find a more robust way to update a store and persist
            // via localStorage.
            persist(getState());
        }
    };
}
/**
 * Persists the state of the VOD resume watching data.
 * @private
 *
 * @param {Object} state
 *        Current state of the reducer that needs to be persisted.
 */
function persist(state) {
    localStore.set(KEY_VOD_RESUME_TIMES, state.resumeWatch.times);
    localStore.set(KEY_VOD_RESUME_WATCHEDS, state.resumeWatch.watch);
    localStore.set(KEY_LIVESTREAM_RESUME_TIMES, state.resumeWatch.streamTimes);
}
