import { channelInfo, streamInfo, videoInfo } from 'api';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { updateViewerCount } from 'actions/viewercount';
import assign from 'lodash/assign';

export const ACTION_SET_STREAMMETADATA = 'set streammetadata';
export const VIEWERS_NOT_SET = 0;

// 'exponential' backoff:              15s  > 30s  > 1min > 2min  > 5min  > 10min  > give up
export const FETCH_METADATA_TIMEOUT = [15000, 30000, 60000, 120000, 300000, 600000];

/**
 * Updates the state's streamMetadata with provided videoInfo
 * API response.
 *
 * @param {Object} videoInfo API response
 * @return {Object} action
 */
export function setVODStreamMetadata(videoInfo) {
    return {
        type: ACTION_SET_STREAMMETADATA,
        streamMetadata: normalizedVODVideoInfo(videoInfo),
    };
}

/**
 * Updates the state's streamMetadata with provided streamInfo
 * API response.
 *
 * @param {Object} info API response
 * @return {Object} action
 */
export function setLiveStreamMetadata(info) {
    return {
        type: ACTION_SET_STREAMMETADATA,
        streamMetadata: normalizedLiveStreamInfo(info),
    };
}

/**
 * Attempts to fetch metadata for current live stream. Upon failure (determined by
 * no stream metadata returned or stream metadata not for current stream), will
 * set up retries, at an exponential backoff.
 *
 * @return {Function} function to dispatch/run to fetch stream metadata (and retry if necessary)
 */
export function fetchLiveStreamMetadata(broadcastID) {
    return function(dispatch, getState) {
        if (!(getState().stream.contentType === CONTENT_MODE_LIVE)) {
            return;
        }

        if (getState().streamMetadata.broadcastID === broadcastID) {
            return;
        }

        const windowObj = getState().window;
        const channelName = getState().stream.channel;
        let retries = 0;

        return new Promise((resolve, reject) => {
            function getInfo() {
                if (channelName !== getState().stream.channel) {
                    reject();
                    return;
                }

                streamInfo(channelName).then(info => {
                    if (info.stream && info.stream._id === broadcastID) {
                        dispatch(updateViewerCount(info.stream.viewers));
                        dispatch(setLiveStreamMetadata(info.stream));
                        resolve(info);
                    } else if (retries < FETCH_METADATA_TIMEOUT.length) {
                        windowObj.setTimeout(getInfo, FETCH_METADATA_TIMEOUT[retries]);
                        retries += 1;
                    } else {
                        reject();
                    }
                });
            }

            getInfo();
        });
    };
}

/**
 * Given a LiveTwitchContentStream or VODTwitchContentStream, makes the appropriate API call for
 * metadata and dispatches the appropriate action to normalize and set the data.
 *
 * @param {Object} stream Stream object to fetch and set metadata for
 * @return {Function} function to dispatch
 */
export function fetchAndSetStreamMetadata(stream) {
    return function(dispatch, _) {
        if (stream.contentType === CONTENT_MODE_LIVE) {
            streamInfo(stream.channel).then(info => {
                if (info.stream) {
                    dispatch(setLiveStreamMetadata(info.stream));
                } else {
                    // we get here if the channel is offline
                    channelInfo(stream.channel).then(channelInfo => {
                        dispatch({
                            type: ACTION_SET_STREAMMETADATA,
                            streamMetadata: normalizedChannelInfo(channelInfo),
                        });
                    });
                }
            });
        } else if (stream.contentType === CONTENT_MODE_VOD) {
            videoInfo(stream.videoId).then(videoInfo => {
                dispatch(setVODStreamMetadata(videoInfo));
            });
        }
    };
}

export function fetchDeadLtvStreamMetadata(channelName) {
    return function(dispatch, _) {
        channelInfo(channelName).then(channelInfo => {
            dispatch({
                type: ACTION_SET_STREAMMETADATA,
                streamMetadata: assign({}, normalizedChannelInfo(channelInfo), {
                    streamType: 'dead_ltv',
                }),
            });
        });
    };
}

export function setClipStreamMetadata(clipInfo) {
    return {
        type: ACTION_SET_STREAMMETADATA,
        streamMetadata: assign({}, normalizedClipInfo(clipInfo)),
    };
}

/**
 * Normalizes api's videoInfo data into state's streamMetadata property
 *
 * @param {Object} api's videoInfo data
 * @return {Object} data in the form of state's streamMetadata property
 */
function normalizedVODVideoInfo(videoInfo) {
    return {
        broadcastID: videoInfo.broadcast_id,
        type: videoInfo.broadcast_type,
        name: videoInfo.title,
        channel: {
            displayName: videoInfo.channel.display_name,
            name: videoInfo.channel.name,
            id: videoInfo.channel._id,
            partner: videoInfo.channel.partner,
            status: videoInfo.status,
            videoBanner: videoInfo.channel.video_banner,
            logo: videoInfo.channel.logo,
            mature: videoInfo.channel.mature,
        },
        createdAt: videoInfo.created_at,
        game: videoInfo.game,
        preview: {
            template: videoInfo.preview,
        },
        url: videoInfo.url,
        viewers: VIEWERS_NOT_SET,
    };
}

/**
 * Normalizes api's streamInfo data into state's streamMetadata property
 *
 * @param {Object} api's streamInfo stream object
 * @return {Object} data in the form of state's streamMetadata property
 */
function normalizedLiveStreamInfo(streamData) {
    return {
        broadcastID: streamData._id,
        type: 'live',
        streamType: streamData.stream_type,
        name: '',
        channel: {
            displayName: streamData.channel.display_name,
            name: streamData.channel.name,
            id: streamData.channel._id,
            partner: streamData.channel.partner,
            status: streamData.channel.status,
            videoBanner: streamData.channel.video_banner,
            logo: streamData.channel.logo,
            mature: streamData.channel.mature,
        },
        createdAt: streamData.created_at,
        game: streamData.game,
        preview: {
            template: streamData.preview.template,
        },
        url: streamData.channel.url,
        viewers: streamData.viewers,
    };
}

/**
 * Normalizes api's channelInfo data into state's streamMetadata property
 *
 * @param {Object} api's channelInfo data object
 * @return {Object} data in the form of state's streamMetadata property
 */
function normalizedChannelInfo(channelInfo) {
    return {
        channel: {
            displayName: channelInfo.display_name,
            name: channelInfo.name,
            id: channelInfo._id,
            partner: channelInfo.partner,
            status: channelInfo.status,
            videoBanner: channelInfo.video_banner,
            logo: channelInfo.logo,
            mature: channelInfo.mature,
        },
        game: channelInfo.game,
        url: channelInfo.url,
    };
}

/**
 * Normalizes api's clipInfo data into state's streamMetadata property
 *
 * @param {Object} clipinfo - API's clipInfo data object
 * @return {Object} data in the form of state's streamMetadata property
 */
function normalizedClipInfo(clipInfo) {
    const baseClipProperties = {
        id: clipInfo.tracking_id,
        channel: {
            channelUrl: clipInfo.broadcaster.channel_url,
            displayName: clipInfo.broadcaster.display_name,
            name: clipInfo.broadcaster.name,
            id: clipInfo.broadcaster.id,
            logo: clipInfo.broadcaster.logo,
        },
        curator: {
            channelUrl: clipInfo.curator.channel_url,
            displayName: clipInfo.curator.display_name,
            name: clipInfo.curator.name,
            id: clipInfo.curator.id,
            logo: clipInfo.curator.logo,
        },
        game: clipInfo.game,
        url: clipInfo.url,
        title: clipInfo.title,
        thumbnails: clipInfo.thumbnails,
        createdAt: clipInfo.created_at,
        duration: clipInfo.duration,
        embedHtml: clipInfo.embed_html,
        embedUrl: clipInfo.embed_url,
        language: clipInfo.language,
        slug: clipInfo.slug,
        viewCount: clipInfo.views,
        clipHasVod: false,
        segments: clipInfo.segments,
    };

    const additionalClipProperties = {};

    if (clipInfo.vod) {
        additionalClipProperties.clipVideoId = clipInfo.vod.id;
        additionalClipProperties.clipVideoOffset = clipInfo.vod.offset;
        additionalClipProperties.clipVideoUrl = clipInfo.vod.url;
        additionalClipProperties.clipVideoPreviewImageUrl = clipInfo.vod.preview_image_url;
        additionalClipProperties.clipHasVod = true;
    }

    return assign(baseClipProperties, additionalClipProperties);
}

