import { REALTIME_QOS, FAST_BREAD } from '../experiments';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { ProvidedContentStream } from 'stream/provided';
import { ClipContentStream } from 'stream/clip';
import { VideoSourceContentStream } from 'stream/video-source';
import { krakenRequestv5, oauthToken } from '../api';
import { VIDEO_PLAY_LOAD_START, VIDEO_PLAY_OAUTH, VIDEO_PLAY_NAUTH } from '../analytics/analytics';
import { resetPlaySession } from './analytics';
import { fetchExtensions, subscribeToExtensionControl } from './extensions';
import { fetchAndSetStreamMetadata, setClipStreamMetadata } from './stream-metadata';
import { nullContentStream } from 'stream/null';
import { trackEvent } from 'actions/analytics-tracker';
import { setClipQualities } from 'actions/quality';
import { clipsAPIHost } from '../settings';
import { incrementClipViewCount } from 'ui/player-types/clips/api';

export const ACTION_SET_STREAM = 'set stream';
export const TYPE_CHANNEL = 'channel';
export const TYPE_VIDEO = 'video';
export const TYPE_PROVIDED = 'provided';
export const TYPE_CLIP = 'clip';
export const TYPE_VIDEO_SOURCE = 'video-source';

/**
 * Sets the current playback stream to the given channel or video.
 *
 * @param {String} contentType Type of stream being requested (i.e. "video" or "channel")
 * @param {String} contentId Channel name (e.g. "monstercat") or ID associated with the video (e.g. "v1234567890")
 * @return {Function}
 */
export function setStream(contentInfo) {
    return function(dispatch, getState) {
        dispatch(resetPlaySession());

        switch (contentInfo.contentType) {
        case TYPE_PROVIDED:
            return setProvidedStream(contentInfo, dispatch, getState);
        case TYPE_CLIP:
            return setClipStream(contentInfo, dispatch, getState);
        case TYPE_VIDEO_SOURCE:
            return setVideoSourceStream(contentInfo, dispatch, getState);
        case TYPE_CHANNEL:
        case TYPE_VIDEO:
            return setTwitchStream(contentInfo, dispatch, getState);
        }
    };
}

function setProvidedStream({ customerId, contentId }, dispatch, getState) {
    const { playback, usher } = getState();

    dispatch(trackEvent(VIDEO_PLAY_LOAD_START, {
        autoplay: playback.autoplay,
        customer_id: customerId, // eslint-disable-line camelcase
        content_id: contentId, // eslint-disable-line camelcase
    }));

    const stream = new ProvidedContentStream({
        contentId,
        customerId,
        usherHostOverride: usher.hostOverride,
    });

    dispatch({
        type: ACTION_SET_STREAM,
        stream,
    });
}

function setClipStream({ contentId }, dispatch, getState) {
    // TODO: VP-2679 update metrics with clips data
    const { playback } = getState();
    dispatch(trackEvent(VIDEO_PLAY_LOAD_START, {
        autoplay: playback.autoplay,
    }));

    const clipStatusUrl = `${clipsAPIHost}${contentId}/status`;

    return fetch(clipStatusUrl).
        then(response => response.json()).
        then(clipStatus => {
            dispatch(setClipQualities(clipStatus.quality_options));

            const stream = new ClipContentStream(contentId);
            dispatch({
                type: ACTION_SET_STREAM,
                stream,
            });

            return krakenRequestv5(`clips/${contentId}`).then(clipInfo => {
                // Note: clipInfo does not have partner/mature info but it is
                // not needed for clips at this time (used for ads/age gating)
                dispatch(setClipStreamMetadata(clipInfo));

                incrementClipViewCount(contentId);
            });
        });
}

function setVideoSourceStream({ contentId }, dispatch, getState) {
    const { playback } = getState();
    dispatch(trackEvent(VIDEO_PLAY_LOAD_START, {
        autoplay: playback.autoplay,
    }));

    const stream = new VideoSourceContentStream(contentId);
    dispatch({
        type: ACTION_SET_STREAM,
        stream,
    });
}

function setTwitchStream({ contentType, contentId }, dispatch, getState) {
    const { accessToken, analytics, experiments, playback, usher } = getState();

    const channelName = contentType === TYPE_CHANNEL ? sanitizeChannel(contentId) : null;
    const videoId = contentType === TYPE_VIDEO ? sanitizeVodId(contentId) : null;

    dispatch(trackEvent(VIDEO_PLAY_LOAD_START, {
        autoplay: playback.autoplay,
        channel: channelName,
        // eslint-disable-next-line camelcase
        vod_id: videoId,
    }));

    const oAuthToken = fetchOauthToken().then(token => {
        dispatch(trackEvent(VIDEO_PLAY_OAUTH, {
            // eslint-disable-next-line camelcase
            time_since_load_start: Date.now() - analytics.playSessionStartTime,
        }));
        return token;
    });

    const experimentSettings = {
        realtimeQos: experiments.get(REALTIME_QOS),
        fastBread: usher.params.fast_bread !== undefined ?
            Promise.resolve(usher.params.fast_bread) :
            experiments.get(FAST_BREAD),
    };

    let stream;
    if (contentType === TYPE_CHANNEL) {
        stream = new LiveTwitchContentStream(
            channelName,
            oAuthToken,
            usher.params,
            accessToken.params,
            experimentSettings
        );
    } else if (contentType === TYPE_VIDEO) {
        stream = new VODTwitchContentStream(
            videoId,
            oAuthToken,
            usher.params,
            accessToken.params,
            experimentSettings
        );
    }

    stream.accessToken.then(() => {
        dispatch(trackEvent(VIDEO_PLAY_NAUTH, {
            // eslint-disable-next-line camelcase
            time_since_load_start: Date.now() - analytics.playSessionStartTime,
        }));
    });

    dispatch({
        type: ACTION_SET_STREAM,
        stream,
    });

    dispatch(fetchAndSetStreamMetadata(stream));
    dispatch(fetchExtensions(stream.channel));
    dispatch(subscribeToExtensionControl(stream.channel));
}

/*
 * Sets the content stream to NullContentStream
 */
export function clearStream() {
    return {
        type: ACTION_SET_STREAM,
        stream: nullContentStream,
    };
}

/**
 * Converts a given channel name into its canonical form.
 *
 * @param {String} channelName
 * @return {String}
 */
function sanitizeChannel(channelName) {
    return String(channelName).replace(/[^A-Za-z0-9_]/g, '');
}

/**
 * Converts a given VOD ID into its canonical form.
 *
 * @param {String} vodId
 * @return {String}
 */
function sanitizeVodId(vodId) {
    return String(vodId).replace(/[^A-Za-z0-9_]/g, '');
}

/**
 * Fetches the user's OAuth token from the API. If the user is not logged in,
 * returns a resolved promise containing `null`.
 *
 * @return {Promise<Object>}
 */
function fetchOauthToken() {
    return oauthToken().then(auth => auth.token, () => null);
}
