import {
    getSubscriptionInfo,
    getSubscriptionProducts,
} from '../api';

export const ACTION_CHANNEL_SUBSCRIPTIONS_FETCHED = 'action channel subscriptions fetched';
export const ACTION_SUBSCRIPTION_PRODUCTS_FETCHED = 'action subscription products fetched';

/**
 * Fetch subscription status of the current user for the channel
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @return {Function} thunk returning getSubscriptionInfo call
 */
export function fetchChannelSubscriptions(userId, channelId) {
    return function(dispatch, getState) {
        return getSubscriptionInfo(userId, channelId).
            then(() => {
                if (channelId === getState().streamMetadata.channel.id) {
                    // Any successful response means the user is subscribed.
                    return dispatch(channelSubscriptionsFetched({
                        isSubscribed: true,
                    }));
                }
            }).
            catch(() => {
                // Twitch API returns errors to indicate no subscriptions. Also if we fail
                // for another reason we'll just show the unsubscribed state.
                if (channelId === getState().streamMetadata.channel.id) {
                    return dispatch(channelSubscriptionsFetched({
                        isSubscribed: false,
                    }));
                }
            });
    };
}

/**
 * Fetch subscription products for the subscriability of a channel
 *
 * @param {String} channelName
 * @return {Function} thunk returning getSubscriptionProducts call
 */
export function fetchSubscriptionProducts(channelName) {
    return function(dispatch, getState) {
        return getSubscriptionProducts(channelName).
            then(() => {
                if (channelName === getState().streamMetadata.channel.name) {
                    // Any successful response means the channel has a subscription program
                    return dispatch(subscriptionsProductsFetched({
                        isSubscribable: true,
                    }));
                }
            }).
            catch(() => {
                if (channelName === getState().streamMetadata.channel.name) {
                    return dispatch(subscriptionsProductsFetched({
                        isSubscribable: false,
                    }));
                }
            });
    };
}

/**
 * Sets the subscription status
 *
 * @param {Object} payload - a payload to merge with subscriptions data
 * @return {Object}
 */
const channelSubscriptionsFetched = payload => ({
    type: ACTION_CHANNEL_SUBSCRIPTIONS_FETCHED,
    subscriptions: payload,
});

/**
 * Sets the subscription.isSubcribable value
 *
 * @param {Object} payload - a payload to merge with subscriptions data
 * @return {Object}
 */
const subscriptionsProductsFetched = payload => ({
    type: ACTION_SUBSCRIPTION_PRODUCTS_FETCHED,
    subscriptions: payload,
});
