import { getRequestContext, AdRollTypes, DEFAULT_AD_DURATION } from '../actions/ads';
import { getDeclineReasons, willDeclineAds } from './declining-manager';
import { subscribe } from '../util/subscribe';
import '../../../vendor/apstag';
import includes from 'lodash/includes';
import * as AdSpadeEvents from '../analytics/spade-events';
import { initializeAdSpadeEvent, sendAdSpadeEvent } from '../ads/ads-spade';
import { CONTENT_MODE_VOD } from '../stream/twitch-vod';
import { CONTENT_MODE_LIVE } from '../stream/twitch-live';

export const AAX_TWITCH_PUBLISHER_ID = 3036;
export const AAX_VIDEO_AD_SERVER = 'DFP';

const AAX_VIDEO_MEDIA_TYPE = 'video';
const AAX_SLOT_ID_PREROLL = 'twitch-preroll';
const AAX_SLOT_ID_MIDROLL_PREFIX = 'twitch-midroll-';
const AAX_SLOT_ID_POSTROLL = 'twitch-postroll';

// Timeout in milliseconds
const PREROLL_TIMEOUT = 500;
const MIDROLL_TIMEOUT = 1000;
const POSTROLL_TIMEOUT = 1000;

/**
 * Amazon Ad Exchange Manager
 * Responsible for requesting bids from AAX server
 * Bids will be added to ima-tags as custom parameters
 */
export class AAXManager {
    constructor(publisherId, videoAdServer, store, options) {
        this._store = store;
        this._preloadPrerollBidsPromise = null;
        this._options = options;
        this._unsubs = [];

        try {
            const { window: windowObj } = this._store.getState();
            if (windowObj.apstag) {
                this._apstag = windowObj.apstag;
                this._apstag.init({
                    pubID: publisherId,
                    videoAdServer: videoAdServer,
                }, () => {
                    // We need to reassign apstag after init is completed because A9's code modifies the original apstag
                    // object and our local pointer does not reflect this change
                    this._apstag = windowObj.apstag;
                });
            } else {
                this._apstag = new NullApstag();
            }
        } catch (e) {
            this._apstag = new NullApstag();

            const spadeData = { reason: e.message };
            sendAdSpadeEvent(this._store, null, AdSpadeEvents.AAX_AD_AUCTION_INIT_ERROR, spadeData);
        }

        this._unsubFromStreamChanged = subscribe(this._store, ['stream'], this._onStreamChange.bind(this));
    }

    destroy() {
        this._unsubFromStreamChanged();
    }

    /**
     * Fetch a preroll bid and save the promise for later
     * @returns {Promise}
     */
    preloadPrerollBids() {
        this._preloadPrerollBidsPromise = this._getAdsRequestContextForPreroll().then(context => {
            // Check that a preroll is eligible to be played before fetching the bid
            if (!this._willDeclineAds(context)) {
                return this._fetchVideoBids([AAX_SLOT_ID_PREROLL], context, PREROLL_TIMEOUT);
            }

            return [];
        });

        return this._preloadPrerollBidsPromise;
    }

    /**
     * Fetch bids for a specific ad roll type
     * @param {AdsRequestContext} adRequestContext
     * @returns {Promise}
     */
    fetchBids(adRequestContext) {
        switch (adRequestContext.adType) {
        case AdRollTypes.PREROLL:
            return this._fetchPrerollBids(adRequestContext);
        case AdRollTypes.MIDROLL:
            return this._fetchMidrollBids(adRequestContext, MIDROLL_TIMEOUT);
        case AdRollTypes.POSTROLL:
            return this._fetchPostrollBids(adRequestContext, POSTROLL_TIMEOUT);
        default:
            return Promise.resolve([]);
        }
    }

    /**
     * Extracts bid ids and adds them to AdRequestContext
     * @param {Array<Object>} bids
     * @param adRequestContext
     */
    getAdRequestContextWithAmazonBids(bids, adRequestContext) {
        const localContext = adRequestContext;
        if (bids && bids[0]) {
            // Currently we only support one bid per adRequest.
            localContext.amzniid = bids[0].amzniid;
            localContext.amznbid = bids[0].amznbid;
            localContext.qsParams = bids[0].qsParams;
        }

        return localContext;
    }

    /**
     * Return a promise that will fetch a preroll bid
     * @returns {Promise}
     * @private
     */
    _fetchPrerollBids(adRequestContext) {
        if (!this._preloadPrerollBidsPromise) {
            // No preloaded bids available, make a new fetchBids request now
            return this._fetchVideoBids([AAX_SLOT_ID_PREROLL], adRequestContext, PREROLL_TIMEOUT);
        }

        // Null out the saved promise so that additional calls do not get the same preloaded bids
        const preloadPrerollBidsPromise = this._preloadPrerollBidsPromise;
        this._preloadPrerollBidsPromise = null;

        return preloadPrerollBidsPromise;
    }

    /**
     * Return a promise that will fetch a midroll bid
     * @param {AdsRequestContext} adRequestContext
     * @param {int} timeout
     * @returns {Promise}
     * @private
     */
    _fetchMidrollBids(adRequestContext, timeout) {
        // Currently we only support fetching a single AAX bid, regardless of duration
        // const numAds = Math.floor(adRequestContext.duration / DEFAULT_AD_DURATION);
        const numAds = 1;

        const slotIds = Array.apply(null, new Array(numAds)).map(function(x, i) {
            const midrollIndex = i + 1;
            return `${AAX_SLOT_ID_MIDROLL_PREFIX}${midrollIndex.toString()}`;
        });

        return this._fetchVideoBids(slotIds, adRequestContext, timeout);
    }

    /**
     * Return a promise that will fetch a postroll bid
     * @param {AdsRequestContext} adRequestContext
     * @param {int} timeout
     * @returns {Promise}
     * @private
     */
    _fetchPostrollBids(adRequestContext, timeout) {
        return this._fetchVideoBids([AAX_SLOT_ID_POSTROLL], adRequestContext, timeout);
    }

    /**
     * Return promise that will fetch a video bid
     * @param {Array<String>} slotIds
     * @param {AdsRequestContext} adRequestContext
     * @param {int} timeout
     * @returns {Promise}
     * @private
     */
    _fetchVideoBids(slotIds, adRequestContext, timeout) {
        const extraData = {
            // eslint-disable-next-line camelcase
            slot_ids: slotIds.join(','),
        };
        this._initAndSendAdSpadeEvent(AdSpadeEvents.AAX_AD_AUCTION, adRequestContext, extraData);
        const slots = this._createSlotsFromSlotIds(slotIds, timeout);

        return new Promise(resolve => {
            const { Date } = this._store.getState().window;
            const fetchBidsTime = Date.now();
            this._apstag.fetchBids(slots, bids => {
                const latency = Date.now() - fetchBidsTime;
                // eslint-disable-next-line camelcase
                extraData.aax_latency = latency;

                const videoBids = bids.filter(function(bid) {
                    return (bid.mediaType === AAX_VIDEO_MEDIA_TYPE) && includes(slotIds, bid.slotID);
                });

                if (this._options.debug) {
                    // eslint-disable-next-line no-console
                    console.log('aax fetchBids: ', videoBids);
                }
                if (videoBids.length > 0) {
                    extraData.amzniid = videoBids.map(bid => bid.amzniid).join(',');
                    extraData.amznbid = videoBids.map(bid => bid.amznbid).join(',');
                    extraData.qsParams = videoBids.map(bid => bid.qsParams).join(',');
                    this._initAndSendAdSpadeEvent(AdSpadeEvents.AAX_AD_AUCTION_RESPONSE, adRequestContext, extraData);
                    resolve(videoBids);
                } else {
                    this._initAndSendAdSpadeEvent(AdSpadeEvents.AAX_AD_AUCTION_ERROR, adRequestContext, extraData);
                    resolve([]);
                }
            });
        });
    }

    /**
     * Returns formatted array of slotIds that apstag can understand
     * @param {Array<String>} slotIds
     * @param {int} timeout
     * @returns {{slots}}
     * @private
     */
    _createSlotsFromSlotIds(slotIds, timeout) {
        return {
            slots: slotIds.map(function(slotId) {
                return {
                    mediaType: AAX_VIDEO_MEDIA_TYPE,
                    slotID: slotId,
                };
            }),
            timeout: timeout,
        };
    }

    /**
     * Called on the first stream change after this manager is instantiated.
     * This indicates we have a new channel and we can attempt to preload a preroll bid
     * @private
     */
    _onStreamChange() {
        // Unsubscribe from additional stream change events. Switching streams is very fast so preloading doesn't
        // give us any advantage
        this._unsubFromStreamChanged();

        const { contentType } = this._store.getState().stream;

        if (
            contentType === CONTENT_MODE_VOD ||
            contentType === CONTENT_MODE_LIVE
        ) {
            this.preloadPrerollBids();
        }
    }

    /**
     * Returns an Ads Request Context used for checking if preroll should be declined
     * @return {Promise}
     * @private
     */
    _getAdsRequestContextForPreroll() {
        let forcePreroll = false;
        if (this._options && this._options.hasOwnProperty('force_preroll')) {
            forcePreroll = true;
        }
        return getRequestContext(AdRollTypes.PREROLL, DEFAULT_AD_DURATION, this._store.getState(), forcePreroll);
    }

    /**
     * Check if we should decline the ad
     * @param {AdsRequestContext} adsRequestContext
     * @returns {bool}
     * @private
     */
    _willDeclineAds(adsRequestContext) {
        const declineReasons = getDeclineReasons(adsRequestContext);
        return willDeclineAds(declineReasons, adsRequestContext);
    }

    /**
     * Initialize and send an ad spade event
     * @param {String} eventName
     * @param {AdsRequestContext} adRequestContext
     * @param {Object} extraData
     * @private
     */
    _initAndSendAdSpadeEvent(eventName, adRequestContext, extraData = {}) {
        const spadeData = initializeAdSpadeEvent(adRequestContext);
        sendAdSpadeEvent(this._store, null, eventName, Object.assign({}, spadeData, extraData));
    }
}

/**
 * Null Apstag object that is returned if AAX is disabled
 */
class NullApstag {
    init() {}
    debug() {}

    fetchBids(slots, callback) {
        callback([]);
    }
}
