import assign from 'lodash/assign';
import { VODTwitchContentStream } from '../stream/twitch-vod';

/**
 * Create an object containing all fields for spade event tracking
 * @param {AdsRequestContext} adRequestContext
 * @param {google.ima.Ad} ad
 * @returns {Object}
 */
export function initializeAdSpadeEvent(adRequestContext, ad = null) {
    /* eslint-disable camelcase */
    let spadeData = {
        ad_session_id: adRequestContext.adSessionId,
        adblock: adRequestContext.adblock,
        roll_type: adRequestContext.adType,
        time_break: adRequestContext.duration,
        twitch_correlator: adRequestContext.twitchCorrelator,
    };
    if (adRequestContext.adWhitelist) {
        spadeData.ads_opt_in = adRequestContext.adWhitelist;
    }

    spadeData = setAmazonSpadeData(spadeData, adRequestContext);

    if (ad !== null) {
        spadeData.ad_system = ad.getAdSystem();
        spadeData.ad_id = ad.getAdId();
        spadeData.creative_id = ad.getCreativeId();
        spadeData.title = ad.getTitle();
        spadeData.duration = ad.getDuration();
        if (ad.getWrapperAdIds().length > 0) {
            spadeData.wrapper_ad_id = ad.getWrapperAdIds()[0];
            spadeData.wrapper_ad_system = ad.getWrapperAdSystems()[0];
            spadeData.wrapper_creative_id = ad.getWrapperCreativeIds()[0];
        }
    }
    /* eslint-enable camelcase */
    return spadeData;
}

/**
 * Send an ad spade event
 * @param {Object} store
 * @param {String} provider
 * @param {String} eventName
 * @param {Object} data
 */
export function sendAdSpadeEvent(store, provider = '', eventName = '', data = {}) {
    const { analyticsTracker, stream, streamMetadata } = store.getState();
    const eventPayload = data;

    if (provider) {
        assign(eventPayload, {
            provider,
        });
    }

    if (stream instanceof VODTwitchContentStream) {
        assign(eventPayload, {
            /* eslint-disable camelcase */
            vod_id: stream.videoId,
            vod_type: streamMetadata.type,
            /* eslint-enable camelcase */
        });
    }

    analyticsTracker.trackEvent(eventName, eventPayload);
}

/**
 * Adds fields like amzniid and amznbid to spade data
 * @param {Object} spadeData
 * @param {AdsRequestContext} adRequestContext
 * @returns {Object}
 */
function setAmazonSpadeData(spadeData, adRequestContext) {
    const localSpadeData = spadeData;
    if (adRequestContext.amzniid) {
        localSpadeData.amzniid = adRequestContext.amzniid;
    }
    if (adRequestContext.amznbid) {
        localSpadeData.amznbid = adRequestContext.amznbid;
    }
    if (adRequestContext.qsParams) {
        localSpadeData.qsParams = adRequestContext.qsParams;
    }

    return localSpadeData;
}
