import { generate } from '../util/unique-id';
import { sanitizeQuery } from '../util/url';
import { PLAYER_CURSE } from '../util/player-type';
import { CONTENT_MODE_VOD } from '../stream/twitch-vod';
import { formatCustomTag } from './ima-tags';
import * as DeviceId from '../util/device-id';

const AD_SESSION_ID_LENGTH = 30;

const getCommunityIds = communities => {
    if (Array.isArray(communities) && communities.length !== 0) {
        return communities.map(c => c._id).join(',');
    }
    return null;
};

/**
 * Describes the context of the player at the time an ads request was made; this
 * struct class holds all information required to request (or decline to request)
 * ads.
 */
export class AdsRequestContext {
    // TODO coalesce some of these properties into state so we don't have to
    // pull an excessive number of keys out of the object param
    // eslint-disable-next-line
    constructor({
        adType,
        duration,
        requestType,
        state,
        lastAdDisplay,
        forced,
        userInfo,
        channelInfo,
        channelAPIInfo,
        viewerInfo,
        channelAdProperties,
        creativeId,
        padtimeExpGroup,
        communitiesInfo,
        skippableExpGroup,
        raid,
        embedPromo,
        leviathanExpGroup,
    }) {
        this.contentType = state.stream.contentType;
        this.login = userInfo.login;
        this.adSessionId = generate(AD_SESSION_ID_LENGTH);
        this.adType = adType;
        this.raid = raid;
        this.channel = channelInfo.name;
        this.channelId = channelInfo._id;
        this.chansubToken = viewerInfo.chansub;
        this.communityIds = getCommunityIds(communitiesInfo.communities);
        this.twitchCorrelator = generate();
        this.deviceId = DeviceId.get(false);
        this.duration = duration;
        this.requestType = requestType;
        this.hasAdFreeSubscription = Boolean(viewerInfo.has_ad_free_subscription);
        this.lastAdDisplay = lastAdDisplay;
        this.forced = forced;
        this.padtime = padtimeExpGroup;
        this.requestTime = Date.now();
        this.skippableExpGroup = skippableExpGroup;
        this.creativeId = creativeId;
        this.embedPromo = embedPromo;
        this.leviathanExpGroup = leviathanExpGroup;

        this._initUserInfoFields(userInfo);
        this._initChannelInfoFields(channelInfo, channelAPIInfo);
        this._initStateFields(state);
        this._initAmazonFields();
        this._initChannelAdPropertiesFields(channelAdProperties);
    }

    _initUserInfoFields(userInfo) {
        this.login = userInfo.login;
        this.turboToken = userInfo.turbo;
    }

    _initChannelInfoFields(channelInfo, channelAPIInfo) {
        this.channel = channelInfo.name;
        this.channelId = channelInfo._id;
        if (channelInfo.game) {
            this.game = formatCustomTag(channelInfo.game);
        }
        this.mature = channelInfo.mature;
        this.partner = channelInfo.partner;
        this.postrollsEnabled = Boolean(channelAPIInfo.postrolls);
        this.prerollsEnabled = Boolean(channelAPIInfo.prerolls);
        /* eslint-disable-next-line camelcase */
        this.broadcasterSoftware = channelAPIInfo.broadcaster_software;
    }

    _initChannelAdPropertiesFields(channelAdProperties) {
        /* eslint-disable camelcase, max-len */
        this.vodAdsEnabled = (channelAdProperties !== null ? channelAdProperties.valid_responses.vod_ads_enabled : true);
        /* eslint-enable camelcase, max-len */
    }

    _initStateFields(state) {
        const { ads, adsManager, window } = state;
        const { platform, playerType } = state.env;

        this.contentType = state.stream.contentType;
        this.adblock = ads.adblock;
        this.vodMidrollArchiveSetting = ads.vodMidrollArchiveSetting;
        this.isMobileLocation = state.env.isMobileLocation;
        this.kruxId = (window.Krux ? window.Krux.user || '' : null);
        this.platform = (playerType === PLAYER_CURSE ? playerType : platform);
        this.playerType = state.env.playerType;
        this.embedUrl = window.document.referrer;
        this.url = sanitizeQuery(window.location.href);
        this.sdk = adsManager.sdk;

        this._initVodFields(state);
    }

    _initVodFields(state) {
        const { playback } = state;

        this.vod = {
            id: '',
            name: '',
            type: '',
            duration: -1,
        };
        if (state.stream.contentType === CONTENT_MODE_VOD) {
            this.vod.id = state.stream.videoId;
            this.vod.name = formatCustomTag(state.streamMetadata.name);
            this.vod.type = state.streamMetadata.type;
            this.vod.duration = playback.duration;
        }
    }

    _initAmazonFields() {
        this.amzniid = '';
        this.amznbid = '';
        this.qsParams = '';
    }
}
