import assign from 'lodash/assign';
import reduce from 'lodash/reduce';
import { PREROLL, MIDROLL, POSTROLL } from './ima-tags';
import { PLAYER_CREATIVE, PLAYER_EMBED, PLAYER_FACEBOOK,
         PLAYER_FRONTPAGE, PLAYER_HIGHLIGHTER, PLAYER_DASHBOARD,
         PLAYER_ANIMATED_THUMBNAILS } from '../util/player-type';
import { CONTENT_MODE_VOD } from '../stream/twitch-vod';
import { PLATFORM_MOBILE_WEB } from 'state/env';
import { ARCHIVE, VOD_MIDROLL_SETTING_BROADCASTER } from '../vod-midroll-manager';
import { ID3_AD } from '../video';

/**
 * Time during which ads will not be displayed to users to avoid excessive ad
 * plays.
 *
 * @const {Number} in milliseconds
 */
const AD_RATE_LIMIT = 5 * 60 * 1000;
/**
 * Minimum duration for VODs to show preroll ads
 *
 * @const {Number} in seconds
 */
const SHORT_VOD_DURATION = 60;

export class DecliningAdsManager {
    constructor(store, adManager) {
        this._store = store;
        this._adManager = adManager;

        this._streamRollTypes = {};
    }

    destroy() {
        this._adManager.destroy();
    }

    /* eslint-disable complexity */
    requestAds(adRequestContext) {
        if (!this._streamRollTypes.hasOwnProperty(adRequestContext.channel)) {
            this._streamRollTypes[adRequestContext.channel] = {
                [PREROLL]: false,
                [MIDROLL]: false,
                [POSTROLL]: false,
            };
        }

        const declineReasons = getDeclineReasons(adRequestContext);
        const willDecline = willDeclineAds(declineReasons, adRequestContext);

        if (willDecline) {
            trackDeclinedAds(this._store.getState(), adRequestContext, declineReasons);
        } else {
            this._adManager.requestAds(adRequestContext);
            this._streamRollTypes[adRequestContext.channel][adRequestContext.adType] = true;
        }
    }
    /* eslint-enable complexity */

    addEventListener(event, listener) {
        this._adManager.addEventListener(event, listener);
    }

    pause() {
        this._adManager.pause();
    }

    play() {
        this._adManager.play();
    }

    setVolume(volumeLevel) {
        this._adManager.setVolume(volumeLevel);
    }

    setMuted(value) {
        this._adManager.setMuted(value);
    }

    get paused() {
        return this._adManager.paused;
    }

    get sdk() {
        return this._adManager.sdk;
    }
}

function trackDeclinedAds(state, adRequestContext, declineReasons) {
    const { analyticsTracker } = state;
    analyticsTracker.trackEvent('video_ad_request_declined', assign({
        /* eslint-disable camelcase */
        ad_session_id: adRequestContext.adSessionId,
        // ad_type: 'video', // TODO confirm this field with AdOps
        roll_type: adRequestContext.adType,
        time_break: adRequestContext.duration,
        provider: 'ima', // TODO declining isn't relevant to the provider...
        adblock: getAdblockStatus(adRequestContext),
        twitch_correlator: adRequestContext.twitchCorrelator,
        /* eslint-enable camelcase */
    }, declineReasons));
}

/**
 * Derives the reasons for which an ad opportunity would be declined from the
 * ad request context, which contains all of the information relevant to making
 * an ads request.
 *
 * @param {AdsRequestContext} adRequestContext
 * @return {Object<String, Boolean>}
 *         set of reasons matched with whether they apply to the given request.
 */
// eslint-disable-next-line complexity
export function getDeclineReasons(adRequestContext) {
    /* eslint-disable camelcase */
    const reasons = {
        // Decline preroll ads for short clips
        reason_short_vod: (
            adRequestContext.adType === PREROLL &&
            adRequestContext.contentType === CONTENT_MODE_VOD &&
            adRequestContext.vod.duration >= 0 &&
            adRequestContext.vod.duration < SHORT_VOD_DURATION
        ),

        // Never show preroll ads on VODs that belong to users
        reason_vod_creator: (
            adRequestContext.adType === PREROLL &&
            adRequestContext.contentType === CONTENT_MODE_VOD &&
            adRequestContext.login === adRequestContext.channel
        ),

        // Never show ads to Turbo users
        reason_turbo: (
            Boolean(adRequestContext.turboToken)
        ),

        // Don't show prerolls to users whose last ad view is within the last 5 minutes
        reason_ratelimit: (
            adRequestContext.adType === PREROLL &&
            (adRequestContext.requestTime - adRequestContext.lastAdDisplay < AD_RATE_LIMIT)
        ),

        // Don't show ads to users who have an ad-free subscription
        // This requires the following:
        //   1. The user is a subscriber to the channel (isChansub)
        //   2. The channel's subscriptions grant ad-free playback (hasAdFreeSubscription),
        //      which is actually a property of the channel subscription and NOT the user
        reason_channelsub: (
            Boolean(adRequestContext.chansubToken) &&
            adRequestContext.hasAdFreeSubscription
        ),

        // Some channels are configured to not show prerolls or postrolls;
        // both of those configs will result in this reason being set to `true`
        // They can be distinguished by the `roll_type` property associated
        // with the `video_ad_request_declined` event.
        reason_channeladfree: (
            (adRequestContext.adType === PREROLL && !adRequestContext.prerollsEnabled) ||
            (adRequestContext.adType === POSTROLL && !adRequestContext.postrollsEnabled)
        ),

        // Don't show ads on the front page player
        reason_frontpage: adRequestContext.playerType === PLAYER_FRONTPAGE,

        // Don't show ads for the creative player
        reason_creative_player: adRequestContext.playerType === PLAYER_CREATIVE,

        // Don't show ads when we're embedded in Facebook
        reason_facebook: adRequestContext.playerType === PLAYER_FACEBOOK,

        // Don't show ads when we're embeeded in Highlighter
        reason_highlighter: adRequestContext.playerType === PLAYER_HIGHLIGHTER,

        // Don't show ads when we're embedded in Broadcaster Dashboard
        reason_dashboard: adRequestContext.playerType === PLAYER_DASHBOARD,

        // Don't show ads when a user hovers over a thumbnail in the directory page
        reason_animated_thumbnails: adRequestContext.playerType === PLAYER_ANIMATED_THUMBNAILS,

        // Don't show ads on the mobile pages since they're intended as a component to
        // upsell into the native mobile apps.
        reason_mobile_web_upsell: (
            Boolean(adRequestContext.isMobileLocation) &&
            adRequestContext.platform === PLATFORM_MOBILE_WEB
        ),

        // Don't show preroll ads on vods if brought to vod from digest
        reason_raid: (
            adRequestContext.adType === PREROLL &&
            adRequestContext.raid &&
            adRequestContext.leviathanExpGroup === 'yes'
        ),

        // Don't show broadcaster-triggered midroll ads on archives if setting is not Broadcaster
        reason_vod_midroll: (
            adRequestContext.contentType === CONTENT_MODE_VOD &&
            adRequestContext.vod.type === ARCHIVE &&
            adRequestContext.adType === MIDROLL &&
            adRequestContext.requestType === ID3_AD &&
            adRequestContext.vodMidrollArchiveSetting !== VOD_MIDROLL_SETTING_BROADCASTER
        ),

        // Don't show ads on VODs if VOD ads are not enabled
        reason_vod_ads_disabled: (
            adRequestContext.contentType === CONTENT_MODE_VOD &&
            adRequestContext.vodAdsEnabled === false
        ),

        reason_embed_promo: (
            adRequestContext.contentType === CONTENT_MODE_VOD &&
            adRequestContext.playerType === PLAYER_EMBED &&
            adRequestContext.embedPromo
        ),

        // Don't show ads to users who have already seen a pre- or post-roll on
        // the current channel.
        // Commenting this reason out. We are going to deprecate the experiment
        // associated with this reason for now and investigate this
        // further before enabling it.
        // reason_already_seen_rolltype: (
        //     Boolean(rollTypesSeen[adRequestContext.adType]) &&
        //     (adRequestContext.adType === PREROLL || adRequestContext.adType === POSTROLL)
        // ),
    };
    /* eslint-enable camelcase */

    return reasons;
}

function getAdblockStatus(adRequestContext) {
    return adRequestContext.adblock === 'unknown' ? null : adRequestContext.adblock;
}

/**
 * Determines if you will decline ads based on all declineReasons and context
 *
 * @param declineReasons
 * @param adRequestContext
 * @returns bool
 */
export function willDeclineAds(declineReasons, adRequestContext) {
    let willDecline = reduce(declineReasons, (hasDeclined, isDecline) => {
        return hasDeclined || isDecline;
    }, false);

    if (adRequestContext.forced) {
        willDecline = false;
    }

    return willDecline;
}
