/**
 * dfp-creative-service is a service where you can provide a DFP creative id and get back the
 * VAST URL associated with that creative. This is useful for forcing the player to play a
 * specific creative
 *
 * https://git-aws.internal.justin.tv/ads/dfp-creative-service
 */

// Url to service for fetching creative from DFP
const DFP_GET_CREATIVE_URL = 'https://dfp-creative-service.twitch.tv/?creative_id=';

export class DfpCreativeService {
    /**
     * Makes a call to dfp-creative-service to fetch the Vast URL for a creative id
     * @param creativeId
     * @returns {Promise}
     */
    getVastXmlResponse(creativeId) {
        const url = `${DFP_GET_CREATIVE_URL}${creativeId}`;
        return this._httpsGet(url).then(response => {
            if (!response || !response.vast_xml_response) {
                return Promise.reject(new Error(`Invalid response from dfp-creative-service ${response}`));
            }

            return response.vast_xml_response;
        });
    }

    /**
     * Helper to make a GET request to a URL using Promises
     * @param {String} url
     * @returns {Promise}
     * @private
     */
    _httpsGet(url) {
        const secureUrl = this._convertHttpToHttps(url);
        return fetch(secureUrl).then(response => response.json());
    }

    /**
     * Converts an HTTP url into an HTTPS url
     * @param {String} url
     * @return {String}
     * @private
     */
    _convertHttpToHttps(url) {
        return url.replace(/^http:\/\//i, 'https://');
    }
}
