import { buildIMATags } from './ima-tags';
import { setCurrentAdMetadata, clearCurrentAdMetadata, AdContentTypes,
         AdRollTypes } from '../actions/ads';
import { FLASH_AD_EVENTS } from '../backend/flash';
import * as AdEvents from '../ads/advertisement-event';
import * as AdSpadeEvents from '../analytics/spade-events';
import EventEmitter from 'event-emitter';
import { sendAdSpadeEvent } from '../ads/ads-spade';

const FLASH_SDK_NAME = 'flash';
const IMA_PROVIDER = 'ima';

const ROLLTYPE_MAP = {
    preroll: AdRollTypes.PREROLL,
    midroll: AdRollTypes.MIDROLL,
    postroll: AdRollTypes.POSTROLL,
};

export class FlashManager {
    constructor(backend, store) {
        this._flashBackend = backend;
        this._store = store;
        this._eventEmitter = new EventEmitter();
        this._initializeEventListeners();
    }

    _initializeEventListeners() {
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_START, this.onAdStarted.bind(this));
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_END, this.onAdEnded.bind(this));
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.COMPANION_RENDERED, this.onAdCompanionRendered.bind(this));
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_REQUEST, this.onAdRequest.bind(this));
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_REQUEST_DECLINED, this.onAdRequestDeclined.bind(this));
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_REQUEST_RESPONSE, this.onAdRequestResponse.bind(this));
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_REQUEST_ERROR, this.onAdRequestError.bind(this));
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_ERROR, this.onAdError.bind(this));
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_LOADED, this.onAdLoaded.bind(this));
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_IMPRESSION, this.onAdImpression.bind(this));
        // eslint-disable-next-line max-len
        this._flashBackend.addEventListener(FLASH_AD_EVENTS.AD_IMPRESSION_COMPLETE, this.onAdImpressionComplete.bind(this));
    }

    onAdStarted(data) {
        this._store.dispatch(setCurrentAdMetadata({
            contentType: AdContentTypes.IMA,
            rollType: ROLLTYPE_MAP[data.roll_type],
        }));
        this._eventEmitter.emit(AdEvents.AD_START, {
            // eslint-disable-next-line camelcase
            roll_type: data.roll_type,
        });
    }

    onAdEnded(data) {
        this._store.dispatch(clearCurrentAdMetadata());
        this._eventEmitter.emit(AdEvents.AD_END, {
            // eslint-disable-next-line camelcase
            roll_type: data.roll_type,
        });
    }

    onAdCompanionRendered() {
        this._eventEmitter.emit(AdEvents.COMPANION_RENDERED, { provider: IMA_PROVIDER });
    }

    onAdRequest(data) {
        sendAdSpadeEvent(this._store, '', AdSpadeEvents.AD_REQUEST, data);
    }

    onAdRequestDeclined(data) {
        sendAdSpadeEvent(this._store, '', AdSpadeEvents.AD_REQUEST_DECLINED, data);
    }

    onAdRequestResponse(data) {
        sendAdSpadeEvent(this._store, '', AdSpadeEvents.AD_REQUEST_RESPONSE, data);
    }

    onAdRequestError(data) {
        sendAdSpadeEvent(this._store, '', AdSpadeEvents.AD_REQUEST_ERROR, data);
    }

    onAdError(data) {
        sendAdSpadeEvent(this._store, '', AdSpadeEvents.AD_ERROR, data);
    }

    onAdLoaded(data) {
        sendAdSpadeEvent(this._store, '', AdSpadeEvents.AD_LOADED, data);
    }

    onAdImpression(data) {
        sendAdSpadeEvent(this._store, '', AdSpadeEvents.AD_IMPRESSION, data);
        this._eventEmitter.emit(AdEvents.AD_IMPRESSION, {
            // eslint-disable-next-line camelcase
            time_break: data.time_break,
        });
    }

    onAdImpressionComplete(data) {
        sendAdSpadeEvent(this._store, '', AdSpadeEvents.AD_IMPRESSION_COMPLETE, data);
        this._eventEmitter.emit(AdEvents.AD_IMPRESSION_COMPLETE);
    }

    requestAds(adRequestContext) {
        const { duration, adType } = adRequestContext;
        const adTagUrl =  buildIMATags(adRequestContext);

        this._flashBackend.requestAdFill(duration, adType, adTagUrl);
    }

    pause() {
        if (!this._isAdRunning()) {
            return;
        }

        this._flashBackend.pause();
    }

    play() {
        if (!this._isAdRunning()) {
            return;
        }

        this._flashBackend.play();
    }

    setVolume(value) {
        if (!this._isAdRunning()) {
            return;
        }

        this._flashBackend.setVolume(value);
    }

    setMuted(muted) {
        if (!this._isAdRunning()) {
            return;
        }

        this._flashBackend.setMuted(muted);
    }

    /* used by video.js to listen to ad events */
    addEventListener(name, callback) {
        this._eventEmitter.on(name, callback);
    }

    destroy() {
    }

    get paused() {
        return this._flashBackend.getPaused();
    }

    get muted() {
        return this._flashBackend.getMuted();
    }

    get sdk() {
        return FLASH_SDK_NAME;
    }

    /* helper functions */
    _isAdRunning() {
        const { ads } = this._store.getState();
        return ads.current.contentType !== AdContentTypes.NONE;
    }
}

