import EventEmitter from 'event-emitter';
import { buildIMATags } from './ima-tags';
import assign from 'lodash/assign';
import * as AdSpadeEvents from '../analytics/spade-events';
import { LiveTwitchContentStream } from '../stream/twitch-live';
import { VODTwitchContentStream } from '../stream/twitch-vod';
import * as AdEvents from 'ads/advertisement-event';
import { setCurrentAdMetadata, clearCurrentAdMetadata, AdContentTypes, AdRollTypes } from '../actions/ads';
import { MoatAnalytics } from '../analytics/moat';
import { play, automatedPause } from '../actions/video-api';
import { adCompanionRendered } from '../actions/embed-event-emitter';
import { subscribe } from '../util/subscribe';
import { AAXManager, AAX_TWITCH_PUBLISHER_ID, AAX_VIDEO_AD_SERVER } from 'ads/aax-manager';
import { initializeAdSpadeEvent, sendAdSpadeEvent } from 'ads/ads-spade';
import { DfpCreativeService } from 'ads/dfp-creative-service';
import { emitAdPlayEvent, emitAdPauseEvent, emitAdPlayingEvent } from '../actions/event-emitter';
import bowser from 'bowser';

const IMA_SDK_NAME = 'html5';
export const IMA_PROVIDER = 'ima';

const COMPANION_AD_SIZES = [
    {
        width: 300,
        height: 250,
    }, {
        width: 300,
        height: 60,
    },
];

const ROLLTYPE_MAP = {
    preroll: AdRollTypes.PREROLL,
    midroll: AdRollTypes.MIDROLL,
    postroll: AdRollTypes.POSTROLL,
};

export const AD_BITRATE_MAX = 3000;
export const AD_BITRATE_MIN = 500;

/*
 * Handles IMA SDK Ads layer by managing the content and ad streams
 * @param {Object} DOM Element container that contains the video playing element
 * @param {Object} store reference
 * @param {Object} AdsContext object
 */
export class IMAManager {
    constructor(videoContainer, backend, store, options = {}) {
        this._videoContainer = videoContainer;
        this._backend = backend;
        this._store = store;

        this._unsubs = [];

        this._paused = false;

        this._contentPauseRequested = false;

        this._eventEmitter = new EventEmitter();
        // Only one AdsManager per AdRequest (https://developers.google.com/interactive-media-ads/docs/sdks/html5/faq#8)
        this._currentAdsManager = NULL_GOOGLE_ADS_MANAGER;

        this._aaxManager = new AAXManager(AAX_TWITCH_PUBLISHER_ID, AAX_VIDEO_AD_SERVER, this._store, options);

        const { window } = this._store.getState();
        const google = window.google;

        if (!google) {
            // abort... everything is simply going to fail.
            return;
        }

        this._unsubs.push(subscribe(this._store, ['playerDimensions'], this._resizeAd.bind(this)));
        this._configureVpaidMode();
        this.initializeAdDisplayContainer();

        this._moat = new MoatAnalytics(this._adContainer, this._store);

        this._setupAdsLoader();
    }

    initializeAdDisplayContainer() {
        const { window } = this._store.getState();
        const google = window.google;

        if (!google) {
            return;
        }

        this._adContainer = window.document.createElement('div');
        this._adContainer.classList.add('js-ima-ads-container', 'ima-ads-container');

        this._videoContainer.appendChild(this._adContainer);

        this._adDisplayContainer = new google.ima.AdDisplayContainer(
            this._adContainer,
            this._videoContainer.querySelector('video, object')
        );
        this._adDisplayContainer.initialize();
    }

    destroy() {
        this._adsLoader.destroy();
        this._adDisplayContainer.destroy();
        this._aaxManager.destroy();
        this._unsubs.forEach(unsub => unsub());
    }

    requestAds(adsRequestContext) {
        // Do not request an ad if there is already an ad playing
        if (this._currentAdsManager.getRemainingTime() > 0) {
            return Promise.resolve();
        }

        if (this._isCreativeIdSet(adsRequestContext)) {
            return this._requestAdsInternalWithCreativeId(adsRequestContext);
        }

        return this._requestAdsInternal(adsRequestContext);
    }

    pause() {
        this._paused = true;
        this._currentAdsManager.pause();
    }

    play() {
        this._paused = false;
        this._currentAdsManager.resume();
    }

    setVolume(volumeLevel) {
        this._currentAdsManager.setVolume(volumeLevel);
    }

    setMuted(muted) {
        const { playback } = this._store.getState();

        this._currentAdsManager.setVolume(muted ? 0 : playback.volume);
    }

    /* used by video.js to listen to ad events */
    addEventListener(name, callback) {
        this._eventEmitter.on(name, callback);
    }

    _setupAdsLoader() {
        const google = this._store.getState().window.google;

        this._adsLoader = new google.ima.AdsLoader(this._adDisplayContainer);
        this._adsLoader.addEventListener(
            google.ima.AdsManagerLoadedEvent.Type.ADS_MANAGER_LOADED,
            event => this._onAdsManagerLoaded(event)
        );
        this._adsLoader.addEventListener(
            google.ima.AdErrorEvent.Type.AD_ERROR,
            event => this._onAdError(event)
        );
    }

    /* event listeners */
    _onAdsManagerLoaded(adsManagerLoadedEvent) {
        const { google } = this._store.getState().window;
        const adRequestContext = adsManagerLoadedEvent.getUserRequestContext();
        sendAdSpadeEvent(this._store, IMA_PROVIDER, AdSpadeEvents.AD_REQUEST_RESPONSE,
            initializeAdSpadeEvent(adRequestContext));

        // Get the ads manager.
        this._currentAdsManager = adsManagerLoadedEvent.getAdsManager(
            new ContentPlayback(this._backend),
            this._getAdRenderSettings());

        // Add listeners to the required events.
        this._currentAdsManager.addEventListener(
            google.ima.AdErrorEvent.Type.AD_ERROR,
            evt => this._onAdError(evt));

        this._currentAdsManager.addEventListener(
            google.ima.AdEvent.Type.RESUMED,
            () => this._store.dispatch(emitAdPlayingEvent()));

        this._currentAdsManager.addEventListener(
            google.ima.AdEvent.Type.PAUSED,
            () => this._store.dispatch(emitAdPauseEvent()));

        this._currentAdsManager.addEventListener(
            google.ima.AdEvent.Type.CONTENT_PAUSE_REQUESTED,
            () => this._onContentPauseRequested(adRequestContext));

        this._currentAdsManager.addEventListener(
            google.ima.AdEvent.Type.CONTENT_RESUME_REQUESTED,
            () => this._onContentResumeRequested(adRequestContext));

        this._currentAdsManager.addEventListener(
            google.ima.AdEvent.Type.LOADED,
            evt => this._onAdLoaded(evt, adRequestContext));

        this._currentAdsManager.addEventListener(
            google.ima.AdEvent.Type.STARTED,
            evt => this._onAdStarted(evt));

        this._currentAdsManager.addEventListener(
            google.ima.AdEvent.Type.IMPRESSION,
            evt => this._onAdImpression(evt, adRequestContext)
        );

        this._currentAdsManager.addEventListener(
            google.ima.AdEvent.Type.COMPLETE,
            evt => this._onAdEnded(evt, adRequestContext));

        this._currentAdsManager.addEventListener(
            google.ima.AdEvent.Type.SKIPPED,
            evt => this._onAdSkipped(evt, adRequestContext));

        this._currentAdsManager.init(
            this._videoContainer.offsetParent.offsetWidth,
            this._videoContainer.offsetParent.offsetHeight,
            google.ima.ViewMode.NORMAL
        );
        this._currentAdsManager.start();
    }

    _interruptContent() {
        const { stream } = this._store.getState();

        if (stream instanceof LiveTwitchContentStream) {
            this._backend.setMuted(true);
        } else if (stream instanceof VODTwitchContentStream && !this._backend.getEnded()) {
            this._store.dispatch(automatedPause());
        }
    }

    _resumeContent() {
        const { playback, stream } = this._store.getState();

        this._backend.setVolume(playback.volume);
        this._backend.setMuted(playback.muted);

        if (stream instanceof VODTwitchContentStream && !this._backend.getEnded()) {
            this._store.dispatch(play());
        }
    }

    /**
     * Callback invoked when an error occurs in loading or playing ads.
     *
     * @param {google.ima.AdErrorEvent} event
     */
    _onAdError(event) {
        const error = event.getError();
        const adRequestContext = event.getUserRequestContext() || {
            // omit each field from the tracking data; nulls are elided by Spade
            adSessionId: null,
            adblock: null,
            adType: null,
            duration: null,
        };
        const spadeEvent = initializeAdSpadeEvent(adRequestContext);
        spadeEvent.reason = error.getMessage();
        /* eslint-disable camelcase */
        spadeEvent.error_code = error.getErrorCode();
        spadeEvent.error_type = error.getType();
        /* eslint-enable camelcase */
        sendAdSpadeEvent(this._store, IMA_PROVIDER, AdSpadeEvents.AD_ERROR, spadeEvent);

        if (this._contentPauseRequested) {
            this._contentPauseRequested = false;
            this._store.dispatch(clearCurrentAdMetadata());
            this._resumeContent();
            this._eventEmitter.emit(AdEvents.AD_ERROR, {
            // eslint-disable-next-line camelcase
                roll_type: adRequestContext.adType,
            });
        }

        if (this._isCreativeIdSet(adRequestContext)) {
            // eslint-disable-next-line no-console
            console.error(`Error playing creative ${adRequestContext.creativeId}: ${error.getMessage()}`);
        }

        this._currentAdsManager.destroy();
        this._currentAdsManager = NULL_GOOGLE_ADS_MANAGER;
    }

    _onContentPauseRequested(adRequestContext) {
        this._contentPauseRequested = true;
        if (this._store.getState().ads.currentMetadata.contentType !== AdContentTypes.NONE) {
            return;
        }

        const { playback } = this._store.getState();
        this._currentAdsManager.setVolume(playback.muted ? 0 : playback.volume);

        this._interruptContent();

        this._store.dispatch(setCurrentAdMetadata({
            contentType: AdContentTypes.IMA,
            rollType: ROLLTYPE_MAP[adRequestContext.adType],
        }));

        this._eventEmitter.emit(AdEvents.AD_START, {
            // eslint-disable-next-line camelcase
            roll_type: adRequestContext.adType,
        });
    }

    _onContentResumeRequested(adRequestContext) {
        this._contentPauseRequested = false;
        if (this._store.getState().ads.currentMetadata.contentType === AdContentTypes.NONE) {
            return;
        }

        this._currentAdsManager = NULL_GOOGLE_ADS_MANAGER;
        this._store.dispatch(clearCurrentAdMetadata());
        this._eventEmitter.emit(AdEvents.AD_END, {
            // eslint-disable-next-line camelcase
            roll_type: adRequestContext.adType,
        });

        this._resumeContent();
    }

    /**
     * Callback function invoked when an ad is loaded.
     */
    _onAdLoaded(event, adRequestContext) {
        // VPAID ads should respect current player volume settings
        const { playback } = this._store.getState();
        this._currentAdsManager.setVolume(playback.muted ? 0 : playback.volume);
        // Track VPAID ads with Moat
        this._moat.trackAd(this._currentAdsManager, adRequestContext);

        sendAdSpadeEvent(this._store, IMA_PROVIDER, AdSpadeEvents.AD_LOADED,
            initializeAdSpadeEvent(adRequestContext, event.getAd()));
    }

    /**
     * Callback function invoked when an ad starts.
     * Get and signal companion ads from here
     */
    _onAdStarted(event) {
        const ad = event.getAd();
        const companionAdCount = COMPANION_AD_SIZES.reduce(
            (count, size) => count + ad.getCompanionAds(size.width, size.height).length,
            0
        );
        if (companionAdCount > 0) {
            // emit 'provider' for tracking
            this._eventEmitter.emit(AdEvents.COMPANION_RENDERED, { provider: IMA_PROVIDER });
            this._store.dispatch(adCompanionRendered(AdEvents.COMPANION_RENDERED, { provider: IMA_PROVIDER }));
        }
        this._store.dispatch(emitAdPlayEvent());
    }

    /**
     * Callback function invoked when an ad impression is registered with DFP.
     * TODO: This is not exactly equivalent to the Science wiki's description
     * for video_ad_impression, but it is consistent with the current Flash
     * implementation.
     *
     * @param {google.ima.AdEvent} event
     * @param {AdsRequestContext} adRequestContext
     */
    _onAdImpression(event, adRequestContext) {
        const spadeEvent = initializeAdSpadeEvent(adRequestContext, event.getAd());
        // eslint-disable-next-line camelcase
        const { Date } = this._store.getState().window;
        // eslint-disable-next-line camelcase
        spadeEvent.request_to_impression_latency = Date.now() - this._adRequestTime;
        sendAdSpadeEvent(this._store, IMA_PROVIDER, AdSpadeEvents.AD_IMPRESSION, spadeEvent);

        this._eventEmitter.emit(AdEvents.AD_IMPRESSION, {
            // eslint-disable-next-line camelcase
            time_break: adRequestContext.duration,
        });
    }

    /**
     * Callback function invoked when an individual ad in a pod has completed
     * playing.
     *
     * @param {google.ima.AdEvent} event
     * @param {AdsRequestContext} adRequestContext
     */
    _onAdEnded(event, adRequestContext) {
        const spadeEvent = initializeAdSpadeEvent(adRequestContext, event.getAd());
        // eslint-disable-next-line camelcase
        sendAdSpadeEvent(this._store, IMA_PROVIDER, AdSpadeEvents.AD_IMPRESSION_COMPLETE, spadeEvent);

        this._eventEmitter.emit(AdEvents.AD_IMPRESSION_COMPLETE);
    }

    /**
     * Callback function invoked when an individual ad is eligible to be skipped.
     *
     * @param {google.ima.AdEvent} event
     * @param {AdsRequestContext} adRequestContext
     */
    _onAdSkipped(event, adRequestContext) {
        const ad = event.getAd();
        const spadeEvent = initializeAdSpadeEvent(adRequestContext);
        assign(spadeEvent, {
            // eslint-disable-next-line camelcase
            ad_id: ad.getAdId(),
            // eslint-disable-next-line camelcase
            client_time: Date.now(),
            duration: ad.getDuration(),
        });

        sendAdSpadeEvent(this._store, IMA_PROVIDER, AdSpadeEvents.AD_SKIPPED, spadeEvent);

        this._eventEmitter.emit(AdEvents.AD_SKIPPED);
    }

    _resizeAd({ playerDimensions }) {
        const { google } = this._store.getState().window;
        this._currentAdsManager.resize(
            playerDimensions.width,
            playerDimensions.height,
            google.ima.ViewMode.NORMAL
        );
    }

    /**
     * Constructs a google ima ad request and set the slot width/height
     * @returns {google.ima.AdsRequest}
     * @private
     */
    _getNewAdsRequest() {
        const { google } = this._store.getState().window;
        const offsetParent = this._videoContainer.offsetParent;

        const adsRequest = new google.ima.AdsRequest();

        adsRequest.linearAdSlotWidth = offsetParent.offsetWidth;
        adsRequest.linearAdSlotHeight = offsetParent.offsetHeight;
        adsRequest.nonLinearAdSlotWidth = offsetParent.offsetWidth;
        adsRequest.nonLinearAdSlotHeight = offsetParent.offsetHeight;

        return adsRequest;
    }

    /**
     * Internal method for requesting ads
     * @param {AdsRequestContext} adsRequestContext
     * @returns {Promise}
     * @private
     */
    _requestAdsInternal(adsRequestContext) {
        let localContext = adsRequestContext;

        return this._requestAaxBids(localContext).then(amazonBids => {
            localContext = this._aaxManager.getAdRequestContextWithAmazonBids(amazonBids, localContext);

            // before even trying to request ads, so as to prevent any level of error
            // from interrupting the sending of this tracking event, send the
            // video_ad_request event.
            const { Date } = this._store.getState().window;
            this._adRequestTime = Date.now();
            sendAdSpadeEvent(
                this._store,
                IMA_PROVIDER,
                AdSpadeEvents.AD_REQUEST,
                initializeAdSpadeEvent(localContext)
            );

            const adsRequest = this._getNewAdsRequest();
            adsRequest.adTagUrl = buildIMATags(localContext);
            this._adsLoader.requestAds(adsRequest, localContext);
        }).catch(e => {
            const spadeEvent = initializeAdSpadeEvent(localContext);
            spadeEvent.reason = e.message;

            sendAdSpadeEvent(this._store, IMA_PROVIDER, AdSpadeEvents.AD_REQUEST_ERROR, spadeEvent);
        });
    }

    /**
     * Request a specific ad by creative ID
     * Playing a specific creative is a dev option and should not be a normal use case
     * @param {AdsRequestContext} adsRequestContext
     * @returns {Promise}
     * @private
     */
    _requestAdsInternalWithCreativeId(adsRequestContext) {
        return this._getVastXmlResponse(adsRequestContext).then(response => {
            const adsRequest = this._getNewAdsRequest();
            adsRequest.adTagUrl = '';
            adsRequest.adsResponse = response;
            this._adsLoader.requestAds(adsRequest, adsRequestContext);
        }).catch(error => {
            // eslint-disable-next-line no-console
            console.error(`Error requesting creative=${adsRequestContext.creativeId}`, error.message);

            // Failed to request a specific ad creative, default back to playing normal ads
            // This is to prevent users from bypassing ads by passing in invalid creativeIds
            return this._requestAdsInternal(adsRequestContext);
        });
    }

    /**
     * Requests bids from Amazon Ad Exchange
     * @param {AdsRequestContext} adsRequestContext
     * @returns {Promise}
     * @private
     */
    _requestAaxBids(adsRequestContext) {
        return this._aaxManager.fetchBids(adsRequestContext).catch(e => {
            // If AAX throws error, send spade event but resolve promise so we continue with normal ad request flow
            const spadeEvent = initializeAdSpadeEvent(adsRequestContext);
            spadeEvent.reason = e.message;
            sendAdSpadeEvent(this._store, IMA_PROVIDER, AdSpadeEvents.AAX_AD_AUCTION_ERROR, spadeEvent);
            return Promise.resolve([]);
        });
    }

    _getVastXmlResponse(adsRequestContext) {
        const dfpCreativeService = new DfpCreativeService();
        return dfpCreativeService.getVastXmlResponse(adsRequestContext.creativeId);
    }

    /**
     * Checks if the creativeId is set
     * @param {String} creativeId
     * @returns {boolean}
     * @private
     */
    _isCreativeIdSet({ creativeId }) {
        return creativeId && creativeId > 0;
    }

    /**
     * Set a preferred ad bitrate based on the stream's bitrate
     * @param playbackRate
     * @returns {number}
     * @private
     */
    _getTargetAdBitrate(playbackRate) {
        const targetBitrate = playbackRate * 0.8;
        if (targetBitrate >= AD_BITRATE_MAX) {
            return AD_BITRATE_MAX;
        } else if (targetBitrate < AD_BITRATE_MIN) {
            return AD_BITRATE_MIN;
        }
        return targetBitrate;
    }

    /**
     * Construct rendering settings for ads
     * @returns {google.ima.AdsRenderingSettings}
     * @private
     */
    _getAdRenderSettings() {
        const { google } = this._store.getState().window;
        const { playbackRate } = this._backend.getStats();
        const adsRenderingSettings = new google.ima.AdsRenderingSettings();
        adsRenderingSettings.restoreCustomPlaybackStateOnAdBreakComplete = true;
        if (playbackRate) {
            adsRenderingSettings.bitrate = this._getTargetAdBitrate(playbackRate);
        }
        return adsRenderingSettings;
    }

    /**
     * Configures the correct VPAID mode based on browser
     */
    _configureVpaidMode() {
        const { google } = this._store.getState().window;
        // Disable VPAID entirely on MS Edge browsers
        if (bowser.msedge) {
            google.ima.settings.setVpaidMode(google.ima.ImaSdkSettings.VpaidMode.DISABLED);
        } else {
            google.ima.settings.setVpaidMode(google.ima.ImaSdkSettings.VpaidMode.INSECURE);
        }
    }

    // TODO: move this to a Playback state manager
    get paused() {
        return this._paused;
    }

    get sdk() {
        return IMA_SDK_NAME;
    }
}

/**
 * Interface adapter from player backend to the `contentPlayback` field in
 * `AdsManagerLoadedEvent.getAdsManager()`
 */
class ContentPlayback {
    /**
     * @param {Backend} backend
     */
    constructor(backend) {
        this._backend = backend;
    }

    /**
     * @return {Number}
     */
    get currentTime() {
        return this._backend.getCurrentTime();
    }

    /**
     * @return {Number}
     */
    get duration() {
        return this._backend.getDuration();
    }
}

class NullGoogleAdsManager {
    init() {}
    destroy() {}
    start() {}
    resize() {}
    pause() {}
    resume() {}
    setVolume() {}
    addEventListener() {}

    getRemainingTime() {
        return -1;
    }
}

const NULL_GOOGLE_ADS_MANAGER = new NullGoogleAdsManager();
