import reduce from 'lodash/reduce';
import { doubleClickVODUrl, doubleClickLiveUrl, imaNetworkID } from 'settings';
import { PLAYER_EMBED } from '../util/player-type';
import { parseUri } from '../util/parseuri';
import { getContentInfo } from '../analytics/util';
import { CONTENT_MODE_VOD } from '../stream/twitch-vod';
import includes from 'lodash/includes';

export const PREROLL = 'preroll';
export const MIDROLL = 'midroll';
export const POSTROLL = 'postroll';

const AD_POSITIONS = {
    [PREROLL]: '1',
    [MIDROLL]: '2',
    [POSTROLL]: '3',
};

// These keys’ values are already formatted and have the key embedded in them. Only append the values
const PREFORMATTED_PARAMS = [
    'qsParams',
];

/**
 * Construct the IMA tag URL for the given ads request context.
 *
 * @param {AdsRequestContext} context
 * @return {String} ad tag url
 */
export function buildIMATags(context, adUnit = 'twitch.web') {
    // more info about DFP tags here https://support.google.com/dfp_sb/answer/1068325?hl=en
    /* eslint-disable camelcase */
    const params = {
        iu: `/${imaNetworkID}/${adUnit}/client/desktop/live/${context.channelId}`,
        ius_szs: '300x250',
        sz: '640x480',
        impl: 's', // 's' means sync Request mode
        gdfp_req: '1', // indicates that the user is on the DFP schema, and not GAM (legacy)
        env: 'vp', // 'vp' means Video Player
        output: 'xml_vast3', // VAST ads
        url: context.url,
        correlator: Date.now(),
        cust_params: getCustomParameters(context),
        unviewed_position_start: '1', // turn on delayed impressions for video
    };

    /* eslint-enable camelcase */

    // why is this using a custom reducer that explicitly avoids encodeURIComponent?
    // Probably because whoever works on the IMA SDK product doesn't understand
    // how URIs are supposed to be encoded.
    // https://support.google.com/dfp_premium/answer/1080597
    const queryParam = reduce(params, (acc, value, key) => {
        acc.push(`${key}=${value}`);
        return acc;
    }, []).join('&');

    if (context.contentType === CONTENT_MODE_VOD) {
        return `${doubleClickVODUrl}?${queryParam}`;
    }

    return `${doubleClickLiveUrl}?${queryParam}`;
}

/**
 * Strip targeting strings of special characters and spaces
 *
 * @param {String} tag
 * @return {String}
 */
export function formatCustomTag(tag) {
    return (typeof tag === 'string') ? tag.toLowerCase().replace(/[^a-z0-9]+/g, '_') : '';
}

/**
 * Assemble custom parameters for the ad request from the request context.
 *
 * @param {AdsRequestContext} context
 * @return {Object}
 */
function getCustomParameters(context) {
    let customParams = {
        partner: context.partner,
        game: context.game,
        chan: context.channel,
        chanid: context.channelId,
        deviceId: context.deviceId,
        twitchCorrelator: context.twitchCorrelator,
        embed: (context.playerType === PLAYER_EMBED),
        platform: context.platform,
        playerType: context.playerType,
        mature: context.mature,
        pos: AD_POSITIONS[context.adType],
        timebreak: context.duration,
        adblock: (context.adblock === (void 0) ? 'unknown' : context.adblock),
        sdk: context.sdk,
        padtime: context.padtime,
        /* eslint-disable camelcase */
        broadcaster_software: context.broadcasterSoftware,
    };

    if (context.skippableExpGroup !== 'control') {
        customParams.skip = context.skippableExpGroup;
    }

    if (context.communityIds !== null) {
        customParams.community = context.communityIds;
    }

    if (context.kruxId !== null) {
        customParams.kuid = context.kruxId;
    }

    if (context.contentType === CONTENT_MODE_VOD) {
        /* eslint-disable camelcase */
        customParams.vod_id = context.vod.id;
        customParams.vod_name = context.vod.name;
        customParams.vod_type = context.vod.type;
        /* eslint-enable camelcase */
    }

    if (customParams.embed) {
        /* eslint-disable camelcase */
        const cleanedEmbedUrl = cleanEmbedUrl(context.embedUrl);
        customParams.embed_url = cleanedEmbedUrl;
        customParams.curse_embed = cleanedEmbedUrl;
        /* eslint-enable camelcase */
    }

    const { live, contentMode } = getContentInfo(context.broadcasterSoftware);
    customParams.live = live;
    // eslint-disable-next-line camelcase
    customParams.content_mode = contentMode;

    customParams = getCustomParamsWithAmazonFields(context, customParams);

    // why is this using a custom reducer instead of $.param or some other utility?
    // behold, a failure to understand encoding:
    // https://support.google.com/dfp_premium/answer/1080597
    return reduce(customParams, (params, value, key) => {
        let stringValue = String(value);
        if (includes(PREFORMATTED_PARAMS, key)) {
            // remove the first & so there aren't double &s when joining
            params.push(`${stringValue.replace(/^&/,'')}`);
        } else {
            stringValue = stringValue.toLowerCase();
            params.push(`${key}=${stringValue}`);
        }
        return params;
    }, []).join('&').replace(/[=&, ]/g, ch => {
        switch (ch) {
        case '=':
            return '%3D';
        case '&':
            return '%26';
        case ',':
            return '%2C';
        case ' ':
            return '_';
        }
    });
}

// Grabbed from https://www.w3resource.com/javascript/form/ip-address-validation.php
function validateIPAddress(ipaddress) {
    // eslint-disable-next-line max-len
    if (/^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/.test(ipaddress)) {
        return true;
    }

    return false;
}

function cleanEmbedUrl(url) {
    const parsedUri = parseUri(url);
    if (validateIPAddress(parsedUri)) {
        return parsedUri;
    }

    return parsedUri.host.split('.').slice(-2).join('.');
}

/**
 * Set custom params related to Amazon Ad Exchange
 * @param {AdsRequestContext} context
 * @param {Object} customParams
 */
function getCustomParamsWithAmazonFields(context, customParams) {
    const localCustomParams = customParams;
    if (context.qsParams) {
        localCustomParams.qsParams = context.qsParams;
    }

    return localCustomParams;
}
