import assign from 'lodash/assign';
import extend from 'lodash/extend';
import pick from 'lodash/pick';
import find from 'lodash/find';
import findIndex from 'lodash/findIndex';
import { getContentInfo } from './util';
import { MinutesWatched, EVENT_MINUTE } from '../util/minutes-watched';
import { sessionStore } from '../util/storage';
import { CountessTracker } from './countess';
import { LatencyTracker } from './latency';
import { ComscoreAnalytics } from './comscore';
import * as Settings from '../settings';
import {
    userInfo,
    channelAPIInfo,
    channelViewerInfo,
    channelInfo,
    getCommunitiesFromChannel,
    krakenUserInfo,
    videoInfo,
} from '../api';
import * as ComscoreActions from '../actions/comscore';
import { AdContentTypes } from '../actions/ads';
import { ValveClient } from './valve';
import { NETWORK_PROFILE_COLLECTION } from '../experiments';
import { subscribe } from '../util/subscribe';
import { PLAYER_POPOUT, PLAYER_SITE, PLAYER_SITE_MINI } from '../util/player-type';
import { resetQualityChangeCount } from '../actions/analytics';
import { setCommunitiesData } from '../actions/communities';
import * as TwitchEvents from '../backend/events/twitch-event';
import * as MediaEvents from '../backend/events/media-event';
import { TIER_1_EVENTS, VIDEO_PLAYBACK_ERROR } from './events';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { CONTENT_MODE_PROVIDED } from 'stream/provided';
import Errors from 'errors';
import {
    COLLECTION_SESSION_ID_KEY,
    COLLECTION_SESSION_KEY,
    updateCollectionSessionTimestamp,
} from '../actions/collection';

export const REBROADCAST_TYPE = 'rebroadcast';
export const VIDEO_PLAY_LOAD_START = 'video_play_load_start';
export const VIDEO_PLAY_OAUTH = 'video_play_oauth';
export const VIDEO_PLAY_NAUTH = 'video_play_nauth';
export const VIDEO_PLAY_MASTER_MANIFEST = 'video_play_master_manifest';
export const VIDEO_PLAY_VARIANT_MANIFEST = 'video_play_variant_manifest';

export const SITE_MINIPLAYER_ACTION = 'site_mini_player_action';

const VIDEO_INIT = 'video_init';
const VIDEO_END = 'video_end';
const VOD_SEEK = 'vod_seek';
const CAPTION_PRESET = 'player_caption_preset';

const DISPLAY_MODE_THEATRE = 'theatre mode';
const DISPLAY_MODE_FULLSCREEN = 'fullscreen';
const DISPLAY_MODE_POPOUT = 'popout';
const DISPLAY_MODE_MINI = 'mini';

export const BROADCAST_PLATFORM_WATCH_PARTY = 'watch_party';
export const BROADCAST_PLATFORM_WATCH_PARTY_RERUN = 'watch_party_rerun';
export const BROADCAST_PLATFORM_WATCH_PARTY_PREMIERE = 'watch_party_premiere';

const MINIMUM_SEEK_TIME = 500;

const DEFAULT_VIDEO_INFO = Object.freeze({
    // eslint-disable-next-line camelcase
    segment_protocol: 'https',
});
/**
 * This function removes the `(Variant)` name from the
 * Auto quality. We are doing this to ensure that our tracker
 * will not be confused with multiple kinds of Auto quality.
 *
 * @param {String} qualityStr
 * @return {String}
 */
export function removeVariantName(str) {
    const variantName = String(str).trim();
    const wordBoundaryIndex = variantName.indexOf(' ');

    if (wordBoundaryIndex > 0) {
        return variantName.slice(0, wordBoundaryIndex);
    }
    return variantName;
}

export class Analytics {
    // eslint-disable-next-line max-statements
    constructor(player, tracker, state, store, options) {
        this.player = player;
        this.tracker = tracker;
        this.state = state;
        this.store = store;
        this.options = options;
        this.hasPlayed = false;
        this.bufferEmptyStartTime = null;
        this.bufferEmptyCount = 0;
        this.lastNetworkProfile = -Infinity;
        this.lastSeekTime = null;
        this.timeStampBeforeSeek = 0;
        this.isSeekInProgress = false;
        this.trackNetworkProfile = this.store.getState().experiments.get(NETWORK_PROFILE_COLLECTION);

        // Create the Countess tracking client.
        this.countessTracker = new CountessTracker({
            host: Settings.countessHost,
        });

        this.comscore = new ComscoreAnalytics(this, this.player, this.store);
        this.latencyTracker = new LatencyTracker(this, 1E-3, this.player, this.store);

        this.valveClient = new ValveClient();
        this.initProperties();
        this.initEvents();

        this.unsubscribes = [];
        this.unsubscribes.push(this._subscribeCaptions(this.store));
        // eslint-disable-next-line max-len
        this.unsubscribes.push(subscribe(this.store, ['quality.current', 'quality.selected'], this.onQualityChange.bind(this)));
        this.unsubscribes.push(subscribe(this.store, ['collection.id'], this.onCollectionChange.bind(this)));
        this.unsubscribes.push(subscribe(this.store, ['ui.isMini'], this.onMiniChange.bind(this)));
        this.unsubscribes.push(subscribe(this.store, ['playback.ended'], this.onPlaybackEnded.bind(this)));
        this.unsubscribes.push(subscribe(this.store, ['error'], this.onError.bind(this)));
        this.unsubscribes.push(subscribe(this.store, ['stream'], this.onStreamChange.bind(this)));
        this.unsubscribes.push(subscribe(
            this.store, ['watchParty.vodId'], this.onWatchPartyVodChange.bind(this))
        );
        this.unsubscribes.push(subscribe(
            this.store, ['watchParty.watchPartyId'], this.onWatchPartyChange.bind(this))
        );
        this.unsubscribes.push(subscribe(
            this.store, ['chromecast.castingState'], this.onCastingChange.bind(this))
        );
        this.onError();
    }

    _subscribeCaptions(store) {
        return subscribe(store, ['captions'], ({ captions: newCap }, { captions: oldCap }) => {
            if (newCap.preset !== oldCap.preset) {
                this.trackEvent(CAPTION_PRESET, {
                    // eslint-disable-next-line camelcase
                    captions_preset: newCap.preset,
                });
            }
        });
    }

    initProperties() {
        // TODO: Fill in both properties when they are
        // implemented on the provided content stream
        this.tracker.setProperties({
            content_id: '',// eslint-disable-line camelcase
            customer_id: '',// eslint-disable-line camelcase
        });

        // Gather user info from the API.
        this.tracker.setProperties(
            userInfo().then(function(info) {
                return {
                    login: info.login,
                    turbo: Boolean(info.turbo),
                };
            })
        );
        // also need some user info from kraken API...
        this.tracker.setProperties(
            krakenUserInfo().then(function(info) {
                return {
                    // eslint-disable-next-line camelcase
                    user_id: info._id,
                    staff: (info.type === 'staff'),
                };
            })
        );

        // Unable to use getChannel as it hasn't allocated a value in time...
        if (this.options.channel) {
            const channelInfo = channelAPIInfo(this.options.channel);
            this.valveClient.channelInfo(channelInfo);

            // Gather channel viewer info if live.
            const viewerInfo = channelViewerInfo(this.options.channel);
            this.valveClient.viewerInfo(viewerInfo);
            this.tracker.setProperties(
                viewerInfo.then(function(info) {
                    return {
                        subscriber: Boolean(info.chansub),
                    };
                })
            );
        }
    }

    initMinutesWatchedTimer() {
        this.minutesWatchedTimer = new MinutesWatched();
        this.minutesWatchedTimer.on(EVENT_MINUTE, this.onMinuteWatched.bind(this));
    }

    initEvents() {
        this.initMinutesWatchedTimer();

        this.player.addEventListener(TwitchEvents.PLAYER_INIT, this.onBackendInit.bind(this));
        this.player.addEventListener(MediaEvents.LOADSTART, this.onLoadStart.bind(this));
        this.player.addEventListener(MediaEvents.PAUSE, this.onPause.bind(this));
        this.player.addEventListener(MediaEvents.WAITING, this.onWaiting.bind(this));
        this.player.addEventListener(MediaEvents.PLAYING, this.onPlaying.bind(this));
        this.player.addEventListener(MediaEvents.PLAYING, this.onQualityChange.bind(this));
        this.player.addEventListener(MediaEvents.ENDED, this.onEnded.bind(this));
        this.player.addEventListener(MediaEvents.SEEKING, this.onSeeking.bind(this));
        this.player.addEventListener(MediaEvents.TIME_UPDATE, this.onTimeUpdate.bind(this));
        this.player.addEventListener(MediaEvents.CAN_PLAY, this.onCanPlay.bind(this));
        this.player.addEventListener(TwitchEvents.ABS_STREAM_FORMAT_CHANGE, this.onABSStreamFormatChange.bind(this));
    }

    /**
     * Fired when ABS requests a new playback quality
     *
     * @param {Object} qualityChangeStats
     */
    onABSStreamFormatChange(qualityChangeStats) {
        const { streamMetadata, manifestInfo } = this.store.getState();
        this.tracker.trackEvent(TwitchEvents.ABS_STREAM_FORMAT_CHANGE, assign({}, qualityChangeStats, {
            minutes_logged: this.minutesWatchedTimer.totalMinutes, // eslint-disable-line camelcase
            broadcast_id: streamMetadata.broadcastID, // eslint-disable-line camelcase
            manifest_broadcast_id: manifestInfo.broadcast_id, // eslint-disable-line camelcase
        }));
    }

    /**
     * Compute the aggregate stats for a given network profile, in the format
     * sent as part of minute-watched.
     *
     * @param {Array<NetworkProfile>} networkProfile
     * @return {Object}
     */
    getNetworkProfileStats(networkProfile) {
        if (networkProfile.length === 0) {
            return {};
        }

        function add(a, b) {
            return a + b;
        }

        const firstByteLatency = (networkProfile.map(np => np.firstByteLatency)).reduce(add);
        const segmentDuration = (networkProfile.map(np => np.segmentDuration)).reduce(add);
        const downloadDuration = (networkProfile.map(np => np.downloadDuration)).reduce(add);
        const bytes = (networkProfile.map(np => np.bytes)).reduce(add);

        /* eslint-disable camelcase */
        return {
            transport_segments: networkProfile.length,
            transport_first_byte_latency: firstByteLatency,
            transport_segment_duration: segmentDuration,
            transport_download_duration: downloadDuration,
            transport_download_bytes: bytes,
        };
        /* eslint-enable camelcase */
    }

    trackEvent(name, props) {
        const videoInfo = assign({}, DEFAULT_VIDEO_INFO, this.player.getVideoInfo());

        // There are 30+ properties stuffed into videoInfo, so filter them.
        // Currently not used: buffer_empty_count, deblocking, format,
        // hls_buffer_duration, hls_load_attempted, implementation,
        // paused, play_load_time, playback_bytes_per_second,
        // playing, smoothing, stream_protocol, stream_time,
        // time, video_load_attempt

        const properties = extend(
            {},
            pick(videoInfo, [
                'bandwidth', // TODO Replace with bitrate.
                'cluster', // Cluster serving the video content.
                'current_bitrate', // Average bitrate in kb/s
                'current_fps', // Average fps.
                'dropped_frames', // Number of unrendered/skipped frames.
                'hls_latency_broadcaster', // Delay between streamer -> viewer.
                'hls_latency_encoder', // Delay between encoder -> viewer.
                'hls_latency_broadcaster_send_time', // Timestamp of an I-frame sent by broadcaster
                'hls_latency_ingest_receive_time', // Timestamp of the I-frame received by ingest
                'hls_latency_ingest_send_time', // Timestamp of the I-frame sent off by ingest
                'hls_latency_transcode_receive_time', // Timestamp of the I-frame received by transcoder
                'hls_latency_transcode_send_time', // Timestamp of the I-frame sent off by transcoder
                'hls_target_duration', // Maximum segment duration.
                'manifest_cluster', // Cluster serving the manifest.
                'manifest_node', // Node serving the manifest.
                'manifest_node_type', // Type of manifest_node
                'serving_id', // unique id corresponding to the manifest request.
                'node', // Node serving the video content.
                'user_ip', // User's IP address.
                'vid_display_height', // Size of the video container.
                'vid_display_width',
                'vid_height', // Size of the actual video content. (ex 720p)
                'vid_width',
                'video_buffer_size', // Seconds of video ready to display.
                'vod_cdn_origin', // The origin from which the CDN fetched the VoD. (Typically s3 or swift)
                'vod_cdn_region', // The region of the viewer as per lantern. Returned in VOD Master Manifest.
            ]),
            {
                // The volume of video playback (moment of trackevent)
                volume: this.player.getVolume(),
                // The muted state of video playback (moment of trackevent)
                muted: this.player.getMuted(),
                // Whether the video segments themselves are retrieved via https
                // eslint-disable-next-line camelcase
                is_https: (videoInfo.segment_protocol === 'https'),
            },
            props
        );

        this.tracker.trackEvent(name, properties);
    }

    /**
     * Fired every "minute" while the user is actively trying to watch
     * video, starting when the first frame is visible (video-play) and
     * only stopping on pause or ended.
     */
    onMinuteWatched() {
        const {
            analytics,
            captions,
            communities,
            streamMetadata,
            collection,
            stream,
            manifestInfo,
            lang,
        } = this.store.getState();

        // Filter the network profile data to only new segment stats
        const networkProfile = this.player.getNetworkProfile().filter(function(stats) {
            return stats.startTime > this.lastNetworkProfile;
        }.bind(this));
        this.lastNetworkProfile = networkProfile.reduce(function(max, stats) {
            return Math.max(max, stats.startTime);
        }, this.lastNetworkProfile);

        const networkProfileStats = this.getNetworkProfileStats(networkProfile);

        /* eslint-disable camelcase */
        // if VOD add current timestamp
        if (this.player.video) {
            extend(networkProfileStats, {
                vod_timestamp: this.player.getCurrentTime(),
            });
        }

        const minuteWatchedProperties = {
            seconds_offset: this.minutesWatchedTimer.initialDelay / 1000,
            minutes_logged: this.minutesWatchedTimer.totalMinutes,
            captions_enabled: captions.enabled,
            quality_change_count: analytics.qualityChangeCount,
            player_size_mode: this._getPlayerDisplayMode(),
            broadcast_id: streamMetadata.broadcastID,
            manifest_broadcast_id: manifestInfo.broadcast_id,
            community_ids: communities.ids,
            transcoder_type: manifestInfo.transcodestack,
            autoplayed: this.options.autoplay,
            streamType: streamMetadata.streamType,
            language: lang.langCode,
        };

        if (stream.contentType === CONTENT_MODE_VOD && collection.id) {
            this.store.dispatch(updateCollectionSessionTimestamp());
            const collectionSession = sessionStore.get(COLLECTION_SESSION_KEY);
            const collectionSessionId = collectionSession ? collectionSession[COLLECTION_SESSION_ID_KEY] : '';

            /* eslint-disable max-len, camelcase */
            minuteWatchedProperties.collection_item_position = findIndex(collection.items, { itemId: stream.videoId.substr(1) });
            minuteWatchedProperties.collection_id = collection.id;
            minuteWatchedProperties.collection_session_id = collectionSessionId;
            /* eslint-enable max-len, camelcase */
        }

        this.trackEvent(TIER_1_EVENTS.MINUTE_WATCHED, extend({}, networkProfileStats, minuteWatchedProperties));
        /* eslint-enable camelcase */

        this.store.dispatch(resetQualityChangeCount());

        // Notify Valve of Minute-Watched event
        this.valveClient.notify();

        this.trackNetworkProfile.then(collect => {
            if (collect === 'yes' && networkProfile.length > 0) {
                this.tracker.trackEvents(networkProfile.map(function(stats) {
                    /* eslint-disable camelcase */
                    return {
                        event: 'network_profile',
                        properties: {
                            bytes: stats.bytes,
                            duration: stats.segmentDuration,
                            // eslint-disable-next-line max-len
                            time: (stats.startTime + stats.firstByteLatency + stats.downloadDuration),
                            time_to_first_byte: stats.firstByteLatency,
                            download_duration: stats.downloadDuration,
                            curr_buffer_length: stats.currBufferLength,
                            reason_code: stats.reasonCode,
                            curr_quality: stats.currQuality,
                        },
                    };
                    /* eslint-enable camelcase */
                }));
            }
        });
    }

    // Fired when the backend has fully initialized and is ready to play.
    onBackendInit() {
        this.tracker.setProperty('backend', this.player.getBackend());
        this.tracker.setProperty('backend_version', this.player.getVersion());

        this.tracker.trackEvent(VIDEO_INIT);
    }

    // Fired when the user starts trying to load a channel or video.
    // The channel might actually be offline, we don't know yet.
    onLoadStart() {
        // Initial the tracking properties for this new channel/video.
        if (this.player.getChannel()) {
            this.onLoadChannel();
        } else if (this.player.getVideo()) {
            this.onLoadVideo();
        }

        // (eewayhsu): Whenever we watch a new stream,
        // we will get a new minutesWatchedTimer instance to reset minutes_logged to zero
        this.minutesWatchedTimer.destroy();
        this.initMinutesWatchedTimer();

        // Fire video-play on the next `playing` event.
        this.hasPlayed = false;

        // Reset the buffer empty count and any bufferEmptyStartTime for previous stream.
        this.bufferEmptyCount = 0;
        this.bufferEmptyStartTime = null;
    }

    onLoadChannel() {
        // Fetch the properties we want to track for this channel.
        const channel = this.player.getChannel();
        const request = channelInfo(channel);
        const channelAPIRequest = channelAPIInfo(channel);

        // Count the channel view with countess.
        request.then(function(channelInfo) {
            this.countessTracker.trackView('channel', channelInfo._id);
        }.bind(this));

        let channelData = {};
        let channelAPIData = {};
        const properties = Promise.all([request, channelAPIRequest]).then(function([channelInfo, channelAPIInfo]) {
            channelData = channelInfo;
            channelAPIData = channelAPIInfo;
            return getCommunitiesFromChannel(channel);
        }).then(communitiesData => {
            this.store.dispatch(setCommunitiesData(communitiesData));

            /* eslint-disable camelcase */
            const broadcaster_software = channelAPIData.broadcaster_software;
            const { live, contentMode } = getContentInfo(broadcaster_software);
            return {
                broadcaster_software,
                live,
                channel: channelData.name,
                channel_id: channelData._id,
                game: channelData.game,
                content_mode: contentMode,
                partner: channelData.partner,
                // Set vod_type to undefined to override any other vod_type from vodcast or video.
                vod_type: undefined,
            };
            /* eslint-enable camelcase */
        });

        this.tracker.setProperties(properties);
    }

    onLoadVideo() {
        // Fetch the properties we want to track for this video.
        const video = this.player.getVideo();
        const videoInfoRequest = videoInfo(video);

        // Acquire channel info as well
        this.tracker.setProperties(
            videoInfoRequest.then(function(videoInfo) {
                return channelInfo(videoInfo.channel.name);
            }).then(function(channelInfo) {
                return pick(channelInfo, 'partner');
            })
        );

        // Asynchronously set the analytics properties.
        // This promise avoid sending tracking events until resolved.
        const properties = videoInfoRequest.then(function(videoInfo) {
            /* eslint-disable camelcase */
            return {
                channel: videoInfo.channel.name,
                channel_id: videoInfo.channel._id,
                vod: videoInfo._id, // TODO DELETE when removing x_untrusted
                vod_id: videoInfo._id,
                vod_type: videoInfo.broadcast_type,
                game: videoInfo.game,
                live: false,
                content_mode: 'vod',
            };
            /* eslint-enable camelcase */
        });

        this.tracker.setProperties(properties);
    }

    // Fired when there is not enough video to continue playback.
    onWaiting() {
        const { communities, manifestInfo, streamMetadata } = this.store.getState();

        if (!this.player.getSeeking()) {
            this.bufferEmptyStartTime = (new Date()).getTime();
            this.bufferEmptyCount++;

            this.trackEvent(TIER_1_EVENTS.BUFFER_EMPTY, {
                /* eslint-disable camelcase */
                buffer_empty_count: this.bufferEmptyCount,
                community_ids: communities.ids,
                broadcast_id: streamMetadata.broadcastID,
                manifest_broadcast_id: manifestInfo.broadcast_id,
                /* eslint-enable camelcase */
            });
        }
    }

    // Fired when there is enough data and playback has continued.
    // eslint-disable-next-line complexity, max-statements
    onPlaying() {
        const { analytics, communities, stream, streamMetadata, collection, manifestInfo } = this.store.getState();

        // Fire buffer-refill if we previously fired buffer-empty.
        if (this.bufferEmptyStartTime !== null) {
            const now = (new Date()).getTime();
            this.trackEvent(TIER_1_EVENTS.BUFFER_REFILL, {
                /* eslint-disable camelcase */
                buffering_time: (now - this.bufferEmptyStartTime) / 1000,
                broadcast_id: streamMetadata.broadcastID,
                community_ids: communities.ids,
                manifest_broadcast_id: manifestInfo.broadcast_id,
                /* eslint-enable camelcase */
            });

            this.bufferEmptyStartTime = null;
        }

        if (!this.hasPlayed) {
            const videoPlayParams = {
                /* eslint-disable camelcase */
                time_since_load_start: (Date.now() - analytics.playSessionStartTime),
                community_ids: communities.ids,
                manifest_broadcast_id: manifestInfo.broadcast_id,
                broadcast_id: streamMetadata.broadcastID,
                /* eslint-enable camelcase */
                autoplayed: this.options.autoplay,
            };

            if (stream.contentType === CONTENT_MODE_VOD) {
                // Count the video view with countess.
                videoInfo(stream.videoId).then(videoInfo => {
                    this.countessTracker.trackVODView(videoInfo);
                });

                if (collection.id) {
                    const collectionSession = sessionStore.get(COLLECTION_SESSION_KEY);
                    const collectionSessionId = collectionSession ? collectionSession[COLLECTION_SESSION_ID_KEY] : '';

                    /* eslint-disable max-len, camelcase */
                    videoPlayParams.collection_item_position = findIndex(collection.items, { itemId: stream.videoId.substr(1) });
                    videoPlayParams.collection_id = collection.id;
                    videoPlayParams.collection_session_id = collectionSessionId;
                    /* eslint-enable max-len, camelcase */
                }
            }

            this.trackEvent(TIER_1_EVENTS.VIDEO_PLAY, videoPlayParams);
            this.hasPlayed = true;

            // Notify Valve of video-play event
            this.valveClient.notify();

            this.store.dispatch(ComscoreActions.sendPlayerBeacon());
            this.store.dispatch(ComscoreActions.sendVideoBeacon());
        }

        this.minutesWatchedTimer.start();
    }

    onCanPlay() {
        if (this.isSeekInProgress) {
            this.isSeekInProgress = false;

            /* eslint-disable camelcase */
            this.trackEvent(VOD_SEEK, {
                timestamp_departed: this.timeStampBeforeSeek,
                timestamp_target: this.player.getCurrentTime(),
                time_spent_seeking: (Date.now() - this.lastSeekTime) / 1000,
            });
            /* eslint-enable camelcase */
        }
        this.timeStampBeforeSeek = this.player.getCurrentTime();
    }

    onSeeking() {
        if (!this.hasPlayed || this.isSeekInProgress) {
            return;
        }
        this.isSeekInProgress = true;
        this.lastSeekTime = Date.now();
    }

    onTimeUpdate() {
        if (this.isSeekInProgress) {
            return;
        }

        // Update the timestamp so we can refer to it when the user seeks;
        // otherwise, there's no way to know what the timestamp was
        // before the user initiated a seek
        const currentTime = this.player.getCurrentTime();
        if (currentTime - this.timeStampBeforeSeek < MINIMUM_SEEK_TIME &&
            this.timeStampBeforeSeek < currentTime) {
            this.timeStampBeforeSeek = currentTime;
        }
    }

    // Fired when the user hits the pause button.
    onPause() {
        const { ads, adsManager, stream } = this.store.getState();
        if (ads.currentMetadata.contentType === AdContentTypes.IMA &&
            stream.contentType === CONTENT_MODE_VOD &&
            !adsManager.paused) {
            return;
        }
        this.minutesWatchedTimer.stop();
    }

    // Gets fired in parallel with onEnded, but via store subscribe mechanism
    // Ultimately, this should be merged with onEnded, but this may impact analytics numbers
    onPlaybackEnded() {
        const { playback } = this.store.getState();
        if (playback.ended) {
            this.trackEvent(VIDEO_END, {});
        }
    }

    onError() {
        const { hasError, code } = this.store.getState().error;

        if (!hasError) {
            return;
        }

        this.trackEvent(VIDEO_PLAYBACK_ERROR, {
            reason: Errors.getMessage(code),
            code,
        });
    }

    onStreamChange() {
        const { stream } = this.store.getState();
        let streamProperties = {
            content_id: '', // eslint-disable-line camelcase
            customer_id: '', // eslint-disable-line camelcase
        };

        if (stream.contentType === CONTENT_MODE_PROVIDED) {
            streamProperties = assign(streamProperties, {
                content_id: stream.contentId, // eslint-disable-line camelcase
                customer_id: stream.customerId, // eslint-disable-line camelcase
            });
        }

        this.tracker.setProperties(streamProperties);
    }

    onCollectionChange() {
        const { collection } = this.store.getState();
        if (collection.id) {
            this.countessTracker.trackView('collection', collection.id);
        }
    }

    // Gets fired whenever a watch party VOD changes.
    onWatchPartyVodChange() {
        const { watchParty } = this.store.getState();
        if (watchParty && watchParty.vodId && watchParty.incrementViewCountURL) {
            this.countessTracker.trackVODView({
                increment_view_count_url: watchParty.incrementViewCountURL, // eslint-disable-line camelcase
            });
        }
        // Set vod type regardless of if it's set or not
        if (watchParty) {
            this.tracker.setProperty('vod_type', watchParty.broadcastType);
        }
    }

    onWatchPartyChange() {
        const { watchParty } = this.store.getState();
        if (watchParty && watchParty.watchPartyId) {
            this.countessTracker.trackView('watch_party', watchParty.watchPartyId);
        }
    }

    // Fired when the stream/video is over.
    onEnded() {
        this.minutesWatchedTimer.reset();
        // Don't fire buffer-refill if the stream goes offline
        this.bufferEmptyStartTime = null;
    }

    // Fired when the quality or selectedQuality changes for whatever reason.
    onQualityChange() {
        const { quality } = this.store.getState();

        if (quality.available.length === 0) {
            return;
        }
        const selectedQualityObj = find(quality.available, q => q.group === quality.selected);
        const selectedQualityName = (selectedQualityObj ? selectedQualityObj.name : quality.current);

        this.tracker.setProperty('quality', removeVariantName(selectedQualityName));
        this.tracker.setProperty('stream_format', quality.current);
    }

    // Fired on Chromecast castingState update
    onCastingChange({ chromecast }) {
        const { castingState } = chromecast;
        const { env } = this.store.getState();

        if (castingState === 'connected' || castingState === 'connecting') {
            this.tracker.setProperties({
                player: 'chromecast',
                // eslint-disable-next-line camelcase
                chromecast_sender: env.platform,
            });
        } else {
            this.tracker.setProperties({
                player: env.playerType,
                // eslint-disable-next-line camelcase
                chromecast_sender: null,
            });
        }
    }

    // Fired when the player changes to mini mode
    onMiniChange() {
        const { env, ui }  = this.store.getState();

        if (env.playerType !== PLAYER_SITE) {
            return;
        }

        this.tracker.setProperty('player', ui.isMini ? PLAYER_SITE_MINI : PLAYER_SITE);
    }

    _getPlayerDisplayMode() {
        const { ui, screenMode } = this.store.getState();

        if (screenMode.isFullScreen) {
            return DISPLAY_MODE_FULLSCREEN;
        } else if (this.options.player === PLAYER_POPOUT) {
            return DISPLAY_MODE_POPOUT;
        } else if (this.player.getTheatre()) {
            return DISPLAY_MODE_THEATRE;
        } else if (ui.isMini) {
            return DISPLAY_MODE_MINI;
        }

        return '';
    }

    destroy() {
        this.minutesWatchedTimer.destroy();
        this.comscore.destroy();
        this.unsubscribes.map(fn => {
            fn();
        });
    }
}
