import assign from 'lodash/assign';
import padStart from 'lodash/padStart';
import reduce from 'lodash/reduce';
import * as MediaEvents from '../backend/events/media-event';
import * as AdvertisementEvents from '../ads/advertisement-event';
import { PREROLL, MIDROLL, POSTROLL } from '../ads/ima-tags';
import Comscore from '../../../vendor/streamsense.min.js';
import { videoInfo } from '../api';
import { LiveTwitchContentStream, CONTENT_MODE_LIVE } from '../stream/twitch-live';
import { VODTwitchContentStream, CONTENT_MODE_VOD } from '../stream/twitch-vod';
import { resetTag } from '../actions/comscore';
import { AdContentTypes } from '../actions/ads';

// Stream genres that are not considered "gaming"
const GENRE_WHITELIST = {
    Creative: 'Creative',
    Poker: 'Poker',
    Music: 'Music',
};

// Default genre when the game's genre is not explicitly whitelisted
const DEFAULT_GENRE = 'Gaming';

// Duration cutoff to distinguish between "long form" and "short form" content,
// in milliseconds
const LONG_FORM_DURATION = 10 * 60 * 1000; // 10 minutes

// Standard Content Asset Metadata
const STANDARD_CONTENT_ASSET_METADATA = {
    /* eslint-disable camelcase */
    // Clip Length
    // Length of the media asset (content or ad) in milliseconds
    ns_st_cl: null,

    // Station Title
    // Title of the station or channel for which content was recorded or where
    // content is made available
    ns_st_st: null,

    // Publisher / Brand Name
    // The consumer-facing brand name of the media publisher that owns the content
    ns_st_pu: null,

    // Program Title
    // Top level content title (i.e., the name of the overall program, show, or
    // content series). Can be used with ns_st_ep to tag TV shows on program and
    // episode level.
    ns_st_pr: null,

    // Episode Title
    // Sub level content title (i.e., the title of the specific episode). Can be
    // used with ns_st_pr to tag TV shows on program and episode level.
    ns_st_ep: null,

    // Episode Season Number
    // The season number for episodic content. It is recommended to use a
    // value with 2 digits. Can be omitted or left blank for non-episodic content.
    ns_st_sn: null,

    // Episode Number
    // The episode number for episodic content. It is recommended to use a value
    // with 2 digits
    ns_st_en: null,

    // Content Genre
    // Content genre description
    ns_st_ge: null,

    // TMS / Gracenote ID
    // Standardized identifier for a content video asset. Episode-specific TMS
    // ID (i.e., a value starting with EP) is expected for episodic content. When
    // there are no episodes associated with a show (e.g., The Oscars Live)
    // then a show-specific TMS ID (a value starting with SH) is required.
    ns_st_ti: null,

    // Ad Load Flag
    // Indicates whether the streaming content carries the same ad load that
    // was used during the TV airing. The flag is considered 'set' when the label
    // is present with a non-empty, non-zero value.
    ns_st_ia: false,

    // Complete Episode Flag
    // Identifies a content stream to be a full episode rather than an excerpt.
    // The flag is considered 'set' when the label is present with a non-empty,
    // non-zero value.
    ns_st_ce: false,

    // Digital Airdate
    // The date on which the content was made available for streaming
    // consumption (in yyyy-mm-dd format)
    ns_st_ddt: null,

    // TV Airdate
    // The date on which the content aired on TV (in yyyy-mm-dd format)
    ns_st_tdt: null,
    //
    // Video Metrix Classification Values
    // c values are custom values decided by Ad Ops and comScore
    c3: 'TWITCH',
    c4: 'twitch.tv',
    c6: null,
    /* eslint-enable */
};

// comScore collects metadata about content and ads watched to determine our worth
// to ad networks and assigns a score, which they use to determine whether to spend
// money on ads with us.
export class ComscoreAnalytics {
    /**
     * Create Comscore object that listens for playback events and notifies comScore.
     * Docs: https://twitchtv.atlassian.net/secure/attachment/29326/Streaming_Tag-JS-ReducedReq-API-VideoMetrix-US.pdf
     *
     * @param {Object} analytics Reference to the analytics object.
     * @param {Object} player Reference to the player which emits video events.
     * @param {Object} store Reference to the store for data about playback state.
     * @return {Comscore}
     */
    constructor(analytics, player, store) {
        this.analytics = analytics;
        this.player = player;
        this.store = store;
        // Track ended state to determine when content is replayed
        this.ended = false;

        const { window } = store.getState();

        player.addEventListener(MediaEvents.PLAYING, this.onPlaying.bind(this));
        player.addEventListener(MediaEvents.PAUSE, this.onPause.bind(this));
        player.addEventListener(MediaEvents.SEEKING, this.onSeeking.bind(this));
        player.addEventListener(MediaEvents.ENDED, this.onEnded.bind(this));
        player.addEventListener(AdvertisementEvents.AD_START, this.onAdStart.bind(this));
        player.addEventListener(AdvertisementEvents.AD_IMPRESSION, this.onAdImpression.bind(this));
        player.addEventListener(AdvertisementEvents.AD_IMPRESSION_COMPLETE, this.onAdImpressionComplete.bind(this));
        player.addEventListener(AdvertisementEvents.AD_END, this.onAdEnd.bind(this));
        window.addEventListener('beforeunload', this);
    }

    handleEvent(event) {
        switch (event.type) {
        case 'beforeunload':
            this._stop();
        }
    }

    onPlaying() {
        const state = this.store.getState();
        const { contentType } = state.stream;

        if (
            state.ads.currentMetadata.contentType === AdContentTypes.NONE &&
            (contentType === CONTENT_MODE_VOD || contentType === CONTENT_MODE_LIVE)
        ) {
            Promise.all([getAssetMetadata(state), getAssetMediaType(state)]).
                then(([metadata, mediaType]) => {
                    state.comscore.streamingTag.playVideoContentPart(toComscoreValues(metadata), mediaType);
                });
        }
        this.ended = false;
    }

    onPause() {
        const { ads } = this.store.getState();

        if (ads.currentMetadata.contentType === AdContentTypes.NONE) {
            this._stop();
        }
    }

    onSeeking() {
        // VODs seek before playing--don't call stop if it's initially playing
        if (this.analytics.hasPlayed && !this.player.paused) {
            // If seeking after the end of content (only possible in VODs), tag should reset
            if (this.ended) {
                this.store.dispatch(resetTag());
                this.ended = false;
            } else {
                this._stop();
            }
        }
    }

    onEnded() {
        // TODO: 'ended' should fire post-postroll, but doesn't. (VP-845)
        // When fixed, resetTag should be dispatched here instead and remove all of the this.ended business.
        this._stop();
        this.ended = true;
    }

    // Fired at the beginning of an ad pod, indicates end of content section.
    onAdStart(data) {
        if (data.roll_type !== POSTROLL) {
            // TODO: stop has already been called by onEnded, remove check after VP-845
            // Content stop
            this._stop();
        }
    }

    // Fired at the beginning of an individual ad in an ad pod.
    onAdImpression(data) {
        const state = this.store.getState();
        state.comscore.streamingTag.playVideoAdvertisement(
            toComscoreValues(getAdvertisementMetadata(data)),
            getAdvertisementType(state, data)
        );
    }

    // Fired at the end of an individual ad in an ad pod.
    onAdImpressionComplete() {
        this._stop();
    }

    // Fired at the end of an ad pod, indicates potential start of content section.
    onAdEnd(data) {
        if (data.roll_type !== POSTROLL) {
            // Content start
            this.onPlaying();
        }
    }

    onBeforeUnload() {
        this._stop();
    }

    destroy() {
        this._stop();
    }

    _stop() {
        this.store.getState().comscore.streamingTag.stop();
    }
}

/**
 * Assemble the asset metadata for the current media asset.
 *
 * @param {Object} state
 * @return {Promise<Object>}
 */
function getAssetMetadata(state) {
    if (state.stream instanceof LiveTwitchContentStream) {
        return Promise.resolve(assign({}, STANDARD_CONTENT_ASSET_METADATA, {
            /* eslint-disable camelcase */
            ns_st_ci: state.streamMetadata.broadcastID,
            ns_st_st: state.streamMetadata.channel.displayName,
            ns_st_ep: state.streamMetadata.channel.status,
            ns_st_ge: (GENRE_WHITELIST[state.streamMetadata.game] || DEFAULT_GENRE),
            ns_st_ddt: toComscoreDate(new Date(state.streamMetadata.createdAt), true),
            ns_st_ce: true,
            /* eslint-enable */
        }));
    } else if (state.stream instanceof VODTwitchContentStream) {
        return videoInfo(state.stream.videoId).then(info => {
            return assign({}, STANDARD_CONTENT_ASSET_METADATA, {
                /* eslint-disable camelcase */
                ns_st_ci: info.broadcast_id,
                ns_st_cl: info.length * 1000,
                ns_st_st: info.channel.display_name,
                ns_st_ep: info.title,
                ns_st_ge: (GENRE_WHITELIST[info.game] || DEFAULT_GENRE),
                ns_st_ddt: toComscoreDate(new Date(info.recorded_at), true),
                ns_st_ce: (info.broadcast_type === 'archive'),
                /* eslint-enable */
            });
        });
    }
    return Promise.reject(new Error('No stream is currently playing; no metadata is available.'));
}

/**
 * Gets the Comscore media type for a given media asset.
 *
 * @param {Object} state Current state of the application
 * @return {Comscore.StreamingTag.ContentType}
 */
function getAssetMediaType(state) {
    if (state.stream instanceof LiveTwitchContentStream) {
        if (state.streamMetadata.channel.partner) {
            return Comscore.StreamingTag.ContentType.Live;
        }
        return Comscore.StreamingTag.ContentType.UserGeneratedLive;
    } else if (state.stream instanceof VODTwitchContentStream) {
        return videoInfo(state.stream.videoId).then(info => {
            const longForm = (info.length * 1000 >= LONG_FORM_DURATION);
            if (info.partner) {
                return (longForm ?
                    Comscore.StreamingTag.ContentType.LongFormOnDemand :
                    Comscore.StreamingTag.ContentType.ShortFormOnDemand);
            }
            return (longForm ?
                Comscore.StreamingTag.ContentType.UserGeneratedLongFormOnDemand :
                Comscore.StreamingTag.ContentType.UserGeneratedShortFormOnDemand);
        });
    }

    return Comscore.StreamingTag.ContentType.Other;
}

/**
 * Assembles the metadata relevant to an advertisement beacon.
 *
 * @param {Object} adData Generic advertisement request data
 * @return {Object}
 */
function getAdvertisementMetadata(adData) {
    return {
        /* eslint-disable camelcase */
        ns_st_cl: (adData.time_break * 1000).toString(),
        /* eslint-enable */
    };
}

/**
 * Determines the Comscore AdType of a newly-playing advertisement.
 *
 * @param {Object} state Current state of the application
 * @param {Object} adData Generic advertisement request data
 * @return {Comscore.StreamingTag.AdType}
 */
function getAdvertisementType(state, adData) {
    if (state.stream instanceof LiveTwitchContentStream) {
        return Comscore.StreamingTag.AdType.LinearLive;
    } else if (state.stream instanceof VODTwitchContentStream) {
        switch (adData.roll_type) {
        case PREROLL:
            return Comscore.StreamingTag.AdType.LinearOnDemandPreRoll;
        case MIDROLL:
            return Comscore.StreamingTag.AdType.LinearOnDemandMidRoll;
        case POSTROLL:
            return Comscore.StreamingTag.AdType.LinearOnDemandPostRoll;
        }
    }

    return Comscore.StreamingTag.AdType.Other;
}

/**
 * Convert an object to a Comscore-safe metadata object by replacing non-string
 * values with their Comscore equivalents.
 *
 * @param {Object} metadata
 * @return {Object}
 */
function toComscoreValues(metadata) {
    return reduce(metadata, (m, val, key) => {
        // eslint-disable-next-line no-param-reassign
        m[key] = toComscoreValue(val);
        return m;
    }, {});
}

/**
 * Convert a value to the Comscore equivalent.
 *
 * @param {String|Number|null} v
 * @return {String}
 */
function toComscoreValue(v) {
    switch (v) {
    case true:
        return '1';
    case false:
        return '0';
    case null:
        return '*null';
    }

    return String(v);
}

/**
 * Convert a Date object to a Comscore-style date string.
 *
 * @param {Date} date
 * @return {String}
 */
function toComscoreDate(date) {
    const year = date.getUTCFullYear();
    const month = padStart(date.getUTCMonth() + 1, 2, '0');
    const day = padStart(date.getUTCDate(), 2, '0');

    return `${year}-${month}-${day}`;
}
