import { CONTENT_MODE_LIVE } from '../stream/twitch-live';
import * as murmur from 'murmurhash';
import * as TwitchEvents from '../backend/events/twitch-event';

// Segments contain timestamps indicating when they were handled by
// the video system's various components like the ingest server and
// the transcoder. When we change segments, we want to gather these
// timestamps and report them to Spade to measure the delays caused by
// video systems.
//
// The LatencyTracker also samples its data. It's important that the
// sampling scheme used here stays in sync with the code in
// https://git-aws.internal.justin.tv/video/logster, which is the
// thing that reports on playback latency induced by the replication
// system. LatencyTracker and logster need to agree on their sampling
// scheme so that segments of video are consistently tracked and can
// later be joined together during data analysis, so be careful when
// changing either the _sample or _segmentName methods.
export class LatencyTracker {
    /**
     * Create a new LatencyTracker and wire it up to listen for
     * events.
     *
     * @param {Analytics} analytics Analytics object which will
     * do the work of sending data with trackEvent.
     * @param {number} sampleRate A number in [0, 1] for how
     * frequently to actually send data. 0 means send nothing, 1 means
     * send everything, 0.5 means send data half the time, etc.
     * @param {Object} player A reference to the player which emits
     * video events.
     * @param {Object} store A reference to the state store so that
     * the LatencyTracker can determine the channel name and playback
     * quality.
     * @return {LatencyTracker}
     */
    constructor(analytics, sampleRate, player, store) {
        this._analytics = analytics;
        this._sample = murmurSegmentSampler(sampleRate);
        this._player = player;
        this._store = store;

        // Whenever a segment changes, we have a chance to track a
        // latency_report_playback event.
        player.addEventListener(TwitchEvents.SEGMENT_CHANGE, this._onSegmentChange.bind(this));
    }

    _segmentName(filename) {
        const { stream, quality } = this._store.getState();
        const channel = stream.contentType === CONTENT_MODE_LIVE ? stream.channel.toLowerCase() : '';
        const currentQuality = quality.current;
        return `${channel},${currentQuality},${filename}`;
    }

    _onSegmentChange(data) {
        // It is safe to trust that the store's stream and
        // quality.current should be set up correctly at this point
        // since we have video playing if the SEGMENT_CHANGE event has
        // fired.
        const key = this._segmentName(data.name);
        if (this._sample(key) && this._store.getState().stream.contentType === CONTENT_MODE_LIVE) {
            /* eslint-disable camelcase */
            this._analytics.trackEvent('latency_report_playback', {
                segment: key,
                broadcaster_send: data.broadcaster_send,
                ingest_receive: data.ingest_receive,
                ingest_send: data.ingest_send,
                transcode_receive: data.transcode_receive,
                transcode_send: data.transcode_send,
                playback_start: data.playback_start,
            });
            /* eslint-enable camelcase */
        }
    }
}

/**
 * Creates a sampler which returns true `rate` fraction of the time by
 * using murmurhash, seeded from a key, to generate a random integer.
 *
 * MurmurHash is very fast, widely available, and has great mixing
 * properties, which makes it a pretty good RNG (as opposed to MD5 or
 * something which doesn't have a strong enough avalanche effect for
 * our purposes).
 *
 * @param {number} rate A rate in [0, 1] to accept data
 * @returns {function} Sampler function
 */
export function murmurSegmentSampler(rate) {
    const maxInt32 = 0xffffffff;
    const seed = 0;  // Chosen arbitrarily.

    return function(key) {
        return murmur.v3(key, seed) / maxInt32 <= rate;
    };
}
