import includes from 'lodash/includes';
import * as Settings from '../settings';
import { b64EncodeUnicode } from './util';
import find from 'lodash/find';
import { stringify } from 'query-string';

const UNTRUSTED_MINUTE_WATCHED = 'x_untrusted_minute-watched_filtered';
const MINUTE_WATCHED = 'minute-watched';

/**
 * Spade Client
 * @class
 * @constructor
 *
 * @param {Object} opts
 *        `host`: Spade API host
 */
export function SpadeClient(opts) {
    this._trustedURI = opts.trustedURI;
    this._untrustedHost = opts.untrustedHost;
}

/**
 * Send an event to Spade.
 *
 * @param {String} eventName
 * @param {Object} properties
 */
SpadeClient.prototype.trackEvent = function(eventName, properties) {
    if (includes(Settings.spadeIgnore, eventName)) {
        return;
    }

    this._send({
        event: eventName,
        properties: properties,
    });

    if (eventName === MINUTE_WATCHED) {
        this._sendToUntrustedHost({
            event: UNTRUSTED_MINUTE_WATCHED,
            properties,
        });
    }
};

/**
 * Send a group of events to Spade.
 *
 * @param {Array<{event:String, properties:Object}>} events
 */
SpadeClient.prototype.trackEvents = function(events) {
    var filteredEvents = events.filter(function(event) {
        return !includes(Settings.spadeIgnore, event.event);
    });
    const minutesWatchedEvent = find(filteredEvents, ({ event }) => event === MINUTE_WATCHED);

    if (filteredEvents.length > 0) {
        this._send(filteredEvents);
    }

    if (minutesWatchedEvent) {
        this._sendToUntrustedHost({
            event: UNTRUSTED_MINUTE_WATCHED,
            properties: minutesWatchedEvent.properties,
        });
    }
};

/**
 * Send the tracking data to Spade.
 * @private
 *
 * @param {(Array|Object)} payload
 */
SpadeClient.prototype._send = function(payload) {
    const data = {
        data: b64EncodeUnicode(JSON.stringify(payload)),
    };

    fetch(
        this._trustedURI,
        {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
            },
            body: stringify(data),
        }
    );
};

/**
 * Sends tracking event to spade.twitch.tv (FOR MINUTES WATCHED ONLY)
 * @private
 *
 * @param {(Array|Object)} payload
 */
SpadeClient.prototype._sendToUntrustedHost = function(payload) {
    const data = {
        data: b64EncodeUnicode(JSON.stringify(payload)),
    };

    fetch(
        this._untrustedHost,
        {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
            },
            body: stringify(data),
        }
    );
};
