import extend from 'lodash/extend';
import includes from 'lodash/includes';
import { MixpanelClient } from './mixpanel';
import { SpadeClient } from './spade';
import { FirstPartyClient } from './first-party';
import { setTrackingClients } from '../actions/analytics';
import * as Settings from '../settings';
import { trustedSpadeURI } from '../generated/spade-uri';
import { parseUri } from '../util/parseuri';
import { sanitizeQuery } from '../util/url';
import * as DeviceId from '../util/device-id';
import * as UniqueId from '../util/unique-id';
import * as flash from 'util/flash';
import * as PlayerType from '../util/player-type';
import { clipsBaseProperties, clipsClipProperties } from 'ui/player-types/clips/utils/tracking/base-properties';
import assign from 'lodash/assign';
import bowser from 'bowser';

const CLIENT_APP_EMBER = 'ember';

const FIRST_PARTY_ANALYTICS_PLAYER_TYPES = [
    PlayerType.PLAYER_AMAZON_VSE,
    PlayerType.PLAYER_AMAZON_LIVE,
];

export function AnalyticsTracker(store, options) {
    const self = this;
    const promises = [];

    let globalProperties;

    function init() {
        const clientArray = [];

        clientArray.push(new MixpanelClient({
            host: Settings.mixpanelHost,
            token: Settings.mixpanelToken,
        }));

        clientArray.push(new SpadeClient({
            trustedURI: trustedSpadeURI,
            untrustedHost: Settings.untrustedSpadeHost,
        }));

        const { playerType } = store.getState().env;
        if (includes(FIRST_PARTY_ANALYTICS_PLAYER_TYPES, playerType)) {
            clientArray.push(new FirstPartyClient(store));
        }

        store.dispatch(setTrackingClients(clientArray));

        // Set global properties that will be added to all events
        globalProperties = generateGlobalProperties();

        // Set the passed in tracking properties. (ex. player: frontpage)
        self.setProperties(options.tracking);

        // Set channel/video to what we are attempting to initially load
        self.setProperties({
            channel: options.channel,
            vod: options.video,
            // eslint-disable-next-line camelcase
            vod_id: options.video,
        });

        // Ensure the benchmark ID is set by options or manual generation
        self.setProperty('benchmark_session_id', options.benchmarkId || UniqueId.generate());

        // Track whether or not this browser supports MSE playback
        self.setProperty('mse_support', (typeof MediaSource !== 'undefined'));
    }

    // Returns the last parts of the host name.
    // TODO Move this to a util file.
    // ex. player.twitch.tv -> twitch.tv
    function topLevelHost(host) {
        const parts = host.split('.');
        return parts.slice(-2).join('.');
    }

    function getLocation() {
        let loc;

        // For backwards compatibility, we need the parent's url.
        // This only works for twitch because of cross-origin constaints.
        try {
            document.domain = Settings.domain;
            loc = window.parent.location;

            // Try to access one of the properties. This will throw an
            // exception on Firefox/Chrome or return undefined on Safari.
            if (!loc.host) {
                // Failover to the current window.
                loc = window.location;
            }
        } catch (e) {
            // Failover to the current window.
            loc = window.location;
        }

        // Return the location.
        return loc;
    }

    function getReferrer() {
        // For backwards compatibility, we need the parent's url.
        // This only works for twitch because of cross-origin constaints.
        try {
            document.domain = Settings.domain;
            return window.parent.document.referrer;
        } catch (e) {
            return window.document.referrer;
        }
    }

    // These tracking properties are added to every event.
    function generateGlobalProperties() {
        const fpv = flash.getFlashPlayerVersion();
        const flashVersion = `${fpv.major},${fpv.minor},${fpv.release}`;

        const embedReferrer = getReferrer();
        const embedReferrerURI = parseUri(embedReferrer);

        // false returns device_id, true returns session_device_id
        const deviceId = DeviceId.get(false);
        const sessionDeviceId = DeviceId.get(true);

        const clientApp = PlayerType.getPlayerType() === PlayerType.PLAYER_EMBED ? null : CLIENT_APP_EMBER;

        /* eslint-disable camelcase */
        return {
            // eslint-disable-next-line no-undef
            app_version: APP_VERSION,
            flash_version: flashVersion,
            referrer_url: embedReferrer,
            referrer_host: embedReferrerURI.host,
            referrer_domain: topLevelHost(embedReferrerURI.host),
            browser: store.getState().window.navigator.appVersion || '',
            browser_family: bowser.name,
            browser_version: bowser.version,
            os_name: bowser.osname,
            os_version: bowser.osversion,
            user_agent: store.getState().window.navigator.userAgent || '',
            device_id: deviceId,
            distinct_id: deviceId,
            session_device_id: sessionDeviceId,
            client_app: clientApp,
        };
        /* eslint-enable camelcase */
    }

    /**
     * Send multiple tracking events to the data tracking services.
     *
     * @param {Array<{event:String,properties:Object}>} events
     */
    self.trackEvents = function(events) {
        // Grab the time before waiting for our properties promises.
        const state = store.getState();
        const { window: windowObj } = state;
        const time = windowObj.Date.now() / 1000;

        const referrerUriHost = state.tracking.referrer ? parseUri(state.tracking.referrer).host : '';
        // Block tracking until all of the current promises have returned.
        Promise.all(promises).then(asyncData => {
            // Combine all results from the promises into a single object.
            const asyncProperties = extend.apply(null, asyncData);
            const embedLocation = getLocation();
            const { platform, playerType } = state.env;

            // Loop over all of the events and add the global properties.
            // Properties passed into trackEvent will take precedence in case of collision.
            const eventsWithProps = events.map(e => {
                const properties = extend({}, globalProperties, asyncProperties, e.properties, state.tracking, {
                    // science will still need to determine between desktop app/browser via user agent
                    platform: (playerType === PlayerType.PLAYER_CURSE ? playerType : platform),
                    // eslint-disable-next-line camelcase
                    play_session_id: state.analytics.playSessionId,
                    url: sanitizeQuery(embedLocation.href),
                    host: embedLocation.host,
                    domain: topLevelHost(embedLocation.host),
                    // eslint-disable-next-line camelcase
                    referrer_host: referrerUriHost,
                    // eslint-disable-next-line camelcase
                    referrer_domain: referrerUriHost ? topLevelHost(referrerUriHost) : '',
                    // eslint-disable-next-line camelcase
                    low_latency: state.manifestInfo.future,
                });
                if (!properties.time) {
                    properties.time = time;
                }

                return {
                    event: e.event,
                    properties,
                };
            });

            if (options.debug) {
                eventsWithProps.forEach(({ event, properties }) => {
                    // eslint-disable-next-line no-console
                    console.log('track event:', event, properties);
                });
            }
            const trackingClients = state.analytics.trackingClients;
            trackingClients.forEach(trackingClient => {
                trackingClient.trackEvents(eventsWithProps);
            });
        });
    };

    /**
     * Send a single tracking event to the data tracking services.
     *
     * @param {String} name
     * @param {Object} properties
     */
    self.trackEvent = function(name, properties) {
        self.trackEvents([{
            event: name,
            properties: properties,
        }]);
    };

    self.clipsTrackEvent = function(name, properties) {
        const state = store.getState();

        self.trackEvent(
            name,
            assign(
                {},
                clipsBaseProperties(state),
                clipsClipProperties(state),
                properties
            )
        );
    };

    self.setProperty = function(name, value) {
        const properties = {};
        properties[name] = value;

        self.setProperties(properties);
    };

    // Sets tracking properties part of every event.
    // If the argument is a promise, we block all tracking events from
    // being sent until the promise is resolved. This is useful when
    // tracking properties require an asychronous call which would otherwise
    // block playback.
    self.setProperties = function(arg) {
        // Convert the arg into a promise.
        // If it's an object, then we create an already resolved promise.
        // If it's a promise, then we pass through the promise.
        const promise = Promise.resolve(arg).then(null, function(err) {
            // We treat rejections as warnings instead of fatal.
            // eslint-disable-next-line no-console
            console.warn('failed to resolve properties promise', err);
            return {};
        });

        promises.push(promise);
    };

    init();
}
