import * as Settings from '../settings';
import { stringify } from 'query-string';

/**
 * Valve Client
 * Contacts the Steam Watching end-point to alert Valve when a
 * Steam user watches a Steam channel.
 *
 * @class
 * @constructor
 */
export function ValveClient() {
    this._viewerSteamId = Promise.reject();
    this._channelSteamId = Promise.reject();

    // suppress uncaught (in Promise) errors
    this._viewerSteamId.then(null, () => null);
    this._channelSteamId.then(null, () => null);
}

/**
 * Sets the viewer info allocation promise.
 *
 * @param {Promise} data
 */
ValveClient.prototype.viewerInfo = function(data) {
    this._viewerSteamId = data.then(function(vi) {
        if (vi.steam_id) {
            return vi.steam_id;
        }
        return Promise.reject();
    });

    // suppress uncaught (in Promise) errors
    this._viewerSteamId.then(null, () => null);
};

/**
 * Sets the channel info allocation promise.
 *
 * @param {Promise} data
 */
ValveClient.prototype.channelInfo = function(data) {
    this._channelSteamId = data.then(function(ci) {
        if (ci.steam_id) {
            return ci.steam_id;
        }
        return Promise.reject();
    });

    // suppress uncaught (in Promise) errors
    this._channelSteamId.then(null, () => null);
};

/**
 * Contacts the Steam Watching end-point to alert Valve when a
 * Steam user watches a Steam channel.
 */
ValveClient.prototype.notify = function() {
    // Ensure required data has been allocated before attempting to contact the API
    const notified = Promise.all([this._channelSteamId, this._viewerSteamId]).then(function(ids) {
        const channelSteamId = ids[0];
        const userSteamId = ids[1];

        const data = {
            channel: channelSteamId,
            viewer: userSteamId,
        };

        fetch(`${Settings.valveApiHost}?${stringify(data)}`);
    });

    // suppress uncaught (in Promise) errors
    notified.then(null, () => null);
};
