import assign from 'lodash/assign';
import memoize from 'lodash/memoize';
import merge from 'lodash/merge';
import pick from 'lodash/pick';
import reduce from 'lodash/reduce';
import find from 'lodash/find';
import { stringify } from 'query-string';
import * as Settings from './settings';
import * as PlayerType from './util/player-type';
import { ExtensionCoordinator } from 'extension-coordinator';
import {
    parseExtensionToken,
    EXTENSION_PERMISSION_STATE_GRANTED,
} from './util/extensions';

const API_KRAKEN_VERSION_3 = 'application/vnd.twitchtv.v3+json';
const API_KRAKEN_VERSION_5 = 'application/vnd.twitchtv.v5+json';

const API_REQUEST_HEADERS = {
    'Client-ID': 'jzkbprff40iqj646a697cyrvl0zt2m6',
};

// Only allow alphanumeric characters in the path for security.
function cleanPath(path) {
    return String(path).replace(/[^A-Za-z0-9_]/g, '');
}

/* This check is needed to duplicate jQuery.ajax behaviour, which
   rejects on an unsuccessful response, whereas `fetch` only rejects
   if the request itself failed for some reason.

   See https://github.com/github/fetch#handling-http-error-statuses for more details.
*/
function _checkStatus(response) {
    if (response.status >= 200 && response.status < 300) {
        return Promise.resolve(response);
    }

    return Promise.reject(response);
}

function _createHeadersObject({ oauthToken, opts }) {
    const headers = merge({}, API_REQUEST_HEADERS, opts.headers);
    if (oauthToken) {
        headers.Authorization = `OAuth ${oauthToken}`;
    }

    if (opts.method !== 'GET' && opts.method !== 'HEAD') {
        headers['Content-Type'] = 'application/x-www-form-urlencoded; charset=UTF-8';
    }
    return headers;
}

/**
 * Makes a request to the Twitch API, enhancing the incoming request configuration
 * with necessary headers and authorization to perform the request.
 *
 * @param {Object} opts
 *        Options for the request, which will be augmented with additional
 *        headers as needed to make the request.
 * @return {Promise<Object>}
 */
function apiRequest(opts) {
    return oauthToken().
        then(resp => resp.token, _ => null).
        then(oauthToken => {
            const headers = _createHeadersObject({
                oauthToken,
                opts,
            });

            return fetch(opts.url, {
                method: opts.method || 'GET',
                headers,
                credentials: opts.xhrFields && opts.xhrFields.withCredentials ? 'include' : 'omit',
                body: opts.data ? stringify(opts.data) : undefined,
            });
        }).
        then(_checkStatus).
        then(response => {
            return response.json().catch(() => response); // If the JSON parse fails, return the response object
        });
}

/**
 * Make a request to the v3 kraken specified URL (starting from api.twitch.tv/kraken/)
 * and return the result
 *
 * @param {String} urlFragment the part of the url after /kraken/ to which to send the request
 * @return {Promise<Object>} a promise that will resolve to the api response
 */
export function krakenRequest(urlFragment) {
    return apiRequest({
        url: `${Settings.apiHost}/kraken/${urlFragment}`,
        headers: {
            Accept: API_KRAKEN_VERSION_3,
        },
    });
}

/**
 * Make a request to the v5 kraken specified URL (starting from api.twitch.tv/kraken/)
 * and return the result
 *
 * @param {String} urlFragment the part of the url after /kraken/ to which to send the request
 * @param {Object} opts the options that can be passed into the reqest (i.e. method/headers)
 * @return {Promise<Object>} a promise that will resolve to the api response
 */
export function krakenRequestv5(urlFragment, opts) {
    let headers = {
        Accept: API_KRAKEN_VERSION_5,
    };

    if (opts && opts.headers) {
        headers = merge(headers, opts.headers);
    }

    return apiRequest(merge({}, opts, {
        url: `${Settings.apiHost}/kraken/${urlFragment}`,
        headers,
    }));
}

// TODO These results should expire after a certain amount of time.

export const channelInfo = memoize(function(channel) {
    if (!channel) {
        return Promise.reject(new Error('No channel info available on null channel ID'));
    }
    return krakenRequest(`channels/${cleanPath(channel)}`);
});

export const videoInfo = memoize(function(video) {
    if (video === null || video === undefined) {
        return Promise.reject(new Error('No video info available on null video ID'));
    }

    const request = krakenRequest(`videos/${cleanPath(video)}`);

    // We need the API request just for muted_segments
    // TODO Add it to kraken instead and stop making this request.
    const videoRequest = apiRequest({
        url: `${Settings.apiHost}/api/videos/${cleanPath(video)}`,
    }).then(function(data) {
        return pick(data, [
            'muted_segments',
            'increment_view_count_url',
            'restrictions',
            'seek_previews_url',
        ]);
    });

    return Promise.all([request, videoRequest]).then(function(requests) {
        const channel = requests[0].channel.name;

        return channelInfo(channel).then(function(channelData) {
            const videoData = assign({}, requests[0], requests[1]);
            videoData.channel = channelData;

            return videoData;
        });
    });
});

export function streamInfo(channel) {
    return krakenRequest(`streams/${cleanPath(channel)}`);
}

export function getSubscriptionInfo(userId, channelId) {
    return krakenRequestv5(`users/${userId}/subscriptions/${channelId}`);
}

export function getSubscriptionProducts(channelName) {
    return apiRequest({
        url: `${Settings.apiHost}/channels/${cleanPath(channelName)}/product`,
    });
}

export const getChannelAdProperties = memoize(function(channelName) {
    return channelInfo(channelName).then(channelInfo => {
        const channelId = channelInfo._id;
        return krakenRequestv5(`channels/${channelId}/ads/properties`).then(channelAdProperties => {
            return Promise.resolve(channelAdProperties);
        }).catch(error => {
            return Promise.reject(new Error(`Bad ad properties for ${channelId}: ${error}`));
        });
    });
});

export const channelViewerInfo = memoize(function(channel) {
    return apiRequest({
        url: `${Settings.apiHost}/api/channels/${cleanPath(channel)}/viewer`,
    });
});

export const userInfo = memoize(function() {
    return apiRequest({
        url: `${Settings.apiHost}/api/viewer/info.json`,
        xhrFields: {
            withCredentials: true,
        },
    });
});

/**
 * Generates follow URL with userId and channelId
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @return {String}
 */
function createFollowUrl(userId, channelId) {
    return `users/${cleanPath(userId.toString())}/follows/channels/${cleanPath(channelId.toString())}`;
}

/**
 * Returns following relationship for a channel - GET request
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @return {Promise}
 */
export function getFollowChannel(userId, channelId) {
    const url = createFollowUrl(userId, channelId);
    return krakenRequestv5(url);
}

/**
 * adds/removes following relationship for a channel - PUT/DELETE request
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @param {Boolean} shouldFollow
 * @return {Promise}
 */
export function setFollowChannel(userId, channelId, shouldFollow) {
    const url = createFollowUrl(userId, channelId);
    return krakenRequestv5(url, {
        method: shouldFollow ? 'PUT' : 'DELETE',
    });
}

/**
 * Adds/removes a notifications relationship to the channel - PUT request
 *
 * @param {Number} userId
 * @param {Number} channelId
 * @param {Boolean} notifications
 * @return {Promise}
 */
export function setFollowNotifications(userId, channelId, notifications) {
    const url = createFollowUrl(userId, channelId);
    return krakenRequestv5(url, {
        data: { notifications },
        method: 'PUT',
    });
}

// https://github.com/justintv/Twitch-API/blob/master/v3_resources/users.md#get-user
export const krakenUserInfo = memoize(function() {
    return krakenRequest('user');
});

// Returns the channel info from API.
export const channelAPIInfo = memoize(function(channel) {
    return apiRequest({
        url: `${Settings.apiHost}/api/channels/${channel}`,
    });
});

/**
 * Returns (multiple) communities relationship for a channel - GET request
 * If channel / Broadcaster is a part of any communities, the communities
 * object is returned, otherwise undefined
 *
 * @param {string} channelName
 * @return {Promise <Object>}
 *
 */
export function getCommunitiesFromChannel(channelName) {
    return channelInfo(channelName).then(channelInfo => {
        const channelId = channelInfo._id;
        return krakenRequestv5(`channels/${channelId}/communities`).then(communities => {
            return Promise.resolve(communities);
        }).catch(error => {
            return Promise.reject(`Bad communities for ${channelId}: ${error}`);
        });
    });
}

/**
 * Retrieves the featured collection id from a specified channel
 *
 * @param {Number} channelId
 * @return {Promise}
 */
export function getFeaturedCollection(channelId) {
    return apiRequest({
        url: `${Settings.apiHost}/v5/channels/${channelId}/collections?limit=1&exclude_empty=true`,
    });
}

/**
 * Retrieves the collection info + items from API.
 * It's split into two API calls, so return a promise that combines both.
 *
 * @param {Object} id - ID of the collection to load
 */
export function collectionInfo(id) {
    const metadataApiResult = apiRequest({
        url: `${Settings.apiHost}/v5/collections/${id}`,
    });

    const itemsApiResult = apiRequest({
        url: `${Settings.apiHost}/v5/collections/${id}/items`,
    });

    return Promise.all([metadataApiResult, itemsApiResult]);
}

// Return an oauth token for the current viewer.
export const oauthToken = memoize(function() {
    return fetch(
        `${Settings.apiHost}/api/viewer/token.json`,
        {
            credentials: 'include',
            headers: API_REQUEST_HEADERS,
        }
    ).
        then(_checkStatus).
        then(response => {
            return response.json().catch(() => response); // If the JSON parse fails, return the response object
        });
});

/**
 * Sets the OAuth token.
 *
 * @param {Object} options - the player options
 */
export function setOAuthToken(options) {
    /* eslint-disable camelcase */
    const { player, oauth_token } = options;
    if (
        (
            player === PlayerType.PLAYER_CURSE ||
            player === PlayerType.PLAYER_SITE ||
            player === PlayerType.PLAYER_FRONTPAGE ||
            player === PlayerType.PLAYER_FEED
        )
        && oauth_token
    ) {
        const response = Promise.resolve({ token: oauth_token });
        // https://lodash.com/docs/4.16.4#memoize
        oauthToken.cache.set(undefined, response);
    }
    /* eslint-enable camelcase */
}

// Returns a Boolean value representing if VOD content
// is restricted (ChanSubs) for this viewer.
export function isVODRestricted(viewerInfo, videoInfo) {
    // If this viewer is a Channel Subscriber there is
    // no VOD playback restriction.
    if (viewerInfo.chansub !== null || viewerInfo.is_admin) {
        return false;
    }

    return reduce(videoInfo.restrictions, function(required, restriction) {
        return (required || (restriction === 'chansub'));
    }, false);
}

// Returns the URL to the channel page on Twitch.
export function channelUrl(channel, params) {
    var url = `${Settings.twitchHost}/${cleanPath(channel)}`;
    if (params) {
        url += `?${stringify(params)}`;
    }

    return url;
}

// Returns the URL to the video page on Twitch.
export function videoUrl(channel, video, params) {
    // Start with the channel URL, no params.
    var url = channelUrl(channel);

    // Split the video into parts.
    var videoType = video[0];
    var videoId = video.substring(1);

    // Add the remainder of the url.
    url += `/${cleanPath(videoType)}/${cleanPath(videoId)}`;

    if (params) {
        url += `?${stringify(params)}`;
    }

    return url;
}

/**
 * Retrieves all video_overlay extensions for a given channel.
 *
 * @param {String} channel name, as specified in the URL.
 * @param {Array<Object>} A collection of objects describing the video overlays.
 */
export function overlayExtensionsForChannel(channel) {
    return channelInfo(channel).
        then(resp => {
            return ExtensionCoordinator.ExtensionService.getInstalledExtensions(resp._id);
        }).
        then(resp => {
            return resp.installed_extensions.reduce((extensions, extDoc) => {
                const { extension, installation_status: install } = extDoc;
                const isActive = install.activation_state !== 'active';
                const videoOverlayAnchor = install.activation_config.anchor !== 'video_overlay';
                const hiddenAnchor = install.activation_config.anchor !== 'hidden';
                if (isActive || (videoOverlayAnchor && hiddenAnchor)) {
                    return extensions;
                }

                const tokenResult = find(resp.tokens, token => extension.id === token.extension_id);
                if (!tokenResult) {
                    return extensions;
                }

                const token = parseExtensionToken(tokenResult.token);
                return extensions.concat({
                    token,
                    id: extension.id,
                    name: extension.name,
                    sku: extension.sku,
                    summary: extension.summary,
                    anchor: extension.anchor,
                    vendorCode: extension.vendor_code,
                    version: extension.version,
                    state: extension.state,
                    viewerUrl: extension.viewer_url,
                    lastUserIdentityLinkState: token.permissionsState === EXTENSION_PERMISSION_STATE_GRANTED,
                    supportsIdentityLinking: extension.request_identity_link,
                });
            }, []);
        }).catch(() => []);
}

/**
 * Posts an extension report to moderation
 * @param {Object} data
 */
export function postExtensionReport(data) {
    return fetch('https://www.twitch.tv/user/report', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
        },
        body: stringify(data),
    });
}

export function createClip(url, data) {
    return oauthToken().
        then(resp => resp.token, _ => null).
        then(oauthToken => {
            const opts = {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: stringify(data),
            };

            if (oauthToken) {
                opts.headers.Authorization = `OAuth ${oauthToken}`;
            }
            return fetch(url, opts);
        }).then(res => res.json());
}
