import { subscribe } from 'util/subscribe';
import { localStore } from 'util/storage';
import { addNotification } from './actions/notifications';
import { AutoSuggestNotification } from './ui/components/notifications/auto-suggest-notification-component';

export const TEN_MINUTES = 600000; // Ten minutes
export const MAX_BUFFER_EMPTY_COUNT = 3;
export const WEEK = 604000000; // Seven days
export const AUTO_SUGGEST_NOTIFICATION_TIMESTAMP = 'AutoSuggestNotificationTimestamp';
export const AUTO_SUGGEST_NOTIFICATION_CLOSED_COUNT = 'AutoSuggestNotificationClosedCount';

export class AutoSuggestNotificationManager {
    constructor(store) {
        this._store = store;
        const { window: windowObj } = this._store.getState();

        this._unsubs = [];
        this._unsubs.push(subscribe(this._store, ['playback.bufferEmpties'], this._trackBufferEmpty.bind(this)));

        this._currentBufferEmptyCount = 0;
        this._previousBufferTimestamp = (new windowObj.Date()).getTime();

        if (!localStore.get(AUTO_SUGGEST_NOTIFICATION_TIMESTAMP)) {
            localStore.set(AUTO_SUGGEST_NOTIFICATION_TIMESTAMP, (new windowObj.Date().getTime()) - WEEK);
        }
    }

    _trackBufferEmpty({ playback }) {
        if (playback.bufferEmpties > 0) {
            this._currentBufferEmptyCount++;

            if (this._shouldShowAutoSuggestNotification()) {
                this._showAutoSuggestNotification();
            }
        }
    }

    _shouldShowAutoSuggestNotification() {
        if (this._store.getState().quality.selected !== 'auto') {
            return this._thirdBufferEmptyInTenMinutes() && this._hasNotSeenAutoSuggestNotificationThisWeek();
        }
        return false;
    }

    _showAutoSuggestNotification() {
        if (!localStore.get(AUTO_SUGGEST_NOTIFICATION_CLOSED_COUNT) ||
                localStore.get(AUTO_SUGGEST_NOTIFICATION_CLOSED_COUNT) <= 1) {
            this._store.dispatch(addNotification(AutoSuggestNotification));
            localStore.set(AUTO_SUGGEST_NOTIFICATION_TIMESTAMP, (new Date()).getTime());
        }
    }

    // Check to see if 3+ buffer-empty events are fired within 10 minutes.
    _thirdBufferEmptyInTenMinutes() {
        const { window: windowObj } = this._store.getState();
        const now = (new windowObj.Date()).getTime();
        const timeElapsed = now - this._previousBufferTimestamp;

        if (timeElapsed > TEN_MINUTES) {
            this._previousBufferTimestamp = now;
            this._currentBufferEmptyCount = 0;

            return false;
        }

        if (this._currentBufferEmptyCount >= MAX_BUFFER_EMPTY_COUNT) {
            this._previousBufferTimestamp = now;
            this._currentBufferEmptyCount = 0;
            return true;
        }
    }

    // Check to see if a notification has already been sent within the past 7 days.
    _hasNotSeenAutoSuggestNotificationThisWeek() {
        const { window: windowObj } = this._store.getState();
        const now = new windowObj.Date().getTime();
        const lastSeen = localStore.get(AUTO_SUGGEST_NOTIFICATION_TIMESTAMP);

        if ((now - lastSeen) > WEEK) {
            return true;
        }
        return false;
    }

    destroy() {
        this._unsubs.forEach(unsub => unsub());
    }
}
