/* global chrome */
import EventEmitter from 'event-emitter';
import * as Settings from '../settings';
import { channelInfo, videoInfo } from '../api';
import * as MediaEvents from './events/media-event';
import * as NetworkState from './state/network-state';
import * as ReadyState from './state/ready-state';
import { CASTING_CHANGE, QUALITY_CHANGE } from './events/twitch-event';
import { UNAVAILABLE, AVAILABLE, CONNECTED,
         CONNECTING } from '../state/chromecast';
import Errors from 'errors';
import { setError } from 'actions/error';

// This entire backend is static because of how Chromecast works.
export var BackendChromecast = {};
export const CAST_SENDER_URL = 'https://www.gstatic.com/cv/js/sender/v1/cast_sender.js';
const CAST_SENDER_CHECK_ATTEMPTS = 5;
const AUTO_QUALITY = Object.freeze({
    group: 'auto',
    name: 'Auto',
});

(function() {
    const self = BackendChromecast;

    const events = new EventEmitter();
    const cache = {};

    let session;
    let media;
    let channel;
    let video;
    let seekRequest;
    let available;
    let error;
    let connecting;
    let networkState;
    let readyState;
    let initialized = false;
    let streamUrl = null;
    let allQualities = [];
    let store;

    function initExtension() {
        return new Promise(function(resolve, reject) {
            const $scriptEl = document.createElement('script');
            document.body.appendChild($scriptEl);
            $scriptEl.onerror = reject;
            $scriptEl.onload = resolve;
            $scriptEl.async = true;
            $scriptEl.src = CAST_SENDER_URL;
        }).then(() => {
            if (!window.chrome) {
                // Only supports Chrome.
                return false;
            } else if (chrome.cast && chrome.cast.isAvailable) {
                // Chromecast was already initialized successfully, fire the callback.
                return true;
            }

            return new Promise(function(resolve) {
                // Register the callback called after cast_sender.js is loaded.
                window.__onGCastApiAvailable = function(loaded) {
                    resolve(loaded);
                };

                // A fallback if this file is included after cast_sender.js
                // We have to wait for the extension to load, which takes time.
                // We check if the extension is loaded every second for 5 seconds.
                let attempts = CAST_SENDER_CHECK_ATTEMPTS;
                const retryLoop = setInterval(function() {
                    var loaded = !!(chrome.cast && chrome.cast.isAvailable);
                    if (loaded || attempts <= 0) {
                        resolve(loaded);
                        clearInterval(retryLoop);
                    }

                    attempts--;
                }, 1000);
            });
        });
    }

    function initApi(loaded) {
        if (!loaded) {
            return;
        }

        var sessionRequest = new chrome.cast.SessionRequest(Settings.chromecastId);
        var apiConfig = new chrome.cast.ApiConfig(
            sessionRequest,
            onSession,
            onReceiver,
            chrome.cast.AutoJoinPolicy.ORIGIN_SCOPED
        );

        chrome.cast.initialize(apiConfig);
    }

    function initDefaults() {
        available = false;
        updateState();
    }

    // Called when the extension creates a session.
    // This will happen when the user clicks on the cast browser icon
    // or when the user navigates to the page while already casting
    function onSession(s) {
        session = s;
        session.addUpdateListener(onSessionUpdate);

        // Automatically start casting if we have a channel.
        if (channel && streamUrl) {
            loadChannel();
        } else if (video && streamUrl) {
            loadVideo();
        }

        session.addMessageListener('urn:x-cast:com.twitch.custom', function(event, data) {
            var payload = JSON.parse(data);
            if (payload.qualities) {
                allQualities = payload.qualities;
                events.emit(MediaEvents.CAN_PLAY);
                events.emit(MediaEvents.LOADED_METADATA);
            } // look for other messages
        });
    }

    // Called when a session changes state, usually when killed.
    function onSessionUpdate() {
        switch (session.status) {
        case chrome.cast.SessionStatus.CONNECTED:
            var receiver = session.receiver;
            if (cache.volume !== receiver.volume.level) {
                cache.volume = receiver.volume.level;
                events.emit(MediaEvents.VOLUME_CHANGE);
            }

            if (cache.muted !== receiver.volume.muted) {
                cache.muted = receiver.volume.muted;
                events.emit(MediaEvents.VOLUME_CHANGE);
            }
            break;
        case chrome.cast.SessionStatus.STOPPED:
            onMediaChange(null);

            session.removeUpdateListener(onSessionUpdate);
            session = null;

            connecting = false;
            updateState();

            break;
        }
    }

    // Called when a castable device is connected.
    // We don't know any specifics about the device though.
    function onReceiver(e) {
        available = (e === chrome.cast.ReceiverAvailability.AVAILABLE);
        updateState();
    }

    // Called when the media stream changes.
    // TODO: Figure out what happens when our tab is no longer cast.
    function onMediaChange(newmedia) {
        if (media) {
            media.removeUpdateListener(onMediaUpdate);
        }

        media = newmedia;
        connecting = false;

        if (media) {
            media.addUpdateListener(onMediaUpdate);
            if (seekRequest) {
                media.seek(seekRequest);
                seekRequest = null;
            }
        }
        updateState();
    }

    // Called when the media changes for whatever reason.
    function onMediaUpdate() {
        if (cache.currentTime !== media.currentTime) {
            cache.currentTime = media.currentTime;
            events.emit(MediaEvents.TIME_UPDATE);
        }

        if (cache.volume !== media.volume.level) {
            cache.volume = media.volume.level;
            events.emit(MediaEvents.VOLUME_CHANGE);
        }

        if (cache.muted !== media.volume.muted) {
            cache.muted = media.volume.muted;
            events.emit(MediaEvents.VOLUME_CHANGE);
        }

        if (cache.playbackRate !== media.playbackRate) {
            cache.playbackRate = media.playbackRate;
            events.emit(MediaEvents.RATE_CHANGE);
        }

        var state = media.playerState;
        if (cache.playerState !== media.playerState) {
            if (state === chrome.cast.media.PlayerState.PLAYING) {
                events.emit(MediaEvents.PLAYING);
            } else if (state === chrome.cast.media.PlayerState.PAUSED) {
                events.emit(MediaEvents.PAUSE);
            } else if (state === chrome.cast.media.PlayerState.BUFFERING) {
                // TODO?
            } else if (state === chrome.cast.media.PlayerState.IDLE) {
                events.emit(MediaEvents.ENDED);
            }

            cache.playerState = media.playerState;
        }
    }

    // Starts casting the given player.
    function loadChannel() {
        connecting = true;
        updateState();

        // Request the channel url and metadata.
        var infoRequest = channelInfo(channel);

        infoRequest.then(function(info) {
            var mediaInfo = new chrome.cast.media.MediaInfo(streamUrl, 'application/x-mpegurl');
            mediaInfo.streamType = chrome.cast.media.StreamType.LIVE;

            var metadata = new chrome.cast.media.GenericMediaMetadata();
            metadata.subtitle = info.game;
            metadata.title = info.display_name;
            metadata.images = [new chrome.cast.Image(info.logo)];

            mediaInfo.metadata = metadata;

            // TODO TODO: Analytics?
            /* eslint-disable camelcase */
            mediaInfo.customData = {
                channel: channel,
                analytics: {
                    chromecast_sender: 'player-ui-web',
                },
            };
            /* eslint-enable camelcase */
            var request = new chrome.cast.media.LoadRequest(mediaInfo);
            request.autoplay = true;

            session.loadMedia(request, onMediaChange);
        }, function(e) {
            // Clean up the session on error.
            session.stop();
            handleError(e);
        });
    }

    function loadVideo() {
        connecting = true;
        updateState();

        // Request the channel url and metadata.
        var infoRequest = videoInfo(video);

        infoRequest.then(function(info) {
            var mediaInfo = new chrome.cast.media.MediaInfo(streamUrl, 'application/x-mpegurl');
            mediaInfo.streamType = chrome.cast.media.StreamType.OTHER;

            var metadata = new chrome.cast.media.GenericMediaMetadata();
            metadata.subtitle = info.game;
            metadata.title = info.title;
            metadata.images = [new chrome.cast.Image(info.preview)];

            mediaInfo.metadata = metadata;

            // TODO TODO: Analytics?
            /* eslint-disable camelcase */
            mediaInfo.customData = {
                vod_id: video,
                analytics: {
                    chromecast_sender: 'player-ui-web',
                },
            };
            /* eslint-enable camelcase */
            var request = new chrome.cast.media.LoadRequest(mediaInfo);
            request.autoplay = true;

            session.loadMedia(request, onMediaChange);
        }, function(e) {
            // Clean up the session on error.
            session.stop();
            handleError(e);
        });
    }

    function handleError() {
        if (store) {
            store.dispatch(setError(Errors.CODES.ABORTED));
        }
    }

    // We use networkState and readyState to signal if we are connected.
    // It's a little hacky but compliant with the HTML5 video spec.
    // eslint-disable-next-line max-statements
    function updateState() {
        var oldNetworkState = networkState;
        var oldReadyState = readyState;

        if (media) {
            // "connected"
            networkState = NetworkState.NETWORK_LOADING;
            readyState = ReadyState.HAVE_METADATA;
        } else if (connecting) {
            // "connecting"
            networkState = NetworkState.NETWORK_LOADING;
            readyState = ReadyState.HAVE_NOTHING;
        } else if (available) {
            // "available"
            networkState = NetworkState.NETWORK_IDLE;
            readyState = ReadyState.HAVE_NOTHING;
        } else {
            // "unavailable"
            networkState = NetworkState.NETWORK_EMPTY;
            readyState = ReadyState.HAVE_NOTHING;
        }

        // Fire any events needed because of the state change.
        if (networkState !== oldNetworkState) {
            if (networkState === NetworkState.NETWORK_LOADING) {
                events.emit(MediaEvents.LOADSTART);
            }
        }

        if (readyState !== oldReadyState) {
            if (readyState === ReadyState.HAVE_METADATA) {
                events.emit(MediaEvents.LOADED_METADATA);
            } else if (readyState === ReadyState.HAVE_NOTHING) {
                events.emit(MediaEvents.ENDED);
            }
        }

        events.emit(CASTING_CHANGE, getCastingState());
    }

    /**
     * A method that determines the casting state of chromecast
     *
     * @return {String}
     */
    function getCastingState() {
        if (readyState === ReadyState.HAVE_NOTHING) {
            if (networkState === NetworkState.NETWORK_EMPTY) {
                return UNAVAILABLE;
            } else if (networkState === NetworkState.NETWORK_IDLE) {
                return AVAILABLE;
            }
            return CONNECTING;
        }
        return CONNECTED;
    }

    function onLaunchError(e) {
        handleError(e);
    }

    self.init = function(stateStore) {
        // Remove listeners in case old player is gone.
        events.removeAllListeners();

        // Return existing state to new player instance
        if (initialized) {
            return getCastingState();
        }
        initialized = true;

        store = stateStore;
        initDefaults();

        initExtension().then(initApi);
        return getCastingState();
    };

    self.destroy = function() {
        // empty
    };

    self.addEventListener = function(name, callback) {
        events.on(name, callback);
    };

    self.removeEventListener = function(name, callback) {
        events.off(name, callback);
    };

    self.getNetworkProfile = function() {
        // Chromecast doesn't appear to expose any of the internal data required
        // to generate a network profile.
        return [];
    };

    self.getError = function() {
        return error;
    };

    self.getSrc = function() {
        // unimplemented
    };

    self.setSrc = function() {
        // unimplemented
    };

    self.getNetworkState = function() {
        return networkState;
    };

    self.load = function() {
        // Start a new session and try to autoplay.
        chrome.cast.requestSession(onSession, onLaunchError);
    };

    self.getBuffered = function() {
        // TODO Implement properly
        return {
            length: 0,
        };
    };

    self.getReadyState = function() {
        return readyState;
    };

    self.getSeeking = function() {
        // unimplemented
    };

    self.getCurrentTime = function() {
        if (media) {
            return media.currentTime;
        }

        return cache.currentTime || 0;
    };

    self.setCurrentTime = function(time) {
        if (!(window.chrome && window.chrome.cast)) {
            return;
        }

        seekRequest = new chrome.cast.media.SeekRequest();
        seekRequest.currentTime = time;
        if (media) {
            media.seek(seekRequest);
            seekRequest = null;
        }
    };

    self.getDuration = function() {
        // unimplemented
    };

    self.getPaused = function() {
        if (media) {
            return media.playerState === chrome.cast.media.PlayerState.PAUSED;
        }
    };

    self.getPlaybackRate = function() {
        return 1.0;
    };

    self.setPlaybackRate = function() {
        // unimplemented
    };

    self.getPlayed = function() {
        // unimplemented
    };

    self.getEnded = function() {
        // unimplemented
    };

    self.getAutoplay = function() {
        // unimplemented
    };

    self.setLoop = function() {
        // unimplemented
    };

    self.play = function() {
        if (media) {
            var playRequest = new chrome.cast.media.PlayRequest();
            media.play(playRequest);
        } else {
            self.load();
        }
    };

    self.pause = function() {
        if (media) {
            var pauseRequest = new chrome.cast.media.PauseRequest();
            media.pause(pauseRequest);
        }
    };

    self.getControls = function() {
        // unimplemented
    };

    self.setControls = function() {
        // unimplemented
    };

    self.elapsedTime = function() {
        // unimplemented
    };

    self.getVolume = function() {
        if (media) {
            return media.volume.level;
        }
    };

    self.setVolume = function(value) {
        if (media) {
            var volume = new chrome.cast.Volume(value, null);
            var volumeRequest = new chrome.cast.media.VolumeRequest(volume);
            media.setVolume(volumeRequest);
        }
    };

    self.getMuted = function() {
        if (media) {
            return media.volume.muted;
        }

        if (session) {
            return session.receiver.volume.muted;
        }
    };

    self.setMuted = function(value) {
        if (media) {
            var volume = new chrome.cast.Volume(null, value);
            var volumeRequest = new chrome.cast.media.VolumeRequest(volume);
            media.setVolume(volumeRequest);
        }
    };

    self.getQuality = function() {
        return 'auto';
    };

    self.setQuality = function(quality) {
        if (session) {
            session.sendMessage('urn:x-cast:com.twitch.custom', {
                quality: quality,
            });
            events.emit(QUALITY_CHANGE, quality);
        }
    };

    self.getQualities = function() {
        // Create a mapping function to create compatible quality objects
        const createCompatibleQuality = quality => Object.assign({}, quality, {
            bandwidth: quality.bitrate,
        });
        return [AUTO_QUALITY, ...allQualities].map(createCompatibleQuality);
    };

    self.getChannel = function() {
        return channel;
    };

    self.setChannel = function(value, contentStream) {
        channel = value;
        video = null;
        contentStream.streamUrl.then(url => {
            streamUrl = url;
            if (session && channel && streamUrl) {
                // Automatically load if we already have a session.
                loadChannel();
            }
        });
    };

    self.getVideo = function() {
        return video;
    };

    self.setVideo = function(contentId, contentStream) {
        video = contentId;
        channel = null;
        contentStream.streamUrl.then(url => {
            streamUrl = url;
            if (session && video && streamUrl) {
                loadVideo();
            }
        });
    };

    self.getStats = function() {
        // unimplemented
        return {};
    };

    self.getVideoInfo = function() {
        // unimplemented
        return {};
    };

    self.getCaption = function() {
        // unimplemented
    };

    self.getBackend = function() {
        return 'chromecast';
    };

    // Kills the current cast session.
    self.stop = function() {
        if (session) {
            session.stop();
        }
    };

    self.getDevice = function() {
        if (session && session.receiver) {
            return session.receiver.friendlyName;
        }
    };

    self.absAvailable = function() {};
})();
