export const BACKEND_CLIP = 'clip backend';

const DEFAULT_OPTIONS = Object.freeze({
    autoplay: true,
});

const SUPPORTED_TYPES = Object.freeze({
    mp4: 'video/mp4',
});

const DEFAULT_QUALITY = '480p30';

export class BackendClip {
    constructor(opts = DEFAULT_OPTIONS, store) {
        this._store = store;
        this._options = opts;
        this._src = '';

        this._initVideoTag();
    }

    getBackend() {
        return BACKEND_CLIP;
    }

    _initVideoTag() {
        this._video = document.createElement('video');
        this._video.autoplay = this._options.autoplay;
        this._video.preload = 'auto';
        this._video.type = SUPPORTED_TYPES.mp4;
    }

    attach(element) {
        element.insertBefore(this._video, element.firstChild);
    }

    destroy() {
        this._video = null;
        this._events = null;
        this._src = '';
    }

    addEventListener(name, func) {
        this._video.addEventListener(name, func);
    }

    removeEventListener(name, func) {
        this._video.removeEventListener(name, func);
    }

    setSrc(value) {
        this._src = value;
        this._video.src = value;
        this.load();
    }

    load() {
        if (this._src !== '') {
            this._video.load();
        }
    }

    play() {
        this._video.play();
    }

    pause() {
        this._video.pause();
    }

    getVolume() {
        return this._video.volume;
    }

    setVolume(volume) {
        this._video.volume = volume;
    }

    getSeeking() {
        return this._video.seeking;
    }

    getCurrentTime() {
        return this._video.currentTime;
    }

    setCurrentTime(currentTime) {
        this._video.currentTime = currentTime;
    }

    getDuration() {
        return this._video.duration;
    }

    getPaused() {
        return this._video.paused;
    }

    getPlaybackRate() {
        return this._video.playbackRate;
    }

    setPlaybackRate(playbackRate) {
        this._video.playbackRate = playbackRate;
    }

    getPlayed() {
        return this._video.played;
    }

    getEnded() {
        return this._video.ended;
    }

    getAutoplay() {
        return this.options.autoplay;
    }

    setLoop(loop) {
        this._video.loop = loop;
    }

    getMuted() {
        return this._video.muted;
    }

    setMuted(muted) {
        this._video.muted = muted;
    }

    getQuality() { /* TODO */ }
    setQuality(selectedQuality) {
        const { quality } = this._store.getState();

        const qualityAvailable = quality.available.find(({ group }) => group === selectedQuality);
        if (qualityAvailable) {
            return this.setSrc(qualityAvailable.source);
        }

        const defaultUrl = quality.available.find(({ group }) => group === DEFAULT_QUALITY);
        if (defaultUrl) {
            this.setSrc(defaultUrl.source);
        }
    }
    getQualities() { /* TODO */ }

    // UNIMPLEMENTED
    absAvailable() {
        return false;
    }

    getVersion() {
        return '';
    }

    getStats() {
        return {};
    }

    getVideoInfo() {
        return {};
    }

    static canPlay() {
        return true;
    }

    setChannel() {}
    setVideo() {}
    elapsedTime() {}
    getCaption() {}
    getNetworkState() {}
    getBuffered() {}
    getReadyState() {}
    getError() {}
    getSrc() {}
}
