import isFinite from 'lodash/isFinite';
import throttle from 'lodash/throttle';
import round from 'lodash/round';
import assign from 'lodash/assign';
import EventEmitter from 'event-emitter';
import { Deferred } from '../util/deferred';
import { ExtrapolatingTimer } from '../util/extrapolating-timer';
import { videoInfo } from '../api';
import * as flash from 'util/flash';
import * as MediaEvents from './events/media-event';
import * as TwitchEvents from './events/twitch-event';
import * as NetworkState from './state/network-state';
import * as ReadyState from './state/ready-state';
import { LiveTwitchContentStream } from '../stream/twitch-live';
import { AdContentTypes } from '../actions/ads';
import { QUALITY_AUTO } from '../actions/quality';
import { adCompanionRendered } from '../actions/embed-event-emitter';
import Errors from 'errors';
import { setError } from 'actions/error';

const TIMEUPDATE_PERIOD = 250; // in milliseoconds
const VIDINFO_THROTTLE = 1500;

const MEDIUM_QUALITY = 'medium';

/* eslint-disable camelcase */
const NULL_VIDEO_INFO = {
    bandwidth: 0.0,
    broadcast_id: 0,
    cluster: '',
    current_bitrate: 0.0,
    current_fps: 0,
    current_fps_exact: 0.0,
    deblocking: 0,
    dropped_frames: 0,
    hls_buffer_duration: 0,
    hls_latency_encoder: 0,
    hls_latency_broadcaster: 0,
    hls_target_duration: 0,
    manifest_node: '',
    manifest_node_type: '',
    manifest_cluster: '',
    node: '',
    playback_bytes_per_second: 0.0,
    segment_protocol: '',
    serving_id: '',
    smoothing: true,
    stageHeight: 0,
    stream_protocol: '',
    stageWidth: 0,
    stream_time: 0,
    stream_time_offset: 0.0,
    totalMemoryNumber: 0,
    user_ip: '',
    vid_display_height: 0,
    vid_display_width: 0,
    vid_height: 0,
    vid_width: 0,
    video_buffer_size: 0.0,
};

// Ad events from flash backend
const BACKEND_AD_DISPLAY_STARTED = 'adDisplayStarted';
const BACKEND_AD_DISPLAY_ENDED = 'adDisplayEnded';
const BACKEND_AD_REQUEST = 'adRequested';
const BACKEND_AD_REQUEST_DECLINED = 'adRequestDeclined';
const BACKEND_AD_REQUEST_RESPONSE = 'adRequestResponse';
const BACKEND_AD_REQUEST_ERROR = 'adRequestError';
const BACKEND_AD_ERROR = 'adError';
const BACKEND_COMPANION_RENDERED = 'adcompanionrendered';
const BACKEND_AD_IMPRESSION = 'adImpression';
const BACKEND_AD_IMPRESSION_COMPLETE = 'adImpressionComplete';
const BACKEND_AD_LOADED = 'adLoaded';

export const BACKEND_FLASH = 'flash';

// Flash Ad Events
export const FLASH_AD_EVENTS = Object.freeze({
    AD_START: 'flashAdstart',
    AD_END: 'flashAdend',
    AD_REQUEST: 'flashAdRequested',
    AD_REQUEST_DECLINED: 'flashAdRequestDeclined',
    AD_REQUEST_RESPONSE: 'flashAdRequestResponse',
    AD_REQUEST_ERROR: 'flashAdRequestError',
    AD_ERROR: 'flashAdError',
    AD_COMPANION_RENDERED: 'flashAdcompanionrendered',
    AD_IMPRESSION: 'flashAdImpression',
    AD_IMPRESSION_COMPLETE: 'flashAdImpressionComplete',
    AD_LOADED: 'flashAdLoaded',
});

// eslint-disable-next-line max-statements
export function BackendFlash(options, store) {
    const self = this;

    // Assign a unique id for this instance.
    // Flash callbacks are static so we need to reference them based on id.
    const id = BackendFlash.counter++;
    BackendFlash.map[id] = self;

    // flashEvents are the incoming events from our Flash API.
    // videoEvents are the renamed events conforming to the html5 spec.
    const flashEvents = new EventEmitter();
    const videoEvents = new EventEmitter();

    const ready = new Deferred();

    const timer = new ExtrapolatingTimer();

    let initialized = false;
    let timeUpdateIntervalID = null;
    let readyState = 0;
    let networkState = 0;
    let channel = null;
    let video = null;
    let pendingChannel;
    let pendingVideo;
    let paused = false;
    let seeking;
    let duration = 0;
    let volume;
    let unmuteVolume;
    let format;
    let formats;
    let caption;
    let buffer;
    let player;
    let error;
    let swfElementId = '';
    let stream;
    let ended = false;
    let loadedStream = false;
    let loadedVideo = false;

    // eslint-disable-next-line max-statements
    function init() {
        flashEvents.on(MediaEvents.CAN_PLAY, onVideoCanPlay);
        flashEvents.on(MediaEvents.DURATION_CHANGE, onVideoDurationChange);
        flashEvents.on(MediaEvents.SEEKING, onVideoSeeking);
        flashEvents.on(MediaEvents.SEEKED, onVideoSeeked);
        flashEvents.on(MediaEvents.ENDED, onVideoEnded);
        flashEvents.on(MediaEvents.ERROR, onVideoError);
        flashEvents.on(MediaEvents.PLAYING, onVideoPlaying);
        flashEvents.on(MediaEvents.WAITING, onVideoWaiting);
        flashEvents.on(MediaEvents.VOLUME_CHANGE, onVideoVolumeChange);
        flashEvents.on(MediaEvents.LOADSTART, onVideoLoadStart);

        flashEvents.on(TwitchEvents.PLAYER_INIT, onVideoInit);
        flashEvents.on(TwitchEvents.SEEK_FAILED, onVideoSeekFailed);
        flashEvents.on(TwitchEvents.CHANSUB_REQUIRED, onVideoChanSubRequired);
        flashEvents.on(TwitchEvents.MANIFEST_EXTRA_INFO, onManifestExtraInfo);
        flashEvents.on(TwitchEvents.VIDEO_FAILURE, onVideoFailure);
        flashEvents.on(TwitchEvents.FORMATS, onVideoFormats);
        flashEvents.on(TwitchEvents.FORMAT_CHANGED, onVideoFormatChanged);
        flashEvents.on(TwitchEvents.TIME_CHANGE, onVideoTimeChange);
        flashEvents.on(TwitchEvents.BUFFER_CHANGE, onVideoBufferChange);
        flashEvents.on(TwitchEvents.SEGMENT_CHANGE, onSegmentChange);
        flashEvents.on(TwitchEvents.USHER_FAIL_ERROR, onUsherFailError);
        flashEvents.on(TwitchEvents.CAPTION_UPDATE, onCaptionUpdate);
        flashEvents.on(TwitchEvents.MIDROLL_REQUESTED, onMidrollRequested);

        flashEvents.on(TwitchEvents.VIDEO_PAUSED, onVideoPaused);
        flashEvents.on(TwitchEvents.STREAM_LOADED, onVideoStreamLoaded);
        flashEvents.on(TwitchEvents.VIDEO_LOADED, onVideoLoaded);

        flashEvents.on(BACKEND_AD_DISPLAY_STARTED, onVideoAdStarted);
        flashEvents.on(BACKEND_AD_DISPLAY_ENDED, onVideoAdEnded);
        flashEvents.on(BACKEND_COMPANION_RENDERED, onVideoAdCompanionRendered);
        flashEvents.on(BACKEND_AD_REQUEST, onVideoAdRequest);
        flashEvents.on(BACKEND_AD_REQUEST_DECLINED, onVideoAdRequestDeclined);
        flashEvents.on(BACKEND_AD_REQUEST_RESPONSE, onVideoAdRequestResponse);
        flashEvents.on(BACKEND_AD_REQUEST_ERROR, onVideoAdRequestError);
        flashEvents.on(BACKEND_AD_ERROR, onVideoAdError);
        flashEvents.on(BACKEND_AD_IMPRESSION, onVideoAdImpression);
        flashEvents.on(BACKEND_AD_IMPRESSION_COMPLETE, onVideoAdImpressionComplete);
        flashEvents.on(BACKEND_AD_LOADED, onAdLoaded);
    }

    /**
     * Exposes stats to video.js
     */
    self.getStats = function() {
        return transformInfoToStats(self.getVideoInfo());
    };

    /**
     * Transforms getVideoInfo into stats
     */
    function transformInfoToStats(info) {
        return {
            playbackRate: round(info.playback_bytes_per_second * 0.0078125, 2),
            fps: round(info.current_fps),
            bufferSize: round(info.video_buffer_size),
            skippedFrames: info.dropped_frames,
            memoryUsage: `${Math.floor((info.totalMemoryNumber / 1024 / 1024)).toString()}MB`,
            // eslint-disable-next-line max-len
            hlsLatencyEncoder: info.hasOwnProperty('hls_latency_encoder') ? round(info.hls_latency_encoder * 0.001) : 0.0,
            // eslint-disable-next-line max-len
            hlsLatencyBroadcaster: info.hasOwnProperty('hls_latency_broadcaster') ? round(info.hls_latency_broadcaster * 0.001) : 0.0,
            videoResolution: info.hasOwnProperty('vid_width') ? `${info.vid_width}x${info.vid_height}` : '',
            displayResolution: info.hasOwnProperty('stageWidth') ? `${info.stageWidth}x${info.stageHeight}` : '',
            backendVersion: self.getVersion(),
        };
    }

    /**
     * Called on video playback init.
     */
    function onVideoInit() {
        initialized = true;
        ready.resolve();
        videoEvents.emit(TwitchEvents.PLAYER_INIT);
    }

    /**
     * Called on caption event.
     *
     * @param {Object} data
     */
    function onCaptionUpdate(data) {
        caption = data;
        videoEvents.emit(TwitchEvents.CAPTION_UPDATE);
    }

    /**
     * Called on video volume change.
     *
     * @param {Object} data
     */
    function onVideoVolumeChange(data) {
        // TODO Fix this on the Flash side.
        // Sometimes we get bad values like NaN.
        if (!isFinite(data.volume)) {
            return;
        }

        // Clamp the volume between 0 and 1
        volume = Math.min(Math.max(data.volume, 0), 1);
        videoEvents.emit(MediaEvents.VOLUME_CHANGE);
    }

    /**
     * Called on video duration change.
     *
     * @param {Object} data
     */
    function onVideoDurationChange(data) {
        duration = data.duration || 0;
        videoEvents.emit(MediaEvents.DURATION_CHANGE);
        // LOADED_METADATA is fired right after to match HTML5 spec.
        if (loadedVideo) {
            videoEvents.emit(MediaEvents.LOADED_METADATA);
        }
    }

    /**
     * Called on video formats list update.
     *
     * @param {Object} data
     */
    function onVideoFormats(data) {
        formats = data.formats;
    }

    /**
     * Called when the video format of the active video
     * stream has changed.
     *
     * @param {Object} data
     */
    function onVideoFormatChanged(data) {
        format = data.format;
        videoEvents.emit(TwitchEvents.QUALITY_CHANGE, {
            quality: data.format,
            isAuto: false,
        });
    }

    /**
     * The user agent begins looking for media data, as part
     * of the resource selection algorithm.
     *
     * @param {Object} data
     * @see http://www.w3.org/TR/html5/embedded-content-0.html#event-media-loadstart
     */
    function onVideoLoadStart(data) {
        // TODO Get the actual network status from Flash.
        if (data && data.format) {
            if (format !== data.format) {
                format = data.format;
                videoEvents.emit(TwitchEvents.QUALITY_CHANGE, {
                    quality: format,
                    isAuto: false,
                });
            }
        }

        networkState = NetworkState.NETWORK_LOADING;
        videoEvents.emit(MediaEvents.LOADSTART);

        duration = 0;
        videoEvents.emit(MediaEvents.DURATION_CHANGE);

        setTimerAndSendTimeUpdate(tryCall('getVideoTime'));
        pauseTimerAndPeriodicTimeUpdates();
    }

    /**
     * Called when the video stream has enough data in the buffer to play
     */
    function onVideoCanPlay() {
        videoEvents.emit(MediaEvents.CAN_PLAY);
    }

    /**
     * Called when the video stream has started or resumed playback.
     */
    function onVideoPlaying(data) {
        // Autoplay is supposed to keep paused true until video actually
        // starts playing.
        if (paused) {
            paused = false;
            videoEvents.emit(MediaEvents.PLAY);
        }

        // TODO: This is probably not needed, try to remove it.
        ended = false;

        // Set initial format selection
        format = data.format;

        setTimerAndSendTimeUpdate(tryCall('getVideoTime'));
        resumeTimerAndPeriodicTimeUpdates();

        // streamLoaded always fires, so this is the first time we actually
        // know if a channel is online or not. videoLoaded fires normally,
        // so only fire this event if we haven't already.
        if (readyState < ReadyState.HAVE_METADATA) {
            if (self.getChannel() !== null) {
                duration = Infinity;
            }
            readyState = ReadyState.HAVE_METADATA;
            videoEvents.emit(MediaEvents.LOADED_METADATA);
        }

        // Always true but doesn't hurt to check.
        if (readyState < ReadyState.HAVE_CURRENT_DATA) {
            readyState = ReadyState.HAVE_CURRENT_DATA;
            videoEvents.emit(MediaEvents.LOADED_DATA);
        }

        readyState = ReadyState.HAVE_FUTURE_DATA;
        videoEvents.emit(MediaEvents.PLAYING);
    }

    /**
     * Called when the video stream has been paused.
     */
    function onVideoPaused() {
        paused = true;
        videoEvents.emit(MediaEvents.PAUSE);
        if (stream instanceof LiveTwitchContentStream) {
            readyState = ReadyState.HAVE_NOTHING;
        }
        pauseTimerAndPeriodicTimeUpdates();
    }

    /**
     * Called on video playback failure.
     */
    function onVideoFailure() {
        videoEvents.emit(TwitchEvents.OFFLINE);

        if (!ended && readyState === ReadyState.HAVE_NOTHING) {
            ended = true;
            videoEvents.emit(MediaEvents.ENDED);
        }

        ended = true;
        readyState = ReadyState.HAVE_NOTHING;
        pauseTimerAndPeriodicTimeUpdates();
    }

    /**
     * Called on video playback error.
     *
     * @param {Object} data
     */
    function onVideoError() {
        store.dispatch(setError(Errors.CODES.ABORTED));
    }

    /**
     * Called when the stream or VOD has ended.
     */
    function onVideoEnded() {
        ended = true;

        readyState = ReadyState.HAVE_NOTHING;
        videoEvents.emit(MediaEvents.ENDED);

        pauseTimerAndPeriodicTimeUpdates();
    }

    /**
     * Called when we start trying to load a live stream.
     * Does NOT mean the channel is actually online.
     *
     * @param {Object} data
     */
    function onVideoStreamLoaded(data) {
        pauseTimerAndPeriodicTimeUpdates();
        setTimerAndSendTimeUpdate(0);

        // We don't know if the stream is online yet.
        channel = data.channel;
        video = null;
    }

    /**
     * Called when a archive video has loaded.
     *
     * @param {Object} data
     */
    function onVideoLoaded(data) {
        pauseTimerAndPeriodicTimeUpdates();
        setTimerAndSendTimeUpdate(0);
        loadedVideo = true;

        // Flash removes the "v", so we have to add it back.
        // ex. "12345" -> "v12345", but "c12345" -> "c12345"
        var videoId = data.videoId;
        if (!isNaN(videoId[0])) {
            videoId = `v${videoId}`;
        }

        video = videoId;
        channel = null;

        readyState = ReadyState.HAVE_METADATA;
    }

    /**
     * Called when the video stream has begun seeking
     * to a new offset in the stream.
     */
    function onVideoSeeking() {
        pauseTimerAndPeriodicTimeUpdates();

        seeking = true;
        videoEvents.emit(MediaEvents.SEEKING);
    }

    /**
     * Called on video stream seek complete.
     */
    function onVideoSeeked() {
        seeking = false;
        videoEvents.emit(MediaEvents.SEEKED);
    }

    /**
     * Called on video stream seek failure.
     */
    function onVideoSeekFailed() {
        seeking = false;
    }

    /**
     * Called when the video stream time has changed.
     *
     * @param {Object} data
     */
    function onVideoTimeChange() {
        // TODO Pass this value in as part of data.
        setTimerAndSendTimeUpdate(tryCall('getVideoTime'));
    }

    /**
     * Called when the video stream buffer has changed.
     *
     * @param {Object} data
     */
    function onVideoBufferChange(data) {
        buffer = data;
        videoEvents.emit(TwitchEvents.BUFFER_CHANGE, data);
    }

    /**
     * Playback has stopped because the next frame is not available,
     * but the user agent expects that frame to become available in
     * due course.
     *
     * @param {Object} data
     * @see http://www.w3.org/TR/html5/embedded-content-0.html#event-media-waiting
     */
    function onVideoWaiting() {
        // readyState is equal to or less than HAVE_CURRENT_DATA, and paused is false.
        // Either seeking is true, or the current playback position is not contained
        // in any of the ranges in buffered. It is possible for playback to stop for
        // other reasons without paused being false, but those reasons do not fire this
        // event (and when those situations resolve, a separate playing event is not
        // fired either): e.g. the element is newly blocked on its media controller,
        // or playback ended, or playback stopped due to errors, or the element has
        // paused for user interaction or paused for in-band content.
        pauseTimerAndPeriodicTimeUpdates();

        readyState = ReadyState.HAVE_CURRENT_DATA;

        videoEvents.emit(MediaEvents.WAITING);
    }

    /**
     * Called when the video stream playback statistics have
     * begun sending data to the UI on a set interval.
     *
     * @param {Object} data
     */
    function onVideoAdStarted(data) {
        paused = false;

        // Pause time only for VOD
        if (video) {
            pauseTimerAndPeriodicTimeUpdates();
        }
        videoEvents.emit(FLASH_AD_EVENTS.AD_START, data);
    }

    /**
     * Called when the video stream playback statistics have
     * stopped sending data to the UI on a set interval.
     *
     * @param {Object} data
     */
    function onVideoAdEnded(data) {
        // Resume time only for VOD
        // Channel time would not have been paused
        if (video && !ended) {
            setTimerAndSendTimeUpdate(tryCall('getVideoTime'));
            resumeTimerAndPeriodicTimeUpdates();
        }
        videoEvents.emit(FLASH_AD_EVENTS.AD_END, data);
    }

    /**
     * Called when a video ad has been requested.
     *
     * @param {Object} data
     */
    function onVideoAdRequest(data) {
        videoEvents.emit(FLASH_AD_EVENTS.AD_REQUEST, data);
    }

    /**
     * Called when a video ad request has been declined.
     *
     * @param {Object} data
     */
    function onVideoAdRequestDeclined(data) {
        videoEvents.emit(FLASH_AD_EVENTS.AD_REQUEST_DECLINED, data);
    }

    /**
     * Called when a video ad request has been responded to.
     *
     * @param {Object} data
     */
    function onVideoAdRequestResponse(data) {
        videoEvents.emit(FLASH_AD_EVENTS.AD_REQUEST_RESPONSE, data);
    }

    /**
     * Called when a video ad request has encountered a error.
     *
     * @param {Object} data
     */
    function onVideoAdRequestError(data) {
        videoEvents.emit(FLASH_AD_EVENTS.AD_REQUEST_ERROR, data);
    }

    /**
     * Called when a video ad encountered a error.
     *
     * @param {Object} data
     */
    function onVideoAdError(data) {
        videoEvents.emit(FLASH_AD_EVENTS.AD_ERROR, data);
    }

    /**
     * Called on video stream advertisment companion rendered.
     *
     * @param {Object} data
     */
    function onVideoAdCompanionRendered(data) {
        // TODO Remove the data payload from this event.
        videoEvents.emit(FLASH_AD_EVENTS.COMPANION_RENDERED, data);
        store.dispatch(adCompanionRendered(FLASH_AD_EVENTS.COMPANION_RENDERED, data));
    }

    /**
     * Called on ad impression URL ping.
     *
     * @param {Object} data
     */
    function onVideoAdImpression(data) {
        videoEvents.emit(FLASH_AD_EVENTS.AD_IMPRESSION, data);
    }

    /**
     * Called on ad impression recorded by ad providers.
     *
     * @param {Object} data
     */
    function onVideoAdImpressionComplete(data) {
        videoEvents.emit(FLASH_AD_EVENTS.AD_IMPRESSION_COMPLETE, data);
    }

    /**
     * Called on video stream Channel Subscription required.
     */
    function onVideoChanSubRequired() {
        videoEvents.emit(TwitchEvents.RESTRICTED);
    }

    /**
     * Called on video stream Channel Subscription required.
     */
    function onUsherFailError() {
        videoEvents.emit(TwitchEvents.USHER_FAIL_ERROR);
    }

    /**
     * Called on every video segment change.
     *
     * @param {Object} data
     */
    function onSegmentChange(data) {
        videoEvents.emit(TwitchEvents.SEGMENT_CHANGE, data);
    }

    /*
     * Called when the backend detects a midroll request
     */
    function onMidrollRequested(data) {
        videoEvents.emit(TwitchEvents.MIDROLL_REQUESTED, data);
    }

    /**
     * Emits a new timeupdate VideoEvent.
     */
    function emitTimeUpdateEvent() {
        videoEvents.emit(MediaEvents.TIME_UPDATE, timer.extrapolateTimeStamp());
    }

    /**
     * Called when current playback position changes in interesting way.
     *
     * @param {String} time
     */
    function setTimerAndSendTimeUpdate(time) {
        timer.setCurrentTimeStamp(time);
        videoEvents.emit(MediaEvents.TIME_UPDATE, timer.extrapolateTimeStamp());
    }

    /**
     * Used to update embed client. And implements html5 spec.
     */
    function resumeTimerAndPeriodicTimeUpdates() {
        if (timeUpdateIntervalID === null) {
            timeUpdateIntervalID = setInterval(emitTimeUpdateEvent, TIMEUPDATE_PERIOD);
            timer.resume();
        }
    }

    /**
     * Used to update embed client. And implements html 5 spec.
     * Called when playback position is paused.
     */
    function pauseTimerAndPeriodicTimeUpdates() {
        if (timeUpdateIntervalID !== null) {
            clearInterval(timeUpdateIntervalID);
            timeUpdateIntervalID = null;
            timer.pause();
        }
    }

    /**
     * Used to send spade event to notify ad has been loaded
     */
    function onAdLoaded(data) {
        videoEvents.emit(FLASH_AD_EVENTS.AD_LOADED, data);
    }

    /**
     * Used to send extra manifest info
     */
    function onManifestExtraInfo(info) {
        videoEvents.emit(TwitchEvents.MANIFEST_EXTRA_INFO, info);
    }

    /**
     * Attaches a new playback container to the DOM.
     *
     * @param {Element} element
     */
    self.attach = function(element) {
        const url = require('file-loader?name=vendor/[name].[hash].[ext]!../../../vendor/TwitchPlayer.swf');
        attachSwf(url, element);

        deferCall('setLastAdDisplay', [options.lastAdDisplay]);
        deferCall('setPlayerType', [options.player]);
        deferCall('setFlashIMAAdsEnabled', [true]);
    };

    function attachSwf(url, element) {
        // We use wmode direct in Chrome/Pepper for best performance.
        // This mode will actually cover the UI in other browsers.
        const PLAYER_TYPE = flash.getFlashPlayerType();

        if (PLAYER_TYPE === '') {
            store.dispatch(setError(Errors.CODES.ABORTED));
            return;
        }

        const PLAYER_VERSION = '10.2';
        const XI_SWF_URL = 'playerProductInstall.swf';
        const WIDTH = '100%';
        const HEIGHT = '100%';

        const FLASH_VARS = {
            eventsCallback: 'window._BackendFlash_emitEvents',
            eventsContext: id,
            initCallback: null,
        };

        const PARAMS = {
            bgcolor: '#000',
            allowscriptaccess: 'always',
            allowfullscreen: 'true',
            // We use wmode direct in Chrome/Pepper for best performance.
            wmode: PLAYER_TYPE === 'ppapi' ? 'direct' : 'transparent',
        };

        const ATTRIBUTES = { align: 'middle' };

        // Create a div for swfobject to replace.
        const swfDiv = document.createElement('div');
        element.insertBefore(swfDiv, element.children[0]);

        // Aassign a unique id so embedSWF can find the div.
        swfElementId = `swfobject-${id}`;
        swfDiv.setAttribute('id', swfElementId);

        flash.embedSWF(
            url,
            swfElementId,
            WIDTH,
            HEIGHT,
            PLAYER_VERSION,
            XI_SWF_URL,
            FLASH_VARS,
            PARAMS,
            ATTRIBUTES,
            function(e) {
                player = e.ref;
            }
        );
    }

    /**
     * Calls an external interface Flash method with the given name and arguments.
     * Calling .apply doesn't work for Debug Flash FOR WHATEVER REASON.
     *
     * @param {String} name
     * @param {Array=} args
     */
    function flashCall(name, args = []) {
        if (options.debug) {
            // eslint-disable-next-line no-console
            console.log('flash call:', name, args);
        }

        // TODO UGH, this is a huge hack.
        switch (args.length) {
        case 0:
            return player[name]();
        case 1:
            return player[name](args[0]);
        case 2:
            return player[name](args[0], args[1]);
        case 3:
            return player[name](args[0], args[1], args[2]);
        default:
            // This will break for the Flash debugger
            // eslint-disable-next-line no-console
            console.log('WARNING, too many arguments passed to Flash');
            return player[name].apply(this, args);
        }
    }

    /**
     * Queues the given function call until the flash player has loaded.
     * Use tryCall instead if you need a blocking call (ie. return value).
     *
     * @param {String} func
     * @param {Array=} args
     */
    function deferCall(func, args) {
        // Queue the function call until the player has loaded.
        ready.promise.then(function() {
            // Don't block on the Flash call; it might take a while.
            setTimeout(function() {
                var result = flashCall(func, args);

                if (result && options.debug) {
                    // eslint-disable-next-line no-console
                    console.log('flash return:', func, '=', result);
                }
            }, 0);
        });
    }

    /**
     * Call a function immediately iff the player has loaded.
     * If the player has not loaded yet, nothing happens.
     *
     * @param {String} func
     * @param {Array=} args
     */
    function tryCall(func, args) {
        if (initialized) {
            var result = flashCall(func, args);
            if (result && options.debug) {
                // eslint-disable-next-line no-console
                console.log('flash return:', func, '=', result);
            }

            return result;
        }
    }

    /**
     * Must be public and visible to _BackendFlash_emitEvents.
     *
     * @param {String} name
     * @param {Object} data
     */
    self._emitEvent = function(name, data) {
        // Defer the event in order to fix exception reporting.
        // No idea why this is still needed but it's a pain without it.
        setTimeout(function() {
            if (options.debug && name !== 'timeupdate' && name !== 'playbackStatistics') {
                // eslint-disable-next-line no-console
                console.log('flash event:', name, data);
            }

            flashEvents.emit(name, data);
        }, 0);
    };

    /**
     * Creates a new VideoEvents listener.
     *
     * @param {String} name
     * @param {function} callback
     */
    self.addEventListener = function(name, callback) {
        videoEvents.on(name, callback);
    };

    /**
     * Removes a VideoEvents listener.
     *
     * @param {String} name
     * @param {function} callback
     */
    self.removeEventListener = function(name, callback) {
        videoEvents.off(name, callback);
    };

    /**
     * Returns the current client network profile.
     */
    self.getNetworkProfile = function() {
        return tryCall('getNetworkProfile') || [];
    };

    /**
     * Returns the current playback error.
     */
    self.getError = function() {
        return error;
    };

    /**
     *
     */
    self.getNetworkState = function() {
        // One of the values defined above.
        // TODO Expose the actual network status from Flash.
        return networkState;
    };

    /**
     * Returns the current video buffer information.
     */
    self.getBuffered = function() {
        // Flash only returns a buffer of length 0 or 1.
        var length = (buffer ? 1 : 0);
        return {
            length: length,
            start(i) {
                if (i >= length) return;
                return buffer.start;
            },
            end(i) {
                if (i >= length) return;
                return buffer.end;
            },
        };
    };

    /**
     * Attempts to load the pending channel or video content stream
     * into the video playback container.
     */
    self.load = function() {
        if (!pendingChannel && !pendingVideo) {
            return;
        }

        readyState = ReadyState.HAVE_NOTHING;
        networkState = NetworkState.NETWORK_LOADING;
        videoEvents.emit(MediaEvents.LOADSTART);
        loadedVideo = false;

        if (pendingVideo) {
            // Because pendingVideo can change, we need to keep a copy of
            // it to verify once the callback resolves.
            const closurePendingVideo = pendingVideo;
            return Promise.all([videoInfo(pendingVideo), stream.streamUrl]).
                then(([videoInfo, streamUrl]) => {
                    const { streamMetadata } = store.getState();
                    if (closurePendingVideo === streamMetadata.videoId) {
                        loadedStream = true;
                        deferCall('loadStream', [streamUrl, videoInfo.channel.name, false]);
                    }
                });
        }
        const closurePendingChannel = pendingChannel;
        return stream.streamUrl.then(streamUrl => {
            const { streamMetadata } = store.getState();
            if (closurePendingChannel === streamMetadata.channelName) {
                loadedStream = true;
                deferCall('loadStream', [streamUrl, pendingChannel, true]);
            }
        });
    };

    /**
     * Returns the ready state.
     */
    self.getReadyState = function() {
        return readyState;
    };

    /**
     * Returns the seeking state.
     */
    self.getSeeking = function() {
        return seeking;
    };

    /**
     * Returns the current playback time of the content stream.
     */
    self.getCurrentTime = function() {
        return timer.extrapolateTimeStamp();
    };

    /**
     * Sets the current playback time in the content stream.
     *
     * @param {Number} time
     */
    self.setCurrentTime = function(time) {
        ended = false;

        // Most calls into the flash backend do not affect the timer
        // This one does because seeking requires us to know where
        // the scrubbing position wants to play at.
        pauseTimerAndPeriodicTimeUpdates();
        setTimerAndSendTimeUpdate(time);

        deferCall('videoSeek', [time]);
    };

    /**
     * Returns the duration of the content stream.
     */
    self.getDuration = function() {
        return duration;
    };

    /**
     * Returns the paused state.
     */
    self.getPaused = function() {
        return paused;
    };

    /**
     * Returns if the endlist has been located in the content stream.
     * Ended events are fired from EXT-X-ENDLIST tags in the HLS manifest.
     */
    self.getEnded = function() {
        if (store.getState().ads.currentMetadata.contentType !== AdContentTypes.NONE) {
            return false;
        }

        return ended;
    };

    /**
     * Begin or resume playback of the loaded content stream.
     */
    self.play = function() {
        paused = false;
        videoEvents.emit(MediaEvents.PLAY);

        const setStreamAndPlay = () => {
            stream.streamUrl.then(streamUrl => {
                deferCall('setStreamURI', [streamUrl]);
                deferCall('playVideo');
            });
        };
        if (!loadedStream) {
            self.load().then(setStreamAndPlay());
        } else {
            setStreamAndPlay();
        }
    };

    /**
     * Pauses playback of the content stream.
     */
    self.pause = function() {
        deferCall('pauseVideo');
    };

    /**
     * Returns the playback volume.
     */
    self.getVolume = function() {
        return volume;
    };

    /**
     * Sets the playback volume.
     *
     * @param {Number} value
     */
    self.setVolume = function(value) {
        if (value > 0) {
            unmuteVolume = value;
        }

        deferCall('setVolume', [value]);
    };

    /**
     * Returns the muted state.
     */
    self.getMuted = function() {
        var volume = self.getVolume();
        return volume === 0;
    };

    /**
     * Sets the volume muted state.
     *
     * @param {Boolean} muted
     */
    self.setMuted = function(muted) {
        var volume = muted ? 0 : (unmuteVolume || 0.5);
        self.setVolume(volume);
    };

    /**
     * Attempts to retrieve the playback video info object from the Flash Backend.
     */
    self.getVideoInfo = throttle(function() {
        const videoInfo = tryCall('getVideoInfo');
        return videoInfo ? assign({}, NULL_VIDEO_INFO, videoInfo) : NULL_VIDEO_INFO;
    }, VIDINFO_THROTTLE);

    /**
     * Returns the current Twitch channel in playback.
     */
    self.getChannel = function() {
        return pendingChannel || channel;
    };

    /**
     * Sets a new Twitch channel for playback.
     *
     * @param {String} value
     * @param {String} stream's URI
     */
    self.setChannel = function(value, contentStream) {
        stream = contentStream;
        pendingChannel = value;
        pendingVideo = null;
        video = null;

        // Instantly load the new channel if autoplay is enabled or we
        // are currently playing another channel.
        if (options.autoplay || networkState > NetworkState.NETWORK_EMPTY) {
            self.load();
        } else {
            self.pause();
        }
    };

    /**
     * Returns the current video ID.
     */
    self.getVideo = function() {
        return pendingVideo || video;
    };

    /**
     * Sets the Twitch VOD ID for playback
     *
     * @param {String} value
     * @param {String} stream's URI
     */
    self.setVideo = function(value, contentStream) {
        stream = contentStream;
        pendingVideo = value;
        pendingChannel = null;
        channel = null;

        // Instantly load the new video if autoplay is enabled or we
        // are currently playing another video.
        if (options.autoplay || networkState > NetworkState.NETWORK_EMPTY) {
            self.load();
        } else {
            self.pause();
        }
    };

    /**
     * Returns the current playback quality.
     */
    self.getQuality = function() {
        return format || '';
    };

    /**
     * Attempts to set a new preferred playback quality.
     *
     * @param {String} quality
     */
    self.setQuality = function(quality) {
        // If user selects auto somehow, set their quality to medium
        if (quality === QUALITY_AUTO) {
            deferCall('setQuality', [MEDIUM_QUALITY]);
            format = MEDIUM_QUALITY;
        } else {
            deferCall('setQuality', [quality]);
            format = quality;
        }
    };

    /**
     * Returns the current list of playback qualities available
     * for the current content stream.
     */
    self.getQualities = function() {
        return formats || [];
    };

    /**
     * Returns the current Backend identifier.
     */
    self.getBackend = function() {
        return BACKEND_FLASH;
    };

    self.elapsedTime = function() {
        // unimplemented
    };

    /**
     * Returns the current Flash Backend version.
     */
    self.getVersion = function() {
        return tryCall('getVersion');
    };

    /**
     * The mediagroup content attribute on media elements can be used to link multiple
     * media elements together by implicitly creating a MediaController. The value is
     * text; media elements with the same value are automatically linked by the user agent.
     */
    self.getMediaGroup = function() {
        return null;
    };

    /**
     * The controller attribute on a media element, on getting, must return the element's
     * current media controller, if any, or null otherwise.
     */
    self.getController = function() {
        return null;
    };

    /**
     *
     */
    self.getControls = function() {
        return false;
    };

    /**
     * Returns an AudioTrackList object representing the audio tracks
     * available in the media resource.
     */
    self.getAudioTracks = function() {
        return null;
    };

    /**
     * Returns a VideoTrackList object representing the video
     * tracks available in the media resource.
     */
    self.getVideoTracks = function() {
        return null;
    };

    /**
     * The textTracks attribute of media elements must return a TextTrackList object
     * representing the TextTrack objects of the text tracks in the media element's list
     * of text tracks, in the same order as in the list of text tracks.
     */
    self.getTextTracks = function() {
        return null;
    };

    /**
     *
     */
    self.getPlaybackRate = function() {
        return 1;
    };

    /**
     * Creates and returns a new MutableTextTrack object, which is also added to the
     * media element's list of text tracks.
     */
    self.addTextTrack = function() {};

    /**
     *
     */
    self.setControls = function() {};

    /**
     * The loop attribute is a boolean attribute that, if specified, indicates that
     * the media element is to seek back to the start of the media resource upon reaching the end.
     */
    self.setLoop = function() {};

    /**
     * The playbackRate attribute gives the effective playback rate (assuming there is no
     * current media controller overriding it), which is the speed at which the media resource
     * plays, as a multiple of its intrinsic speed.
     */
    self.setPlaybackRate = function() {
        videoEvents.emit(MediaEvents.RATE_CHANGE);
    };

    /**
     * The played attribute must return a new static normalised TimeRanges object that
     * represents the ranges of points on the media timeline of the media resource reached through
     * the usual monotonic increase of the current playback position during normal playback, if
     * any, at the time the attribute is evaluated.
     */
    self.getPlayed = function() {};

    /**
     * The src content attribute on media elements gives the address
     * of the media resource (video, audio) to show.
     */
    self.getSrc = function() {};

    /**
     * The src content attribute on media elements gives the address of
     * the media resource (video, audio) to show.
     */
    self.setSrc = function() {};

    /**
     * Returns caption data from stream, if any.
     */
    self.getCaption = function() {
        return caption;
    };

    self.absAvailable = function() {
        // NO-OP
    };

    /**
     * Get the flash player to request an ad with the given duration and ad tag URL
     */
    self.requestAdFill = function(duration, adType, adTagUrl) {
        deferCall('requestFlashAd', [duration, adType, adTagUrl]);
    };

    /**
     * Destroy the Flash Backend.
     */
    self.destroy = function() {
        // Stop intervals and timers
        pauseTimerAndPeriodicTimeUpdates();

        // Remove the SWF
        flash.removeSWF(swfElementId);

        player = null;
    };

    init();
}

// We need some static members to uniquely identify objects.
BackendFlash.map = {};
BackendFlash.counter = 0;

/**
 * Verify that the client is running a > Flash Player 10.2 and also check if genuine
 * player type is available
 */
BackendFlash.canPlay = function() {
    // Checking for flash version returns true in cases of web-view even when no flash player is
    // actually available. Adding another check to check for type to ensure flash backend can really play
    // More details https://jira.twitch.com/browse/VP-2532
    return flash.hasFlashPlayerVersion('10.2') && (flash.getFlashPlayerType() !== '');
};

/**
 * Flash fires this static callback on events.
 * ugh, this needs to be public so Flash can discover it.
 */
window._BackendFlash_emitEvents = function(context, es) { // eslint-disable-line camelcase
    // Map from the id to the BackendFlash object.
    var self = BackendFlash.map[context];

    // Loop over all of the events and fire them sequentially.
    for (var i = 0; i < es.length; i++) {
        var e = es[i];
        self._emitEvent(e.event, e.data);
    }
};
