import isFunction from 'lodash/isFunction';
import EventEmitter from 'event-emitter';
import * as MediaEvents from './events/media-event';
import * as TwitchEvents from './events/twitch-event';
import Errors from 'errors';
import { setError } from 'actions/error';

const EVENT_LOADED_METADATA = 'loadedmetadata';
const EVENT_ERROR = 'error';
const DEFAULT_STATS = {
    playbackRate: 0,
    fps: 0,
    bufferSize: 0,
    skippedFrames: 0,
    memoryUsage: 'Unknown',
    hlsLatencyEncoder: 0.0,
    hlsLatencyBroadcaster: 0.0,
    videoResolution: '0x0',
    displayResolution: '0x0',
};

export const BACKEND_HLS = 'hls';

// eslint-disable-next-line max-statements
export function BackendHls(options, store) {
    const self = this;

    let events;
    let videoElement;
    let channel;
    let video;
    let _srcHasBeenSet;
    const _srcReady = new Promise(resolve => {
        _srcHasBeenSet = resolve;
    });

    function init() {
        events = new EventEmitter();

        videoElement = document.createElement('video');
        videoElement.autoplay = options.autoplay;

        if (options.playsinline) {
            videoElement.setAttribute('webkit-playsinline', '');
            videoElement.setAttribute('playsinline', '');
        }

        if (options.muted) {
            videoElement.setAttribute('muted', '');
        }

        setTimeout(() => {
            events.emit(TwitchEvents.PLAYER_INIT);
        }, 0);
    }

    self.attach = function(element) {
        element.appendChild(videoElement);

        videoElement.addEventListener(EVENT_ERROR, function() {
            if (videoElement.error === null) {
                // unsure if this can happen, but just in case...
                return;
            }

            // if the error from the video element is MEDIA_ERR_SRC_NOT_SUPPORTED,
            // then what most likely happened is that the master manifest 404'd,
            // which we should hide from downstream consumers and instead proxy
            // out as `ended`.
            if (videoElement.error.code === videoElement.error.MEDIA_ERR_SRC_NOT_SUPPORTED) {
                events.emit(TwitchEvents.OFFLINE);
                events.emit(MediaEvents.ENDED);
                return;
            }

            const ERROR_CODE_MAP = {
                [videoElement.error.MEDIA_ERR_NETWORK]: Errors.CODES.NETWORK,
                [videoElement.error.MEDIA_ERR_DECODE]: Errors.CODES.DECODE,
                [videoElement.error.MEDIA_ERR_ABORTED]: Errors.CODES.ABORTED,
            };

            store.dispatch(setError(ERROR_CODE_MAP[videoElement.error.code]));
        });

        videoElement.addEventListener(EVENT_LOADED_METADATA, function() {
            events.emit(TwitchEvents.QUALITY_CHANGE, self.getQuality());
        });
    };

    self.destroy = function() {
        // empty
    };

    self.addEventListener = function(name, callback) {
        // Register the event on the video AND our custom events class.
        // This lets us provide a superset of the events.
        if (name !== EVENT_ERROR) {
            videoElement.addEventListener(name, callback);
        }
        events.on(name, callback);
    };

    self.removeEventListener = function(name, callback) {
        videoElement.removeEventListener(name, callback);
        events.off(name, callback);
    };

    self.getNetworkProfile = function() {
        // TODO implement network profile collection for this backend
        return [];
    };

    self.getError = function() {
        // if the error is MEDIA_ERR_SRC_NOT_SUPPORTED, then most likely the master
        // manifest 404'd, and that means the stream is simply not online. Hide
        // this from consumers.
        if (
            videoElement.error !== null &&
            videoElement.error.code !== videoElement.error.MEDIA_ERR_SRC_NOT_SUPPORTED
        ) {
            return videoElement.error;
        }

        return null;
    };

    self.getSrc = function() {
        return videoElement.src;
    };

    self.setSrc = function(value) {
        videoElement.src = value;
        self.load();
        _srcHasBeenSet();
    };

    self.getNetworkState = function() {
        return videoElement.networkState;
    };

    self.getBuffered = function() {
        return videoElement.buffered;
    };

    self.load = function() {
        videoElement.load();
    };

    self.getReadyState = function() {
        return videoElement.readyState;
    };

    self.getSeeking = function() {
        return videoElement.seeking;
    };

    self.getStats = function() {
        // TODO: Implement
        return DEFAULT_STATS;
    };

    self.getCurrentTime = function() {
        return videoElement.currentTime;
    };

    self.setCurrentTime = function(value) {
        videoElement.currentTime = value;
    };

    self.getDuration = function() {
        return videoElement.duration;
    };

    self.getPaused = function() {
        return videoElement.paused;
    };

    self.getPlaybackRate = function() {
        return videoElement.playbackRate;
    };

    self.setPlaybackRate = function(value) {
        videoElement.playbackRate = value;
    };

    self.getPlayed = function() {
        return videoElement.played;
    };

    self.getEnded = function() {
        return (
            videoElement.ended ||
            // If the video element has an error and it is MEDIA_ERR_SRC_NOT_SUPPORTED,
            // then the most likely cause is that the master manifest 404'ed,
            // which we should communicate to downstream users as `ended`
            (
                !!(videoElement.error) &&
                videoElement.error.code === videoElement.error.MEDIA_ERR_SRC_NOT_SUPPORTED
            )
        );
    };

    self.getAutoplay = function() {
        return videoElement.autoplay;
    };

    self.setLoop = function(value) {
        videoElement.loop = value;
    };

    self.play = function() {
        return _srcReady.
            then(() => videoElement.play()).
            catch(e => e);
    };

    self.pause = function() {
        videoElement.pause();
    };

    // TODO mediaGroup, controller, audioTracks, videoTracks, textTracks

    self.getControls = function() {
        return false;
    };

    self.setControls = function() {
        // nothing
    };

    self.getVolume = function() {
        return videoElement.volume;
    };

    self.setVolume = function(value) {
        videoElement.volume = value;
    };

    self.getMuted = function() {
        return videoElement.muted;
    };

    self.setMuted = function(value) {
        if (value) {
            videoElement.setAttribute('muted', '');
        } else {
            videoElement.removeAttribute('muted');
        }

        videoElement.muted = value;
    };

    // Custom stuff
    self.getChannel = function() {
        return channel;
    };

    self.setChannel = function(value, stream) {
        channel = value;
        video = null;

        stream.streamUrl.then(streamUrl => {
            self.setSrc(streamUrl);
        });
    };

    self.getVideo = function() {
        return video;
    };

    self.setVideo = function(value, stream) {
        video = value;
        channel = null;

        stream.streamUrl.then(streamUrl => {
            self.setSrc(streamUrl);
        });
    };

    // TODO Properly implement quality switching.
    self.getQuality = function() {
        return 'auto';
    };

    self.setQuality = function() {
        // TODO
    };

    self.getQualities = function() {
        return [{
            bandwidth: -1,
            resolution: '0x0',
            group: 'auto',
            name: 'Auto',
        }];
    };

    self.getVideoInfo = function() {
        // unimplemented
        return {};
    };

    self.getCaption = function() {
        // unimplemented
    };

    self.getBackend = function() {
        return BACKEND_HLS;
    };

    self.getVersion = function() {
        // TODO
    };

    self.elapsedTime = function() {
        // unimplemented
    };

    self.absAvailable = function() {};

    init();
}

BackendHls.canPlay = function() {
    // Create a temporary video element for the check.
    const videoElement = document.createElement('video');

    if (isFunction(videoElement.canPlayType)) {
        return videoElement.canPlayType('application/vnd.apple.mpegURL');
    }

    return false;
};
