import { PLAYER_INIT, SEGMENT_CHANGE, QUALITY_CHANGE,
         CAPTION_UPDATE, MANIFEST_EXTRA_INFO, MIDROLL_REQUESTED,
         STITCHED_AD_START, STITCHED_AD_END, ABS_STREAM_FORMAT_CHANGE,
         OFFLINE, BUFFER_CHANGE } from './events/twitch-event';
import { VIDEO_PLAY_MASTER_MANIFEST } from '../analytics/analytics';
import { PLAYBACK_ERROR } from '../analytics/spade-events';
import * as MediaEvent from './events/media-event';
import EventEmitter from 'events';
import find from 'lodash/find';
import reduce from 'lodash/reduce';
import round from 'lodash/round';
import PlayerCoreLoader from 'player-core-loader';
import { MEDIAPLAYER_VERSION_CONTROL, FAST_BREAD } from '../experiments.js';
import * as ReadyState from './state/ready-state';
import * as NetworkState from './state/network-state';
import { trackEvent } from 'actions/analytics-tracker';
import Errors from 'errors';
import { setError } from 'actions/error';
import { OFFLINE_STATUS } from 'state/online-status';

export const BACKEND_MEDIA_PLAYER = 'mediaplayer';

export const FATAL_ERROR_CODE = 8001;
export const OFFLINE_ERROR_CODE = 8002;
export const FATAL_AUTH_ERROR_CODE = 8003;
export const WARN_AUTH_ERROR_CODE = 8004;
export const LVS_CCU_CAP_ERROR_CODE = 509;

export const PLAYER_FATAL_ERROR = 'fatal_error';
export const PLAYER_OFFLINE_ERROR = 'offline_error';
export const PLAYER_FATAL_AUTH_ERROR = 'fatal_auth_error';
export const PLAYER_WARN_AUTH_ERROR = 'warn_auth_error';
export const PLAYER_LVS_CCU_CAP_ERROR = 'ccu_cap_error';

// TODO:
// 1. Handle VIDEO_PLAY_VARIANT_MANIFEST analytics event

const AUTO_QUALITY = {
    group: 'auto',
    name: 'Auto',
};

/* eslint-disable camelcase */
const DEFAULT_INFO_OBJECT = {
    broadcast_id: 0,
    cluster: '',
    manifest_cluster: '',
    manifest_node: '',
    manifest_node_type: '',
    node: '',
    serving_id: 0,
    user_ip: '',
    vod_cdn_origin: '',
    vod_cdn_region: '',
    stream_time_offset: 0,
    segment_protocol: '',
    bandwidth: 0,
    current_bitrate: 0,
    current_fps: 0,
    current_fps_exact: 0,
    dropped_frames: 0,
    hls_latency_broadcaster: 0,
    hls_latency_encoder: 0,
    hls_target_duration: 0,
    paused: false,
    playing: false,
    stream_time: 0,
    totalMemoryNumber: 0,
    vid_display_height: 0,
    vid_display_width: 0,
    vid_height: 0,
    vid_width: 0,
    video_buffer_size: 0,
    volume: 0,
};

const DEFAULT_STATS_OBJECT = Object.freeze({
    playbackRate: -1,
    fps: -1,
    bufferSize: -1,
    skippedFrames: -1,
    memoryUsage: '0 MB',
    hlsLatencyEncoder: -1,
    hlsLatencyBroadcaster: -1,
    videoResolution: '',
    displayResolution: '',
    backendVersion: '',
});
/* eslint-enable camelcase */

export class BackendMediaPlayer {
    constructor(options = {}, store) {
        this.store = store;
        this._options = options;
        this._mediaPlayer = null;
        this._eventEmitter = new EventEmitter();
        this._apiCallQueue = [];
        this._cache = {}; // populated when mediaplayer is initialized
        this._readyState = ReadyState.HAVE_NOTHING;
        this._networkState = NetworkState.NETWORK_NO_SOURCE;

        // Used for firing LOADED_METADATA event.
        // This fix is documented in the jira issue: VP-2604.
        this._fireLoadedMetadata = false;

        // Used for retry logic for reloading mediaplayer if it throws an auth error
        this._hasRetried = false;

        this._src = '';
        this._currentCaptionData = {};
        this._mediaPlayerLogLevel = options['cvp-log'] || 'error'; // choose between debug, warn, info, error
        this.initialize();
    }

    initialize() {
        return this.loadMediaPlayer().then(result => {
            // The result has two properties - {mediaPlayerInstance, mediaPlayerHandle}
            // The mediaPlayerInstance is the instance of mediaplayer to interface with
            // The mediaPlayerHandle is the handle to MediaPlayer which has references to all
            // events and errors
            this._initMediaPlayer(result);
            this.setMuted(this._options.muted);
            this._initMobileAttributes();
        }, err => {
            this.onCoreAnalytics(err);
        });
    }

    _initMediaPlayer({ mediaPlayerInstance, mediaPlayerHandle }) {
        this._mediaPlayer = mediaPlayerInstance;
        this._mediaPlayerHandle = mediaPlayerHandle;

        // Init the cache to observe changes
        this._cache = {
            currentQuality: this._mediaPlayer.getQuality(),
        };

        // Attach other listeners
        this._attachInternalListeners(mediaPlayerHandle);

        // Fire PLAYER_INIT to signal that mediplayer is ready
        this._eventEmitter.emit(PLAYER_INIT);

        // Empty any queued operations
        this._apiCallQueue.forEach(apiCall => {
            apiCall();
        });
        this._apiCallQueue = [];
    }

    _initMobileAttributes() {
        if (this._options.playsinline) {
            this._mediaPlayer.getHTMLVideoElement().setAttribute('webkit-playsinline', '');
            this._mediaPlayer.getHTMLVideoElement().setAttribute('playsinline', '');
        }
    }

    loadMediaPlayer() {
        const { experiments } = this.store.getState();
        const logLevel = this._mediaPlayerLogLevel;

        return Promise.all([
            experiments.get(MEDIAPLAYER_VERSION_CONTROL),
            experiments.get(FAST_BREAD),
        ]).then(function(values) {
            return PlayerCoreLoader.loadMediaPlayer({
                value: values[0],
                latencyValue: values[1],
                logLevel: logLevel,
            });
        });
    }

    getStats() {
        const mediaPlayer = this._mediaPlayer;
        if (mediaPlayer) {
            const videoResolution = `${mediaPlayer.getVideoWidth()}x${mediaPlayer.getVideoHeight()}`;
            const displayResolution = `${mediaPlayer.getDisplayWidth()}x${mediaPlayer.getDisplayHeight()}`;
            return {
                playbackRate: round(mediaPlayer.getVideoBitRate() / 1000),
                fps: round(mediaPlayer.getVideoFrameRate()),
                bufferSize: round(mediaPlayer.getBufferDuration()),
                skippedFrames: mediaPlayer.getDroppedFrames(),
                memoryUsage: '64 MB',
                hlsLatencyEncoder: round(mediaPlayer.getTranscoderLatency() / 1000),
                hlsLatencyBroadcaster: round(mediaPlayer.getBroadcasterLatency() / 1000),
                videoResolution: videoResolution,
                displayResolution: displayResolution,
                backendVersion: mediaPlayer.getVersion(),
            };
        }
        return DEFAULT_STATS_OBJECT;
    }

    /**
     * Called to fetch current caption data.
     */
    getCaption() {
        return this._currentCaptionData;
    }

    /**
     * Called on video player core's CORE_ANALYTICS event
     */
    onCoreAnalytics(payload) {
        const eventName = payload.spadeEventName;
        const data = payload.spadeEventData;
        this.store.dispatch(trackEvent(eventName, data));
    }

    /**
     * Creates a new VideoEvents listener.
     *
     * @param {String} name
     * @param {function} callback
     */
    addEventListener(name, fn) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.addEventListener.bind(this, name, fn));
            return;
        }

        this._eventEmitter.on(name, fn);
    }

    /**
     * Removes a VideoEvents listener.
     *
     * @param {String} name
     * @param {function} callback
     */
    removeEventListener(name, fn) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.removeEventListener.bind(this, name, fn));
            return;
        }

        this._eventEmitter.removeListener(name, fn);
    }

    /**
     * Appends the video element to the supplied container.
     *
     * @param  {DOMElement} element
     */
    attach(element) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.attach.bind(this, element));
            return;
        }

        element.insertBefore(this._mediaPlayer.getHTMLVideoElement(), element.firstChild);
    }

    /**
     * Removes the video element from the supplied container.
     *
     * @param  {DOMElement} element
     */
    detach(element) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.detach.bind(this, element));
            return;
        }
        element.removeChild(this._mediaPlayer.getHTMLVideoElement());
    }

    /**
     * Attempts to load the pending channel or video content stream
     * into the video playback container.
     */
    load() {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.load.bind(this));
            return;
        }

        if (this._src) {
            const { playback } = this.store.getState();
            this._mediaPlayer.load(this._src);
            this._eventEmitter.emit(MediaEvent.LOADSTART);
            if (playback.autoplay) { // TODO: use state store
                this._mediaPlayer.play();
                this._eventEmitter.emit(MediaEvent.PLAY);
            }
        }
    }

    /**
     * Returns the address of the media resource.
     * Returns the empty string when there is no media resource.
     *
     * @return {String}
     */
    getSrc() {
        return this._src;
    }

    /**
     * Sets the address of the current media resource.
     *
     * @param {String} value
     */
    setSrc(src) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.setSrc.bind(this, src));
            return;
        }

        this._src = src;
        if (this._src) {
            this.load();
        }
    }

    /**
     * Returns the playback volume.
     *
     * @return {Number}
     */
    getVolume() {
        return this._mediaPlayer ? this._mediaPlayer.getVolume() : 0;
    }

    /**
     * Sets the playback volume.
     *
     * @param {Number} value
     */
    setVolume(volume) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.setVolume.bind(this, volume));
            return;
        }

        this._mediaPlayer.setVolume(volume);
        this._eventEmitter.emit(MediaEvent.VOLUME_CHANGE);
    }

    /**
     * Sets the volume muted state.
     *
     * @param {Boolean} muted
     */
    setMuted(mute) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.setMuted.bind(this, mute));
            return;
        }

        if (mute) {
            this._mediaPlayer.getHTMLVideoElement().setAttribute('muted', '');
        } else {
            this._mediaPlayer.getHTMLVideoElement().removeAttribute('muted');
        }

        this._mediaPlayer.setMuted(mute);
    }

    /**
     * Returns the muted state.
     *
     * @return {Boolean}
     */
    getMuted() {
        return this._mediaPlayer && this._mediaPlayer.isMuted();
    }

    /**
     * Sets a new Twitch channel for playback.
     *
     * @param {String} contentId
     * @param {Object} content stream data
     * @return {Promise} when src has been set
     */
    setChannel(contentId, contentStream) {
        return contentStream.streamUrl.then(streamUrl => {
            const { streamMetadata } = this.store.getState();
            if (contentId === streamMetadata.channelName) {
                this.setSrc(streamUrl);
            }
        });
    }

    /**
     * Sets the Twitch VOD ID for playback
     *
     * @param {String} contentId
     * @param {Object} content stream data
     */
    setVideo(contentId, contentStream) {
        contentStream.streamUrl.then(streamUrl => {
            const { streamMetadata } = this.store.getState();
            if (contentId === streamMetadata.videoId) {
                this.setSrc(streamUrl);
            }
        });
    }

    /**
     * Attempts to set a new preferred playback quality.
     *
     * @param {String} quality
     */
    setQuality(group) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.setQuality.bind(this, group));
            return;
        }

        if (group === 'auto') {
            this._mediaPlayer.setAutoSwitchQuality(true);
        } else {
            const quality = this._mediaPlayer.getQualities().find(q => q.group === group);
            if (typeof quality !== 'undefined') {
                this._mediaPlayer.setQuality(quality);
            }
        }
    }

    /**
     * Returns the current list of playback qualities available
     * for the current content stream.
     */
    getQualities() {
        if (this._mediaPlayer === null) {
            return [];
        }

        // Create a mapping function to create compatible quality objects
        const createCompatibleQuality = quality => Object.assign({}, quality, {
            bandwidth: quality.bitrate,
        });

        return [AUTO_QUALITY, ...this._mediaPlayer.getQualities()].map(createCompatibleQuality);
    }

    /**
     * Returns the current playback quality.
     */
    getQuality() {
        if (this._mediaPlayer === null) {
            return null;
        }

        if (this._mediaPlayer.getAutoSwitchQuality()) {
            return AUTO_QUALITY.group;
        }
        return this._mediaPlayer.getQuality().group;
    }

    elapsedTime() {
        if (this._mediaPlayer === null) {
            return 0;
        }
        return this._mediaPlayer.getStartOffset() + this._mediaPlayer.getPosition();
    }

    /**
     * Returns the current playback time of the content stream.
     *
     * @return {Number}
     */
    getCurrentTime() {
        return this._mediaPlayer ? this._mediaPlayer.getPosition() : 0;
    }

    /**
     * Sets the current playback time in the content stream.
     *
     * @param {Number} time
     */
    setCurrentTime(time) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.setCurrentTime.bind(this, time));
            return;
        }
        this._mediaPlayer.seekTo(time);
        this._eventEmitter.emit(MediaEvent.SEEKING);
    }

    /**
     * Returns the duration of the content stream.
     *
     * @return {Number}
     */
    getDuration() {
        return this._mediaPlayer ? this._mediaPlayer.getDuration() : 0;
    }

    /**
     * Returns the current client network profile.
     *
     * @return {Array}
     */
    getNetworkProfile() {
        return this._mediaPlayer ? this._mediaPlayer.getNetworkProfile() : [];
    }

    /**
     * Returns a TimeRanges object that represents the ranges of the media resource
     * that the user agent has buffered.
     *
     * @return {Object}
     */
    getBuffered() {
        if (this._mediaPlayer === null) {
            return new CustomTimeRanges();
        }
        const buffered = this._mediaPlayer.getBuffered();
        return new CustomTimeRanges(buffered.start, buffered.end);
    }

    /**
     * Returns a value that expresses the current state of video playback
     *
     * @return {Number}
     */
    getReadyState() {
        return this._readyState;
    }

    /**
     * Returns the current state of network activity for the element.
     *
     * @return {Number}
     */
    getNetworkState() {
        return this._networkState;
    }

    /**
     * Returns if the endlist has been located in the content stream.
     * Ended events are fired from EXT-X-ENDLIST tags in the HLS manifest.
     *
     * @return {Boolean}
     */
    getEnded() {
        return this._mediaPlayer && (this._mediaPlayer.getPlayerState() === this._mediaPlayerHandle.PlayerState.ENDED);
    }

    /**
     * Returns the current playback error.
     *
     * @return {String}
     */
    getError() {
        return this._errorCode;
    }

    /**
     * Returns the current backend version.
     *
     * @return {String}
     */
    getVersion() {
        return this._mediaPlayer ? this._mediaPlayer.getVersion() : '';
    }

    /**
     * Returns the current Backend identifier.
     *
     * @return {String}
     */
    getBackend() {
        return BACKEND_MEDIA_PLAYER;
    }

    /**
     * Begin or resume playback of the loaded content stream.
     */
    play() {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.play.bind(this));
            return;
        }

        this._mediaPlayer.play();
        this._eventEmitter.emit(MediaEvent.PLAY);
    }

    /**
     * Pauses playback of the content stream.
     */
    pause() {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.pause.bind(this));
            return;
        }
        this._mediaPlayer.pause();
    }

    /**
     * Returns the paused state.
     *
     * @return {Boolean}
     */
    getPaused() {
        if (this._mediaPlayer) {
            return this._mediaPlayer.getPlayerState() === this._mediaPlayerHandle.PlayerState.IDLE;
        }
        return false;
    }

    /**
     * Returns true if the user agent is currently seeking.
     *
     * @return {Boolean}
     */
    getSeeking() {
        if (this._mediaPlayer) {
            return this._mediaPlayer.isSeeking();
        }
        return false;
    }

    /**
     * Returns playback info object.
     *
     * @return {Object}
     */
    getVideoInfo() {
        const { manifestInfo, window } = this.store.getState();
        /* eslint-disable camelcase */
        const manifestInfoObject = {
            broadcast_id: parseInt(manifestInfo.broadcastId, 10),
            cluster: manifestInfo.cluster,
            manifest_cluster: manifestInfo.manifest_cluster,
            manifest_node: manifestInfo.manifest_node,
            manifest_node_type: manifestInfo.manifest_node_type,
            node: manifestInfo.node,
            serving_id: manifestInfo.serving_id,
            user_ip: manifestInfo.user_ip,
            vod_cdn_origin: manifestInfo.origin,
            vod_cdn_region: manifestInfo.region,
            stream_time_offset: parseFloat(manifestInfo.stream_time),
            segment_protocol: window.location.protocol === 'https:' ? 'https' : 'http',
        };
        /* eslint-enable camelcase */

        const mediaPlayer = this._mediaPlayer;
        let playerInfo = {};
        if (mediaPlayer) {
            const playerState = mediaPlayer.getPlayerState();
            const bandwidth = mediaPlayer.getVideoBitRate() / 1000; // kbps
            const framerate = mediaPlayer.getVideoFrameRate();
            /* eslint-disable camelcase */
            playerInfo = {
                bandwidth: bandwidth,
                current_bitrate: bandwidth,
                current_fps: round(framerate),
                current_fps_exact: framerate,
                dropped_frames: mediaPlayer.getDroppedFrames(),
                hls_latency_broadcaster: round(mediaPlayer.getBroadcasterLatency()),
                hls_latency_encoder: round(mediaPlayer.getTranscoderLatency()),
                hls_target_duration: 5,
                paused: playerState === this._mediaPlayerHandle.PlayerState.IDLE,
                playing: playerState === this._mediaPlayerHandle.PlayerState.PLAYING,
                stream_time: mediaPlayer.getPosition(),
                totalMemoryNumber: 64,
                vid_display_height: mediaPlayer.getDisplayHeight(),
                vid_display_width: mediaPlayer.getDisplayWidth(),
                vid_height: mediaPlayer.getVideoHeight(),
                vid_width: mediaPlayer.getVideoWidth(),
                video_buffer_size: mediaPlayer.getBufferDuration(),
                volume: mediaPlayer.getVolume(),
            };
            /* eslint-enable camelcase */
        }

        return Object.assign({}, DEFAULT_INFO_OBJECT, manifestInfoObject, playerInfo);
    }

    /**
     * Set playback rate of stream
     */
    setPlaybackRate(rate) {
        if (this._mediaPlayer === null) {
            this._apiCallQueue.push(this.setPlaybackRate.bind(this, rate));
        }

        this._mediaPlayer.setPlaybackRate(rate);
    }

    /**
     * Returns the playback rate of the stream.
     *
     * @return {Number}
     */
    getPlaybackRate() {
        return this._mediaPlayer ? this._mediaPlayer.getPlaybackRate() : 1.0;
    }

    /**
     * Returns the set of non-overlapping time ranges for which the video has
     * played.
     *
     * @return {TimeRanges}
     */
    getPlayed() {
        return new CustomTimeRanges();
    }

    /**
     * Returns whether or not the backend is supplying stats for reporting.
     *
     * @return {Boolean}
     */
    getStatsEnabled() {
        return false;
    }

    /**
     * Returns if ABS feature is available
     * @return {Boolean}
     */
    absAvailable() {
        return true;
    }

    /**
     * Destroy the Html5 Backend.
     */
    destroy() {
        if (this._mediaPlayer) {
            this._mediaPlayer.delete();
            this._mediaPlayer = null;
        } else {
            this._apiCallQueue.push(this.destroy.bind(this));
        }
    }

    // Internal

    _retryStreamLoad() {
        const { stream } = this.store.getState();

        this.src = '';
        this.initialize();
        stream.resetNAuthToken();

        return stream.streamUrl.then(streamUrl => {
            this.setSrc(streamUrl);
            this.play();
        });
    }

    /**
     * Called by mediaplayer when stream url hits authentication error.
     */
    _onAuthError() {
        const isOffline = this.store.getState().onlineStatus === OFFLINE_STATUS;
        if (isOffline) {
            return;
        }

        if (!this._hasRetried) {
            this._hasRetried = true;
            this._retryStreamLoad();

            this.store.dispatch(trackEvent(PLAYBACK_ERROR, {
                // eslint-disable-next-line camelcase
                playback_error_code: WARN_AUTH_ERROR_CODE,
                // eslint-disable-next-line camelcase
                playback_error_msg: PLAYER_WARN_AUTH_ERROR,
            }));
        } else {
            this.store.dispatch(setError(Errors.CODES.CONTENT_NOT_AVAILABLE));
            this.store.dispatch(trackEvent(PLAYBACK_ERROR, {
                // eslint-disable-next-line camelcase
                playback_error_code: FATAL_AUTH_ERROR_CODE,
                // eslint-disable-next-line camelcase
                playback_error_msg: PLAYER_FATAL_AUTH_ERROR,
            }));
        }
    }

    /**
     * Called by mediaplayer when stream is offline.
     */
    _onOfflineError() {
        this.store.dispatch(trackEvent(PLAYBACK_ERROR, {
            // eslint-disable-next-line camelcase
            playback_error_code: OFFLINE_ERROR_CODE,
            // eslint-disable-next-line camelcase
            playback_error_msg: PLAYER_OFFLINE_ERROR,
        }));

        this._eventEmitter.emit(OFFLINE);
        this._eventEmitter.emit(MediaEvent.ENDED);
    }

    _onCCUCapReached() {
        this._errorCode = Errors.CODES.CCU_CAP_REACHED;
        this.store.dispatch(trackEvent(PLAYBACK_ERROR, {
            // eslint-disable-next-line camelcase
            playback_error_code: Errors.CODES.CCU_CAP_REACHED,
            // eslint-disable-next-line camelcase
            playback_error_msg: PLAYER_LVS_CCU_CAP_ERROR,
        }));
        this.store.dispatch(setError(this._errorCode));
    }

    _handleNotAvailableError(code) {
        if (code === LVS_CCU_CAP_ERROR_CODE) {
            return this._onCCUCapReached();
        }

        return this._onOfflineError();
    }

    _attachInternalListeners(mediaPlayerHandle) {
        const mediaPlayer = this._mediaPlayer;

        // Player Events
        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.INITIALIZED, () => {
            this._eventEmitter.emit(PLAYER_INIT);
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerState.PLAYING, () => {
            if (this._readyState <= ReadyState.HAVE_CURRENT_DATA) {
                this._readyState = ReadyState.HAVE_FUTURE_DATA;
            }
            this._eventEmitter.emit(MediaEvent.LOADED_DATA);
            this._eventEmitter.emit(MediaEvent.PLAYING);
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.QUALITY_CHANGED, () => {
            const previousQuality = this._cache.currentQuality;
            const currentQuality = this._mediaPlayer.getQuality();
            const isAuto = this._mediaPlayer.getAutoSwitchQuality();
            this._cache.currentQuality = currentQuality;

            this._eventEmitter.emit(QUALITY_CHANGE, {
                quality: currentQuality.group,
                isAuto,
            });

            // Only report if quality actually changed. QUALITY_CHANGED
            // is emitted for the initial quality when playback starts
            if (
                isAuto &&
                previousQuality.group !== currentQuality.group
            ) {
                const bodyData = Object.assign({
                    /* eslint-disable camelcase */
                    stream_format_previous: previousQuality.group,
                    stream_format_current: currentQuality.group,
                    /* eslint-enable camelcase */
                }, this._mediaPlayer.getABSStats());
                this._eventEmitter.emit(ABS_STREAM_FORMAT_CHANGE, bodyData);
            }
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.DURATION_CHANGED, () => {
            if (this._fireLoadedMetadata) {
                this._fireLoadedMetadata = false;
                this._eventEmitter.emit(MediaEvent.LOADED_METADATA);
            }

            this._eventEmitter.emit(MediaEvent.DURATION_CHANGE);
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.PLAYBACK_RATE_CHANGED, () => {
            this._eventEmitter.emit(MediaEvent.RATE_CHANGE, this._mediaPlayer.getPlaybackRate());
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.ERROR, ({ type, source, code }) => {
            switch (type) {
            case mediaPlayerHandle.ErrorType.NOT_AVAILABLE:
                this._handleNotAvailableError(code);
                return;
            case mediaPlayerHandle.ErrorType.AUTHORIZATION:
                this._onAuthError();
                return;
            case mediaPlayerHandle.ErrorType.NOT_SUPPORTED:
                this._errorCode = Errors.CODES.FORMAT_NOT_SUPPORTED;
                break;
            case mediaPlayerHandle.ErrorType.NETWORK:
                this._errorCode = Errors.CODES.NETWORK;
                break;
            case mediaPlayerHandle.ErrorType.NETWORK_IO:
                this._errorCode = Errors.CODES.NETWORK;
                break;
            default:
                if (source === mediaPlayerHandle.ErrorSource.DECODER) {
                    this._errorCode = Errors.CODES.DECODE;
                } else {
                    // eslint-disable-next-line no-console
                    console.error('MediaPlayer failed', type);
                    this._errorCode = Errors.CODES.ABORTED;
                }
                break;
            }
            this.store.dispatch(trackEvent(PLAYBACK_ERROR, {
                // eslint-disable-next-line camelcase
                playback_error_code: FATAL_ERROR_CODE,
                // eslint-disable-next-line camelcase
                playback_error_msg: PLAYER_FATAL_ERROR,
            }));
            this.store.dispatch(setError(this._errorCode));
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.REBUFFERING, () => {
            this._readyState = ReadyState.HAVE_CURRENT_DATA;
            this._eventEmitter.emit(MediaEvent.WAITING);
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.TIME_UPDATE, () => {
            this._eventEmitter.emit(MediaEvent.TIME_UPDATE);
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.BUFFER_UPDATE, () => {
            this._eventEmitter.emit(BUFFER_CHANGE, this._mediaPlayer.getBuffered());
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.SEEK_COMPLETED, () => {
            this._eventEmitter.emit(MediaEvent.CAN_PLAY);
            this._eventEmitter.emit(MediaEvent.SEEKED);
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerEvent.TRACKING, ({ name, properties }) => {
            // Only support video_error player-core finalizes buffer_empty and other events in CVP-946
            if (name === 'video_error') {
                // eslint-disable-next-line camelcase, no-param-reassign
                properties.broadcast_id = this.store.getState().streamMetadata.broadcastID;
                // eslint-disable-next-line camelcase, no-param-reassign
                properties.manifest_broadcast_id = this.store.getState().manifestInfo.broadcast_id;
                this.store.dispatch(trackEvent(name, properties));
            }
        });

        // PlayerState events
        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerState.BUFFERING, () => {
            this._readyState = ReadyState.HAVE_CURRENT_DATA;
            this._networkState = NetworkState.NETWORK_LOADING;
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerState.IDLE, () => {
            this._readyState = ReadyState.HAVE_NOTHING;
            this._networkState = NetworkState.NETWORK_IDLE;
            this._eventEmitter.emit(MediaEvent.PAUSE);
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerState.ENDED, () => {
            this._readyState = ReadyState.HAVE_NOTHING;
            this._networkState = NetworkState.NETWORK_EMPTY;
            this._eventEmitter.emit(MediaEvent.ENDED);
        });

        mediaPlayer.addEventListener(mediaPlayerHandle.PlayerState.READY, () => {
            this._hasRetried = false;
            this._readyState = ReadyState.HAVE_METADATA;
            this._networkState = NetworkState.NETWORK_IDLE;
            const { analytics } = this.store.getState();
            this.store.dispatch(trackEvent(VIDEO_PLAY_MASTER_MANIFEST, {
                // eslint-disable-next-line camelcase
                time_since_load_start: Date.now() - analytics.playSessionStartTime,
            }));

            const normalizedManifestInfo = reduce(this._mediaPlayer.getManifestInfo(), (result, value, key) => {
                const normalizedKey = key.toLowerCase().replace(/-/g, '_');
                // eslint-disable-next-line no-param-reassign
                result[normalizedKey] = value;
                return result;
            }, {});
            this._eventEmitter.emit(MANIFEST_EXTRA_INFO, normalizedManifestInfo);
            this._eventEmitter.emit(MediaEvent.CAN_PLAY);
            this._fireLoadedMetadata = true;
            this._cache.currentQuality = mediaPlayer.getQuality();
        });

        // MetadataEvents
        mediaPlayer.addEventListener(mediaPlayerHandle.MetadataEvent.ID3, id3 => {
            const tagTOFN = find(id3, tag => tag.id === 'TOFN');
            const tagTXXX = find(id3, tag => tag.id === 'TXXX');
            if (tagTOFN) {
                // We emit the name of the segment as { name: segment } because state-tracker expects that
                // This should be removed once we move current segment to state store
                this._eventEmitter.emit(SEGMENT_CHANGE, {
                    name: tagTOFN.info[0],
                });
            }
            if (tagTXXX) {
                if (tagTXXX.desc !== 'content') {
                    const info = JSON.parse(tagTXXX.info[0]);
                    if (info.cmd === 'commercial') {
                        this._eventEmitter.emit(MIDROLL_REQUESTED, { duration: info.length });
                    }
                }
            }
        });
        mediaPlayer.addEventListener(mediaPlayerHandle.MetadataEvent.CAPTION, caption => {
            this._currentCaptionData = caption;
            this._eventEmitter.emit(CAPTION_UPDATE);
        });
        mediaPlayer.addEventListener(mediaPlayerHandle.MetadataEvent.SPLICE_OUT, attrs => {
            this._eventEmitter.emit(STITCHED_AD_START, attrs);
        });
        mediaPlayer.addEventListener(mediaPlayerHandle.MetadataEvent.SPLICE_IN, () => {
            this._eventEmitter.emit(STITCHED_AD_END);
        });
    }
}

BackendMediaPlayer.canPlay = function() {
    return PlayerCoreLoader.canLoadMediaplayer();
};

// utilities
/**
 * Custom implementation of time Ranges interface defined here
 * https://developer.mozilla.org/en-US/docs/Web/API/TimeRanges
 */
class CustomTimeRanges {
    constructor(start, end) {
        this.length = typeof start === 'undefined' ? 0 : 1;
        this._start = start;
        this._end = end;
    }

    start() {
        return this._start;
    }

    end() {
        return this._end;
    }
}
