import has from 'lodash/has';
import includes from 'lodash/includes';
import EventEmitter from 'event-emitter';
import * as Settings from '../settings';
import * as MediaEvents from './events/media-event';
import * as TwitchEvents from './events/twitch-event';

// Switches automatically between two backends automatically.
// This is used to prioritize ads and chromecast over the normal player.
// Switching is based on the `canplay` and `ended` events.
export function BackendMulti(primary, secondary) {
    var self = this;

    var events = new EventEmitter();
    var queue = [];

    var current;
    var backends = [primary, secondary];

    // Prevent the duplicate events: loadstart, loadedmetadata, loadeddata.
    var duplicateEvents = {};

    function init() {
        if (primary.getReadyState() >= primary.HAVE_CURRENT_DATA && !primary.getEnded()) {
            current = primary;
        } else {
            current = secondary;
        }

        // Switch from secondary to primary when primary starts.
        primary.addEventListener(MediaEvents.CAN_PLAY, function() {
            switchBackend(secondary, primary);
        });

        // Switch from primary to secondary when primary ends.
        primary.addEventListener(MediaEvents.ENDED, function() {
            if (current !== primary) {
                return;
            }

            switchBackend(primary, secondary);
        });

        // Forward events from both backends.
        backends.forEach(function(backend) {
            Settings.allEvents.forEach(function(name) {
                backend.addEventListener(name, function() {
                    var args = Array.prototype.slice.call(arguments);
                    queueEvent(backend, name, args);
                });
            });
        });
    }

    function queueEvent(backend, name, args) {
        // Add the event name to the args to make things easier.
        args.unshift(name);

        // Prevent firing "ended" unless the secondary is ended also.
        if (name === MediaEvents.ENDED && backend === primary && !secondary.getEnded()) {
            return;
        }

        // We can't fire both backends events at once because the UI would
        // get confused. Rather than drop the non-active backend events,
        // we queue them and fire them all on backend activation.
        if (current === backend) {
            handleEvent(args);
        } else {
            queue.push(args);
        }
    }

    function flushEvents() {
        queue.forEach(function(args) {
            handleEvent(args);
        });

        queue = [];
    }

    function handleEvent(args) {
        var name = args[0];

        // These events should only be fired once, so prevent duplicates.
        var duplicateKeys = [MediaEvents.LOADSTART];
        var duplicate = false;

        if (includes(duplicateKeys, name)) {
            duplicate = has(duplicateEvents, name);
            duplicateEvents[name] = true;
        }

        if (duplicate) {
            return;
        }

        // Clear duplicate events when we fully end.
        if (name === MediaEvents.ENDED) {
            duplicateEvents = [];
        }

        events.emit.apply(events, args);
    }

    function switchBackend(previous, next) {
        previous.pause();

        current = next;
        flushEvents();

        if (previous.getDuration() !== next.getDuration()) {
            events.emit(MediaEvents.DURATION_CHANGE);
        }

        if (previous.getCurrentTime() !== next.getCurrentTime()) {
            events.emit(MediaEvents.TIME_UPDATE);
        }

        // TODO This will always be true because buffered is an object.
        if (previous.getBuffered() !== next.getBuffered()) {
            events.emit(TwitchEvents.BUFFER_CHANGE);
        }

        if (previous.getQuality() !== next.getQuality()) {
            events.emit(TwitchEvents.QUALITY_CHANGE);
        }

        next.play();
    }

    self.destroy = function() {
        primary.destroy();
        secondary.destroy();
    };

    self.addEventListener = function(name, func) {
        events.on(name, func);
    };

    self.removeEventListener = function(name, func) {
        events.off(name, func);
    };

    self.getNetworkProfile = function() {
        return current.getNetworkProfile();
    };

    self.getError = function() {
        return current.getError();
    };

    self.getSrc = function() {
        return current.getSrc();
    };

    self.absAvailable = function() {
        return current.absAvailable();
    };

    self.setSrc = function(value) {
        backends.forEach(function(backend) {
            backend.setSrc(value);
        });
    };

    self.getNetworkState = function() {
        return current.getNetworkState();
    };

    self.getBuffered = function() {
        return current.getBuffered();
    };

    self.load = function() {
        current.load();
    };

    self.getReadyState = function() {
        return current.getReadyState();
    };

    self.getSeeking = function() {
        return current.getSeeking();
    };

    self.getStats = function() {
        return current.getStats();
    };

    self.getCurrentTime = function() {
        return current.getCurrentTime();
    };

    self.setCurrentTime = function(value) {
        backends.forEach(function(backend) {
            return backend.setCurrentTime(value);
        });
    };

    self.getDuration = function() {
        return current.getDuration();
    };

    self.getPaused = function() {
        return current.getPaused();
    };

    self.getPlaybackRate = function() {
        return current.getPlaybackRate();
    };

    self.setPlaybackRate = function(value) {
        backends.forEach(function(backend) {
            return backend.setPlaybackRate(value);
        });
    };

    self.getPlayed = function() {
        return current.getPlayed();
    };

    self.getEnded = function() {
        return current.getEnded();
    };

    self.getAutoplay = function() {
        return current.getAutoplay();
    };

    self.setLoop = function(value) {
        backends.forEach(function(backend) {
            backend.setLoop(value);
        });
    };

    self.play = function() {
        current.play();
    };

    self.pause = function() {
        current.pause();
    };

    self.getControls = function() {
        return current.getControls();
    };

    self.setControls = function(value) {
        backends.forEach(function(backend) {
            backend.setControls(value);
        });
    };

    self.getVolume = function() {
        return current.getVolume();
    };

    self.setVolume = function(value) {
        backends.forEach(function(backend) {
            backend.setVolume(value);
        });
    };

    self.getMuted = function() {
        return current.getMuted();
    };

    self.setMuted = function(value) {
        backends.forEach(function(backend) {
            backend.setMuted(value);
        });
    };

    self.getQuality = function() {
        return current.getQuality();
    };

    self.setQuality = function(value) {
        backends.forEach(function(backend) {
            backend.setQuality(value);
        });
    };

    self.getQualities = function() {
        return current.getQualities();
    };

    self.setChannel = function(value, stream = null) {
        duplicateEvents = {};

        backends.forEach(function(backend) {
            backend.setChannel(value, stream);
        });
    };

    self.elapsedTime = function() {
        return current.elapsedTime();
    };

    self.getBackend = function() {
        return current.getBackend();
    };

    self.setVideo = function(value, stream = null) {
        duplicateEvents = {};

        backends.forEach(function(backend) {
            backend.setVideo(value, stream);
        });
    };

    self.getVideoInfo = function() {
        return current.getVideoInfo();
    };

    self.getCaption = function() {
        return current.getCaption();
    };

    init();
}
