import EventEmitter from 'event-emitter';
import { QUALITY_AUTO } from '../actions/quality';
import { PLAYER_INIT, SEGMENT_CHANGE, QUALITY_CHANGE,
         CAPTION_UPDATE, MANIFEST_EXTRA_INFO, MIDROLL_REQUESTED,
         STITCHED_AD_START, STITCHED_AD_END, ABS_STREAM_FORMAT_CHANGE,
         OFFLINE } from './events/twitch-event';
import { ENDED, PAUSE, PLAYING, WAITING, ERROR, RATE_CHANGE } from './events/media-event';
import { LiveTwitchContentStream } from '../stream/twitch-live';
import { AdContentTypes } from '../actions/ads';
import { VIDEO_PLAY_MASTER_MANIFEST, VIDEO_PLAY_VARIANT_MANIFEST } from '../analytics/analytics';
import { PLAYBACK_ERROR } from '../analytics/spade-events';
import { trackEvent } from '../actions/analytics-tracker';
import Errors from 'errors';
import { setError } from 'actions/error';
import { OFFLINE_STATUS } from 'state/online-status';

import find from 'lodash/find';
import PlayerCoreLoader from 'player-core-loader';
import round from 'lodash/round';
import reduce from 'lodash/reduce';
import includes from 'lodash/includes';
import isEqual from 'lodash/isEqual';
import assign from 'lodash/assign';

export const BACKEND_PLAYER_CORE = 'player-core';
export const DEFAULT_STATS = {
    playbackRate: 0,
    fps: 0,
    bufferSize: 0,
    skippedFrames: 0,
    memoryUsage: 'Unknown',
    hlsLatencyEncoder: 0.0,
    hlsLatencyBroadcaster: 0.0,
    videoResolution: '0x0',
    displayResolution: '0x0',
    backendVersion: '',
};

export const FATAL_ERROR_CODE = 8001;
export const OFFLINE_ERROR_CODE = 8002;
export const FATAL_AUTH_ERROR_CODE = 8003;
export const WARN_AUTH_ERROR_CODE = 8004;

export const PLAYER_FATAL_ERROR = 'fatal_error';
export const PLAYER_OFFLINE_ERROR = 'offline_error';
export const PLAYER_FATAL_AUTH_ERROR = 'fatal_auth_error';
export const PLAYER_WARN_AUTH_ERROR = 'warn_auth_error';

const EXCLUDED_VIDEO_TAG_EVENTS = [
    PAUSE,
    PLAYING,
    WAITING,
    ERROR,
];

/**
 * Due to a difference in API from 1.1 to 1.2, we need to check
 * check to see if we get a string or an object. This function
 * will destructure the object and return the string, or just
 * return the string.
 *
 * @param emittedVariant (Object|String)
 * @return String
 */
export function getGroup(emittedVariant) {
    if (typeof emittedVariant === 'string') {
        return emittedVariant;
    }

    if (typeof emittedVariant === 'object' && emittedVariant.group) {
        return emittedVariant.group;
    }
    return 'Quality is unavailable';
}

export class BackendPlayerCore {
    /**
     * BackendPlayerCore Constructor
     *
     * @param {Object} analytics
     * @param {Object} options
     */
    constructor(options = {}, store) {
        this.store = store;
        this.video = document.createElement('video');
        this._autoplay = options.autoplay;
        this._hasRetried = false; // retries reloading player-core when it throws errors
        this.playerCoreLogLevel = options['cvp-log'] || 'error'; // choose between debug, warn, info, error
        this._emitVariantManifestTracking = true;

        // getEnded should return 'true' if we
        // attempt to load an offline channel.
        this.offline = false;

        if (options.playsinline) {
            this.video.setAttribute('webkit-playsinline', '');
            this.video.setAttribute('playsinline', '');
        }

        if (options.muted) {
            this.video.setAttribute('muted', '');
        }

        this.initialize();
    }

    initialize() {
        this.events = this.events instanceof EventEmitter ? this.events : new EventEmitter();
        this.currentCaptionData = {};
        this.loadPlayerCore().then(result => {
            this._initPlayerCore(result.handle, result.config);
            this._initVideoEvents();
            this.load();
        }, err => {
            this.onCoreAnalytics(err);
        });
    }

    /**
     * Attach handlers to relevant video tag events
     */
    _initVideoEvents() {
        this.video.addEventListener(PLAYING, this.onVideoTagPlaying.bind(this));
        this.video.addEventListener(PAUSE, this.onVideoTagPause.bind(this));
    }

    /**
     * Creates a new PlayerCore element to be used for playback.
     *
     * @param   {PlayerCore}
     */
    _initPlayerCore(PlayerCore, startConfig) {
        this.playerCoreEvents = PlayerCore.Event;

        this.core = new PlayerCore(assign({}, startConfig, {
            analyticsTracker: this.store.getState().analyticsTracker,
            logLevel: this.playerCoreLogLevel,
        }));
        this.core.attachMedia(this.video);
        this.core.addEventListener(this.playerCoreEvents.HLS_MASTER_PARSED, this.onHLSMasterParsed.bind(this));
        this.core.addEventListener(this.playerCoreEvents.HLS_VARIANT_PARSED, this.onHLSVariantParsed.bind(this));
        // eslint-disable-next-line max-len
        this.core.addEventListener(this.playerCoreEvents.VARIANT_SWITCH_REQUESTED, this.onVariantSwitchRequested.bind(this));

        // CVP-366: SEGMENT_CHANGED is not fired per segment, it's fired per ID3 tag
        this.core.addEventListener(this.playerCoreEvents.SEGMENT_CHANGED, this.onID3Tag.bind(this));
        this.core.addEventListener(this.playerCoreEvents.SPLICEOUT, this.onSpliceOut.bind(this));
        this.core.addEventListener(this.playerCoreEvents.SPLICEIN, this.onSpliceIn.bind(this));
        this.core.addEventListener(this.playerCoreEvents.CAPTION, this.onCaption.bind(this));
        this.core.addEventListener(this.playerCoreEvents.AUTH_ERROR, this.onAuthError.bind(this));
        this.core.addEventListener(this.playerCoreEvents.CORE_ANALYTICS, this.onCoreAnalytics.bind(this));
        this.core.addEventListener(this.playerCoreEvents.OFFLINE, this.onOfflineError.bind(this));
        this.core.addEventListener(this.playerCoreEvents.BUFFERING, this.onBuffering.bind(this));
        this.core.addEventListener(this.playerCoreEvents.FATAL_ERROR, this.onFatalError.bind(this));
        this.events.emit(PLAYER_INIT);
    }

    /**
     * Called when playercore emits a buffering event indicating the stream is buffering
     */
    onBuffering() {
        this.events.emitEvent(WAITING);
    }

    /**
     * Called when the video tag emits a 'playing' event
     */
    onVideoTagPlaying() {
        this.events.emitEvent(PLAYING);
    }

    /**
     * Called when the video tag indicates it has recently been paused.
     */
    onVideoTagPause() {
        this.events.emitEvent(PAUSE);
    }

    onFatalError({ code }) {
        this.store.dispatch(trackEvent(PLAYBACK_ERROR, {
            // eslint-disable-next-line camelcase
            playback_error_code: FATAL_ERROR_CODE,
            // eslint-disable-next-line camelcase
            playback_error_msg: PLAYER_FATAL_ERROR,
        }));

        // Mapping defined by HTML5 spec
        const ERROR_CODES_MAP = {
            1: Errors.CODES.ABORTED,
            2: Errors.CODES.NETWORK,
            3: Errors.CODES.DECODE,
            4: Errors.CODES.FORMAT_NOT_SUPPORTED,
        };

        const errorCode = ERROR_CODES_MAP[code] || Errors.CODES.ABORTED;
        this.store.dispatch(setError(errorCode));
    }

    /**
     * Called when the PlayerCore has successfully parsed the supplied HLS Manifest.
     */
    onHLSMasterParsed(manifest) {
        this._hasRetried = false;
        this._emitVariantManifestTracking = true;
        const { analytics, analyticsTracker } = this.store.getState();
        analyticsTracker.trackEvent(VIDEO_PLAY_MASTER_MANIFEST, {
            // eslint-disable-next-line camelcase
            time_since_load_start: Date.now() - analytics.playSessionStartTime,
        });

        const normalizedManifestInfo = reduce(manifest, (result, value, key) => {
            const normalizedKey = key.toLowerCase().replace(/-/g, '_');
            // eslint-disable-next-line no-param-reassign
            result[normalizedKey] = value;
            return result;
        }, {});
        this.events.emit(MANIFEST_EXTRA_INFO, normalizedManifestInfo);
        this.events.emit(QUALITY_CHANGE, {
            quality: this.getQuality(),
            isAuto: String(this.getQuality()).toLowerCase() === QUALITY_AUTO,
        });
    }

    /**
     * Called when channel is offline
     */
    onOfflineError() {
        this.store.dispatch(trackEvent(PLAYBACK_ERROR, {
            // eslint-disable-next-line camelcase
            playback_error_code: OFFLINE_ERROR_CODE,
            // eslint-disable-next-line camelcase
            playback_error_msg: PLAYER_OFFLINE_ERROR,
        }));

        this.offline = true;
        this.events.emit(OFFLINE);
        this.events.emit(ENDED);
    }

    /**
     * Called when the PlayerCore has successfully parsed the HLS Variant Playlist.
     */
    onHLSVariantParsed() {
        if (this._emitVariantManifestTracking) {
            const { analytics, analyticsTracker } = this.store.getState();
            analyticsTracker.trackEvent(VIDEO_PLAY_VARIANT_MANIFEST, {
                // eslint-disable-next-line camelcase
                time_since_load_start: Date.now() - analytics.playSessionStartTime,
            });
            this._emitVariantManifestTracking = false;
        }
    }

    /**
     * Fires an ABS_STREAM_FORMAT_CHANGE event If ABS is turned on
     * and a format change has been requested.
     *
     * @param {String} current
     * @param {String} previous
     * @param {Object} stats
     */
    _onABSFormatChange({ current, previous, stats }) {
        const qualityChange = {
            stream_format_previous: previous,   // eslint-disable-line camelcase
            stream_format_current: current,     // eslint-disable-line camelcase
        };
        const bodyData = Object.assign({}, stats, qualityChange);
        this.events.emit(ABS_STREAM_FORMAT_CHANGE, bodyData);
    }
    /**
     * Called when the PlayerCore has had a rendition switch request.
     * current, previous and stats are strings
     */
    onVariantSwitchRequested(variantFormatSwitchEvent) {
        const { current: requestedFormat, previous, stats } = variantFormatSwitchEvent;
        const currentQuality = this.getQuality();
        const autoQualityStats = (String(currentQuality).toLowerCase() === QUALITY_AUTO && previous && stats);

        if (this._onVariantSwitchComplete) {
            this.core.removeEventListener(this.playerCoreEvents.SEGMENT_CHANGED, this._onVariantSwitchComplete);
        }

        /**
         * Called when a segment changes and checks if the variant has changed
         * to the requested format
         *
         * @param {String} variant
         */
        this._onVariantSwitchComplete = ({ variant }) => {
            if (isEqual(requestedFormat, variant)) {
                this.events.emit(QUALITY_CHANGE, {
                    quality: variant,
                    isAuto: String(currentQuality).toLowerCase() === QUALITY_AUTO,
                });
                this.core.removeEventListener(this.playerCoreEvents.SEGMENT_CHANGED, this._onVariantSwitchComplete);
                this._onVariantSwitchComplete = false;
            }
        };
        this.core.addEventListener(this.playerCoreEvents.SEGMENT_CHANGED, this._onVariantSwitchComplete);

        if (autoQualityStats) {
            this._onABSFormatChange(variantFormatSwitchEvent);
        }
    }

    /**
     * Called on every ID3 tag parsed
     */
    // Can't make the complexity less gross unless we get "content" in segmentmetadata
    onID3Tag(eventData) {
        const tagTOFN = find(eventData.ID3, tag => tag.id === 'TOFN');
        const tagTXXX = find(eventData.ID3, tag => tag.id === 'TXXX');
        if (tagTOFN) {
            // We emit the name of the segment as { name: segment } because state-tracker expects that
            // This should be removed once we move current segment to state store
            const data = {
                name: tagTOFN.info[0],
            };
            this.events.emit(SEGMENT_CHANGE, data);
        }
        if (tagTXXX) {
            if (tagTXXX.desc !== 'content') {
                const info = JSON.parse(tagTXXX.info[0]);
                if (info.cmd === 'commercial') {
                    this.events.emit(MIDROLL_REQUESTED, { duration: info.length });
                }
            }
        }
    }

    // called when a stitched ad starts
    onSpliceOut(data) {
        this.events.emit(STITCHED_AD_START, data);
    }

    // called when a stitched ad ends
    onSpliceIn() {
        this.events.emit(STITCHED_AD_END);
    }

    getStats() {
        const stats = this.core ? this.core.getPlaybackStatistics() : DEFAULT_STATS;

        const roundedStats = {
            playbackRate: round(stats.playbackRate, 2) || 0.0,
            fps: round(stats.fps) || 0.0,
            bufferSize: round(stats.bufferSize),
            skippedFrames: stats.skippedFrames,
            memoryUsage: `${stats.memoryUsage} MB`,
            hlsLatencyEncoder: round(stats.hlsLatencyEncoder / 1000) || 0.0,
            hlsLatencyBroadcaster: round(stats.hlsLatencyBroadcaster / 1000) || 0.0,
            videoResolution: stats.videoResolution,
            displayResolution: stats.displayResolution,
            backendVersion: this.getVersion(),
        };
        return roundedStats;
    }

    /**
     * Called on every closed caption update.
     */
    onCaption(data) {
        this.currentCaptionData = data;
        this.events.emit(CAPTION_UPDATE);
    }

    /**
     * Called to fetch current caption data.
     */
    getCaption() {
        return this.currentCaptionData;
    }

    /**
     * Called on player-core authentication error.
     */
    onAuthError() {
        const isOffline = this.store.getState().onlineStatus === OFFLINE_STATUS;
        if (isOffline) {
            return;
        }

        if (!this._hasRetried) {
            this._hasRetried = true;
            this._retryStreamLoad();

            this.store.dispatch(trackEvent(PLAYBACK_ERROR, {
                // eslint-disable-next-line camelcase
                playback_error_code: WARN_AUTH_ERROR_CODE,
                // eslint-disable-next-line camelcase
                playback_error_msg: PLAYER_WARN_AUTH_ERROR,
            }));
        } else {
            // We have already retried, so display `Content not available`
            this.store.dispatch(setError(Errors.CODES.CONTENT_NOT_AVAILABLE));

            this.store.dispatch(trackEvent(PLAYBACK_ERROR, {
                // eslint-disable-next-line camelcase
                playback_error_code: FATAL_AUTH_ERROR_CODE,
                // eslint-disable-next-line camelcase
                playback_error_msg: PLAYER_FATAL_AUTH_ERROR,
            }));
        }
    }

    _retryStreamLoad() {
        const { stream } = this.store.getState();

        this.destroy();
        delete this.src;
        this.initialize();
        stream.resetNAuthToken();

        return stream.streamUrl.then(streamUrl => {
            this.setSrc(streamUrl);
            this.core.play();
        });
    }

    /**
     * Called on video player core's CORE_ANALYTICS event
     */
    onCoreAnalytics(payload) {
        const tracker = this.store.getState().analyticsTracker;
        tracker.trackEvent(payload.spadeEventName, payload.spadeEventData);
    }

    /**
     * Creates a new VideoEvents listener.
     *
     * @param {String} name
     * @param {function} callback
     */
    addEventListener(name, callback) {
        this.events.on(name, callback);

        if (!includes(EXCLUDED_VIDEO_TAG_EVENTS, name)) {
            this.video.addEventListener(name, callback);
        }
    }

    /**
     * Removes a VideoEvents listener.
     *
     * @param {String} name
     * @param {function} callback
     */
    removeEventListener(name, callback) {
        this.events.off(name, callback);
        this.video.removeEventListener(name, callback);
    }

    /**
     * Appends the video element to the supplied container.
     *
     * @param  {DOMElement} element
     */
    attach(element) {
        element.appendChild(this.video);
    }

    /**
     * Removes the video element from the supplied container.
     *
     * @param  {DOMElement} element
     */
    detach(element) {
        element.removeChild(this.video);
    }

    /**
     * Attempts to load the pending channel or video content stream
     * into the video playback container.
     */
    load() {
        if (this.core && this.src) {
            this.offline = false;
            this.core.loadURL(this.src);

            if (this._autoplay) {
                this.core.play();
            }
        }
    }

    /**
     * Returns the address of the media resource.
     * Returns the empty string when there is no media resource.
     *
     * @return {String}
     */
    getSrc() {
        return this.src;
    }

    /**
     * Sets the address of the current media resource.
     *
     * @param {String} value
     */
    setSrc(value) {
        this.src = value;

        if (this.src) {
            this.load();
        }
    }

    /**
     * Returns the playback volume.
     *
     * @return {Number}
     */
    getVolume() {
        return this.video.volume;
    }

    /**
     * Sets the playback volume.
     *
     * @param {Number} value
     */
    setVolume(value) {
        this.video.volume = value;
    }

    /**
     * Sets the volume muted state.
     *
     * @param {Boolean} muted
     */
    setMuted(value) {
        if (value) {
            this.video.setAttribute('muted', '');
        } else {
            this.video.removeAttribute('muted');
        }

        this.video.muted = value;
    }

    /**
     * Returns the muted state.
     *
     * @return {Boolean}
     */
    getMuted() {
        return this.video.muted;
    }

    /**
     * Sets a new Twitch channel for playback.
     *
     * @param {String} contentId
     * @param {Object} content stream data
     * @return {Promise} when src has been set
     */
    setChannel(contentId, contentStream) {
        return contentStream.streamUrl.then(streamUrl => {
            const { streamMetadata } = this.store.getState();
            if (contentId === streamMetadata.channelName) {
                this.setSrc(streamUrl);
            }
        });
    }

    /**
     * Sets the Twitch VOD ID for playback
     *
     * @param {String} contentId
     * @param {Object} content stream data
     */
    setVideo(contentId, contentStream) {
        contentStream.streamUrl.then(streamUrl => {
            const { streamMetadata } = this.store.getState();
            if (contentId === streamMetadata.videoId) {
                this.setSrc(streamUrl);
            }
        });
    }

    /**
     * Attempts to set a new preferred playback quality.
     *
     * @param {String} quality
     */
    setQuality(value) {
        if (this.core) {
            this.core.setQuality(value);
        }
    }

    /**
     * Returns the current list of playback qualities available
     * for the current content stream.
     */
    getQualities() {
        if (this.core) {
            return this.core.getQualities();
        }
        return [];
    }

    /**
     * Returns the current playback quality.
     */
    getQuality() {
        if (this.core) {
            return this.core.getQuality();
        }
        return '';
    }

    elapsedTime() {
        return this.core ? this.core.elapsedTime() : 0;
    }

    /**
     * Returns the current playback time of the content stream.
     *
     * @return {Number}
     */
    getCurrentTime() {
        return this.video.currentTime;
    }

    /**
     * Sets the current playback time in the content stream.
     *
     * @param {Number} time
     */
    setCurrentTime(time) {
        this.video.currentTime = time;
    }

    /**
     * Returns the duration of the content stream.
     *
     * @return {Number}
     */
    getDuration() {
        return this.video.duration || 0;
    }

    /**
     * Returns the current client network profile.
     *
     * @return {Array}
     */
    getNetworkProfile() {
        return this.core ? this.core.getNetworkProfile() : [];
    }

    /**
     * Returns a TimeRanges object that represents the ranges of the media resource
     * that the user agent has buffered.
     *
     * @return {Object}
     */
    getBuffered() {
        return this.video.buffered;
    }

    /**
     * Returns a value that expresses the current state of the element with respect
     * to rendering the current playback position.
     *
     * @return {Number}
     */
    getReadyState() {
        return this.video.readyState;
    }

    /**
     * Returns the current state of network activity for the element.
     *
     * @return {Number}
     */
    getNetworkState() {
        return this.video.networkState;
    }

    /**
     * Returns if the endlist has been located in the content stream.
     * Ended events are fired from EXT-X-ENDLIST tags in the HLS manifest.
     *
     * @return {Boolean}
     */
    getEnded() {
        // If we attempt load an offline channel,
        // we consider the stream to be ended.
        return this.offline || (this.core && this.core.ended());
    }

    /**
     * Returns the current playback error.
     *
     * @return {String}
     */
    getError() {
        return this.video.error;
    }

    /**
     * Returns the current backend version.
     *
     * @return {String}
     */
    getVersion() {
        if (this.core) {
            return this.core.getVersion();
        }
        return '';
    }

    /**
     * Returns the current Backend identifier.
     *
     * @return {String}
     */
    getBackend() {
        return BACKEND_PLAYER_CORE;
    }

    /**
     * Begin or resume playback of the loaded content stream.
     */
    play() {
        const { ads, stream } = this.store.getState();
        const isContentTypeEqual = ads.currentMetadata.contentType === AdContentTypes.STITCHED;
        const isInstanceOfLiveTwitchContentStream = stream instanceof LiveTwitchContentStream;
        if (!isContentTypeEqual && isInstanceOfLiveTwitchContentStream) {
            this.setChannel(stream.channel, stream).then(() => {
                this.core.play();
            });
        } else {
            this.core.play();
        }
    }

    /**
     * Pauses playback of the content stream.
     */
    pause() {
        this.core.pause();
        const { stream } = this.store.getState();
        if (stream instanceof LiveTwitchContentStream) {
            this.core.stop();
        }
    }

    /**
     * Returns the paused state.
     *
     * @return {Boolean}
     */
    getPaused() {
        // HTMLVideoElement.paused is unreliable after we start,
        // but default to it if PlayerCore hasn't been initialized
        return this.core ? this.core.paused() : this.video.paused;
    }

    /**
     * Returns true if the user agent is currently seeking.
     *
     * @return {Boolean}
     */
    getSeeking() {
        return this.video.seeking;
    }

    /**
     * Returns playback info object.
     *
     * @return {Object}
     */
    getVideoInfo() {
        // Certain tracking events may call `getVideoInfo` before the core is initialized :(
        const coreInfo = this.core ? this.core.getVideoInfo() : {};

        const { manifestInfo, window: windowObj } = this.store.getState();

        /* eslint-disable camelcase */
        // The commented out properties in the object below are ones also given out by the flash backend,
        // but without an equivalent way to get the values in html5. They are included for posterity,
        // but not to be uncommented unless we have found a way to get the value for them.

        const videoInfo = {
            bandwidth: coreInfo.playbackRate,
            broadcast_id: parseInt(coreInfo.broadcastId, 10),
            cluster: manifestInfo.cluster,
            current_bitrate: coreInfo.playbackRate,
            current_fps: round(coreInfo.fps),
            current_fps_exact: coreInfo.fps,
            // deblocking: true,
            dropped_frames: coreInfo.skippedFrames,
            // hls_buffer_duration: 3000,
            hls_latency_broadcaster: round(coreInfo.hlsLatencyBroadcaster),
            hls_latency_encoder: round(coreInfo.hlsLatencyEncoder),
            hls_target_duration: coreInfo.targetDuration,
            manifest_cluster: manifestInfo.manifest_cluster,
            manifest_node: manifestInfo.manifest_node,
            manifest_node_type: manifestInfo.manifest_node_type,
            node: manifestInfo.node,
            paused: this.getPaused(),
            // playback_bytes_per_second:72654.1562413122
            playing: !this.getPaused(),
            segment_protocol: windowObj.location.protocol === 'https:' ? 'https' : 'http',
            serving_id: manifestInfo.serving_id,
            // smoothing:true
            // stageHeight:267
            // stageWidth:1680
            // stream_protocol:"hls"
            stream_time: this.video.currentTime,
            stream_time_offset: parseFloat(manifestInfo.stream_time),
            totalMemoryNumber: coreInfo.memoryUsage,
            user_ip: manifestInfo.user_ip,
            vid_display_height: coreInfo.displayHeight,
            vid_display_width: coreInfo.displayWidth,
            vid_height: coreInfo.videoHeight,
            vid_width: coreInfo.videoWidth,
            video_buffer_size: coreInfo.bufferSize,
            volume: this.getVolume(),
            vod_cdn_origin: manifestInfo.origin,
            vod_cdn_region: manifestInfo.region,
        };
        /* eslint-enable camelcase */

        return videoInfo;
    }

    /**
     * Returns the set of non-overlapping time ranges for which the video has
     * played.
     *
     * @return {TimeRanges}
     */
    getPlayed() {
        return this.video.played;
    }

    /**
     * Returns whether or not the backend is supplying stats for reporting.
     *
     * @return {Boolean}
     */
    getStatsEnabled() {
        return false;
    }

    /**
     * Sets the playback rate if the core is available
     *
     * @param {Number} playbackRate
     */
    setPlaybackRate(playbackRate) {
        if (this.core) {
            this.core.setPlaybackRate(playbackRate);
            this.events.emitEvent(RATE_CHANGE);
        }
    }

    /**
     * Gets the current playback rate
     *
     * @return {Number}
     */
    getPlaybackRate() {
        return this.core ? this.core.getPlaybackRate() : 1.0;
    }

    absAvailable() {
        if (this.core && this.core.absAvailable) {
            return this.core.absAvailable();
        }
        return false;
    }

    /**
     * Destroy the Html5 Backend.
     */
    destroy() {
        // We call core.pause here because if we don't, the video will continue
        // to play until the buffer runs out. When `destroy` is called, we assume we want
        // playback to stop immediately.
        if (this.core) {
            this.core.pause();
            this.core.destroy();
        }
        this.store.getState().window.clearInterval(this.statsUpdateLoop);
    }

    /**
     * Load the PlayerCore lib asynchronously.
     * @return {Promise}
     */
    loadPlayerCore() {
        return PlayerCoreLoader.load({
            value: '1.8.13',
        });
    }
}

BackendPlayerCore.canPlay = function() {
    return PlayerCoreLoader.canLoad();
};
