import { BackendFlash, BACKEND_FLASH } from 'backend/flash';
import { BackendMediaPlayer, BACKEND_MEDIA_PLAYER } from 'backend/mediaplayer';
import { BackendPlayerCore, BACKEND_PLAYER_CORE } from 'backend/player-core';
import { BackendHls, BACKEND_HLS } from 'backend/hls';
import { BackendClip, BACKEND_CLIP } from 'backend/clip';
import { BackendVideoTag, BACKEND_VIDEO_TAG } from 'backend/video-tag';
import { BackendBlank, BACKEND_BLANK_TYPE } from 'backend/blank';
import { CLIPS_PLAYER_TYPES, PLAYER_AMAZON_LIVE, PLAYER_AMAZON_VSE } from 'util/player-type';
import intersection from 'lodash/intersection';
import includes from 'lodash/includes';

// Map of backend names to the class reference.
export const BACKEND_MAP = Object.freeze({
    [BACKEND_FLASH]: BackendFlash,
    [BACKEND_HLS]: BackendHls,
    [BACKEND_PLAYER_CORE]: BackendPlayerCore,
    [BACKEND_MEDIA_PLAYER]: BackendMediaPlayer,
    [BACKEND_CLIP]: BackendClip,
    [BACKEND_BLANK_TYPE]: BackendBlank,
    [BACKEND_VIDEO_TAG]: BackendVideoTag,
});

// Names of the available backends, ordered by priority.
const ALL_BACKENDS_PRIORITIZED = Object.freeze([
    BACKEND_MEDIA_PLAYER,
    BACKEND_PLAYER_CORE,
    BACKEND_HLS,
    BACKEND_FLASH,
    BACKEND_CLIP,
    BACKEND_VIDEO_TAG,
]);

// Player Type Allowed Backends
const CLIPS_PLAYER_BACKENDS = Object.freeze([
    BACKEND_CLIP,
]);
const AMAZON_LIVE_PLAYER_BACKENDS = Object.freeze([
    BACKEND_PLAYER_CORE,
    BACKEND_HLS,
    BACKEND_MEDIA_PLAYER,
]);
const AMAZON_VSE_PLAYER_BACKENDS = Object.freeze([
    BACKEND_VIDEO_TAG,
]);
const DEFAULT_PLAYER_BACKENDS = Object.freeze([
    BACKEND_PLAYER_CORE,
    BACKEND_HLS,
    BACKEND_FLASH,
    BACKEND_MEDIA_PLAYER,
]);

export function getAvailableBackends(playerType) {
    const browserSupportedBackends = ALL_BACKENDS_PRIORITIZED.filter(name => {
        return BACKEND_MAP[name].canPlay();
    });
    const playerTypeAllowedBackends = getPlayerTypeAllowedBackends(playerType);
    // The order and references of result values are determined by first array
    const availableBackends = intersection(
        browserSupportedBackends,
        playerTypeAllowedBackends
    );

    return availableBackends;
}

// Return an array of supported backends.
function getPlayerTypeAllowedBackends(playerType) {
    if (includes(CLIPS_PLAYER_TYPES, playerType)) {
        return CLIPS_PLAYER_BACKENDS;
    }

    if (playerType === PLAYER_AMAZON_LIVE) {
        return AMAZON_LIVE_PLAYER_BACKENDS;
    }

    if (playerType === PLAYER_AMAZON_VSE) {
        return AMAZON_VSE_PLAYER_BACKENDS;
    }

    return DEFAULT_PLAYER_BACKENDS;
}

