import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { BACKEND_PLAYER_CORE } from 'backend/player-core';
import { BACKEND_MEDIA_PLAYER } from 'backend/mediaplayer';
import { cookie } from 'cookie_js';
import { PLAYER_CURSE } from 'util/player-type';
import { createClip } from 'api';

export const PRODUCTION_CLIPS_URL = 'https://clips.twitch.tv';
export const STAGING_CLIPS_URL = 'https://clips-staging.twitch.tv';
export const ALPHA_CLIPS_URL = 'https://clips-alpha.twitch.tv'; // Staging environment that auths against prod

export const CLIPS_URL_COOKIE_KEY = 'clips-staging';

// ! This is only used for non player-core backends.
// Due to stream_time's inaccuracy, we subtract by an arbitrary number
// to roughly get the resulting Clip at the right spot.
export const CLIPS_FIXED_OFFSET = 8;

export function getClipsUrl(hostUrl) {
    if (/\/\/(www|player|go|canary|rc|.*\.rc|.*\.canary)\.twitch\.tv/.test(hostUrl)) {
        return PRODUCTION_CLIPS_URL;
    }

    if (/\/\/betaplayer\.twitch\.tv/.test(hostUrl)) {
        return ALPHA_CLIPS_URL;
    }

    const customStagingUrl = cookie.get(CLIPS_URL_COOKIE_KEY);
    if (customStagingUrl) {
        return customStagingUrl;
    }

    return STAGING_CLIPS_URL;
}

export class ClipGenerator {
    /**
    * @param {State} state Legacy file state.
    * @param {StateStore} store store.
    * @param {function} urlOpener Function that handles opening a new browsing
    *   window.
    */
    constructor(state, store) {
        this._state = state;
        this._store = store;

        const hostUrl = this._store.getState().window.location.origin;
        this._clipsUrl = getClipsUrl(hostUrl);
    }

    hookUpForm(root) {
        this._$createClipPlayerBackendType = root.getElementsByClassName('js-create-clip-player-backend-type')[0];
        this._$createClipChannel = root.getElementsByClassName('js-create-clip-channel')[0];
        this._$createClipSegment = root.getElementsByClassName('js-create-clip-segment')[0];
        this._$createClipOffset = root.getElementsByClassName('js-create-clip-offset')[0];
        this._$createClipBroadcastId = root.getElementsByClassName('js-create-clip-broadcast_id')[0];
        this._$createClipVodId = root.getElementsByClassName('js-create-clip-vod_id')[0];
        this._$createClipPlaySessionId = root.getElementsByClassName('js-create-clip-play_session_id')[0];
        this._$createClipForm = root.getElementsByClassName('js-create-clip-form')[0];
        this._$createClipForm.setAttribute('action', `${this._clipsUrl}/clips`);
    }

    recordClip() {
        const { stream, streamMetadata } = this._store.getState();
        const contentType = stream.contentType;
        const name = streamMetadata.channel.name;

        if (contentType === CONTENT_MODE_VOD) {
            this._recordVodClip(name);
        } else if (contentType === CONTENT_MODE_LIVE) {
            this._recordLiveClip(name);
        }
    }

    _recordVodClip(channelName) {
        const { stream, analytics, backend } = this._store.getState();
        const { videoId } = stream;
        const { playSessionId } = analytics;

        const clipValues = {
            playerBackendType: backend.getBackend(),
            channelName,
            offset: Math.round(this._state.currentTime),
            broadcastId: '',
            vodId: videoId.slice(1),
            playSessionId,
        };

        if (this._store.getState().env.playerType === PLAYER_CURSE) {
            this._createWithAPI(clipValues);
        } else {
            this._submitFormWithValues(clipValues);
        }
    }

    _recordLiveClip(channelName) {
        const { backendInfo, manifestInfo, streamMetadata, analytics, backend } = this._store.getState();
        const { playerBackendType } = backendInfo;
        const { broadcastID } = streamMetadata;
        const { playSessionId } = analytics;

        let offset;
        if (playerBackendType === BACKEND_PLAYER_CORE || playerBackendType === BACKEND_MEDIA_PLAYER) {
            offset = backend.elapsedTime();
        } else {
            const streamTime = manifestInfo.stream_time;
            offset = streamTime + this._state.currentTime - CLIPS_FIXED_OFFSET;
        }

        const clipValues = {
            playerBackendType,
            channelName,
            offset: Math.round(Math.max(offset, 0)),
            broadcastId: broadcastID,
            vodId: '',
            playSessionId,
        };

        if (this._store.getState().env.playerType === PLAYER_CURSE) {
            this._createWithAPI(clipValues);
        } else {
            this._submitFormWithValues(clipValues);
        }
    }

    _submitFormWithValues({ playerBackendType, channelName, offset, broadcastId, vodId, playSessionId }) {
        const windowName = `twitch_clips_${Date.now().toString()}`;

        this._$createClipPlayerBackendType.value = playerBackendType;
        this._$createClipChannel.value = channelName;
        this._$createClipOffset.value = offset;
        this._$createClipBroadcastId.value = broadcastId;
        this._$createClipVodId.value = vodId;
        this._$createClipPlaySessionId.value = playSessionId;
        this._$createClipForm.setAttribute('target', windowName);

        this._store.getState().window.open('', windowName);
        this._submitForm();
    }

    // Moved into separate function for easier stubbing during tests
    _submitForm() {
        this._$createClipForm.submit();
    }

    _createWithAPI({ playerBackendType, channelName, offset, broadcastId, vodId, playSessionId }) {
        /* eslint-disable camelcase */
        const urlParams = {
            player_backend_type: playerBackendType,
            channel: channelName,
            offset,
            broadcast_id: broadcastId,
            vod_id: vodId,
            play_session_id: playSessionId,
        };
        /* eslint-enable camelcase */

        createClip(`${this._clipsUrl}/api/v2/clips`, urlParams).then(resp => {
            window.open(`${resp.url}/edit`, '_blank');
        });
    }
}
