import assign from 'lodash/assign';
import { collectionInfo } from './api';
import { TRANSITION_TYPE_COLLECTION } from './state/playback';
import { CONTENT_MODE_VOD } from './stream/twitch-vod';
import { subscribe } from './util/subscribe';
import findIndex from 'lodash/findIndex';
import { setCollection, loadedCollectionItem, loadedLastCollectionItem } from './actions/collection';
import { selectCollectionVideo, setStartTime } from 'actions/playback';
import { pushScreen, COLLECTION_EMPTY_SCREEN } from 'actions/screen';
import { pause } from 'actions/video-api';
import { TYPE_VIDEO, setStream } from 'actions/stream';

export class CollectionManager {
    constructor(store) {
        this._store = store;
        this.unsubs = [];
        this.unsubs.push(subscribe(this._store, ['collection.pendingRequest'], this.onPendingRequestChange.bind(this)));
        this.unsubs.push(subscribe(this._store, ['collection.id', 'stream'], this.onStreamChange.bind(this)));
        this.unsubs.push(subscribe(this._store, ['playback.ended'], this._streamEnded.bind(this)));
    }

    /**
     * Fetch data from the two collection endpoints for metadata and the items
     * and merges the data into one collection object (pre-normalized)
     *
     * @param {String} id of the collection
     * @return {Promise<Object>} promise resolves to the collection object including items in the collection
     */
    _fetchCollectionInfo(collectionId) {
        return collectionInfo(collectionId).then(([metadataData, itemsData]) => {
            const collectionWithItems = assign({}, metadataData, {
                items: itemsData.items,
            });
            return collectionWithItems;
        });
    }

    /**
     * Helper that checks membership of an item in a list of collection items
     *
     * @param {Array<Object>} list of raw collection items
     * @param {String} id of the item to check
     * @return {Boolean} whether the item is the list of items
     */
    _collectionHasVideo(collectionItems, itemId) {
        return collectionItems.some(item => itemId === `v${item.item_id}`);
    }

    _loadVideo(videoId, timestamp = '') {
        this._store.dispatch(setStream({
            contentType: TYPE_VIDEO,
            contentId: videoId,
        }));
        this._store.dispatch(setStartTime(timestamp));
    }

    /**
    * Fired when the pending request for a collection changes, handles attempts to activate
    * a collection via `player.setCollection` (and when the request is cleared). Sets an
    * appropriate video stream before storing the collection in the store.
    */
    onPendingRequestChange({ collection }) {
        const { collectionId, videoId, timestamp, preferVideo } = collection.pendingRequest;
        if (!collectionId) {
            return;
        }

        return this._fetchCollectionInfo(collectionId).
            then(collectionData => {
                if (collectionData.items.length === 0) {
                    this._store.dispatch(setCollection(collectionData));
                    return this._showEmptyCollectionOverlay(collectionData);
                }

                const videoInCollection = this._collectionHasVideo(collectionData.items, videoId);
                const firstVideoId = `v${collectionData.items[0].item_id}`;

                if (videoInCollection) {
                    this._store.dispatch(setCollection(collectionData));
                    return this._loadVideo(videoId, timestamp);
                }

                if (videoId === '' || (!preferVideo && videoId !== '')) {
                    this._store.dispatch(setCollection(collectionData));
                    return this._loadVideo(firstVideoId, timestamp);
                }

                if (preferVideo && videoId !== '') {
                    return this._loadVideo(videoId, timestamp);
                }
            }).catch(() => {
                if (videoId) {
                    this._loadVideo(videoId, timestamp);
                }
            });
    }

    /**
    * Fired when the stream changes, if user is watching a collection and the new stream is the last item,
    * changes the transition scheme to in-player recommendations.
    * If user had just navigated away from the last item, reset the transition scheme back to collection.
    */
    onStreamChange() {
        const { stream, collection } = this._store.getState();
        if (
            !collection.id ||
            collection.items.length === 0
        ) {
            return;
        }

        const lastCollectionItem = collection.items[collection.items.length - 1];
        if (
            stream.contentType === CONTENT_MODE_VOD &&
            stream.videoId === `v${lastCollectionItem.itemId}`
        ) {
            this._store.dispatch(loadedLastCollectionItem());
        } else {
            this._store.dispatch(loadedCollectionItem());
        }
    }

    /**
     * Helper that shows the empty collection overlay when there are no videos in a given collection
     */
    _showEmptyCollectionOverlay() {
        this._store.dispatch(pushScreen(COLLECTION_EMPTY_SCREEN));
        // previous content stream should stop when overlay is shown
        this._store.dispatch(pause());
    }

    /**
     * Fired when a stream ends and plays the next item in a collection if needed.
     */
    _streamEnded() {
        const { playback, stream, ui } = this._store.getState();

        if (
            !ui.isMini &&
            playback.ended &&
            playback.transitionScheme === TRANSITION_TYPE_COLLECTION &&
            stream.contentType === CONTENT_MODE_VOD
        ) {
            this._playNextVideo();
        }
    }

    /**
     * Helper that sets the stream to the next item in a collection.
     */
    _playNextVideo() {
        const { stream, collection, window: windowObj } = this._store.getState();

        if (collection.items.length === 0) {
            return;
        }

        const currentVodID = stream.videoId;
        const currentVodIndex = findIndex(collection.items, v => `v${v.itemId}` === currentVodID);

        if (currentVodIndex + 1 === collection.items.length) {
            // TODO: This should eventually reactivate in-player-recs, but we need to seperate ACTION_ENDED
            // from something like LAUNCH_PLAYER_RECS, which is a lot more work, and not a requirement
            // for Collection staff beta JIRA:  https://twitchtv.atlassian.net/browse/VOD-1328
            return;
        }

        const nextVod = collection.items[currentVodIndex + 1];

        windowObj.setTimeout(() => {
            this._store.dispatch(selectCollectionVideo(`v${nextVod.itemId}`, collection.id));
        }, 0);
    }

    destroy() {
        this.unsubs.forEach(unsub => unsub());
    }
}
