import isFinite from 'lodash/isFinite';
import omit from 'lodash/omit';
import assign from 'lodash/assign';
import EventEmitter from 'event-emitter';
import { parseUri } from '../util/parseuri';
import { toString as paramsToString } from '../util/params';
import { Promise } from 'es6-promise';

const PLAYER_HOST = (function() {
    let src = 'https://player.twitch.tv';
    if (document.currentScript) {
        src = document.currentScript.src;
    } else {
        const scripts = Array.prototype.filter.call(
            document.scripts,
            s => /twitch\.tv.*embed/.test(s.src)
        );
        src = (scripts.length > 0 ?
            scripts[0].src :
            document.scripts[document.scripts.length - 1].src);
    }
    const uri = parseUri(src);

    if (uri.host.toLowerCase() === 'testplayer.twitch.tv') {
        const testPlayerBranchName = uri.path.split('/')[1].replace(/[^a-zA-Z0-9_-]+/g, '');
        return `${uri.protocol}://${uri.authority}/${testPlayerBranchName}`;
    }

    return `${uri.protocol}://${uri.authority}`;
}());

const HOST_WAIT_TIMEOUT = 15000; // milliseconds

// Default state for Embed
const DEFAULT_EMBED_STATE = Object.freeze({
    channelName: '',
    currentTime: 0,
    duration: 0,
    muted: false,
    playback: '',
    quality: '',
    qualitiesAvailable: [],
    stats: {},
    videoID: '',
    viewers: 0,
    volume: 0,
});

const DEFAULT_OPTIONS = Object.freeze({
    height: 390,
    width: 640,
    allowfullscreen: false,
});

export const EVENT_EMBED_READY = 'ready';
export const EVENT_EMBED_PLAY = 'play';
export const EVENT_EMBED_PAUSE = 'pause';
export const EVENT_EMBED_ENDED = 'ended';
export const EVENT_EMBED_ONLINE = 'online';
export const EVENT_EMBED_OFFLINE = 'offline';
export const EVENT_EMBED_WHEEL = 'wheel';
export const EVENT_EMBED_VIEWERS_CHANGE = 'viewerschange';
export const EVENT_THEATRE_ENTERED = 'theatreentered';
export const EVENT_THEATRE_EXITED = 'theatreexited';
export const EVENT_FULLSCREEN_ENTERED = 'fullscreenentered';
export const EVENT_FULLSCREEN_EXITED = 'fullscreenexited';
export const EVENT_EMBED_AD_COMPANION_RENDERED = 'adcompanionrendered';
export const EVENT_EMBED_CONTENT_SHOWING = 'contentShowing';
export const EVENT_EMBED_FLASH_AD_COMPANION_RENDERED = 'flashAdcompanionrendered';
export const EVENT_EMBED_OPEN_STREAM = 'openStream';
export const EVENT_EMBED_TRANSITION_TO_COLLECTION_VOD = 'transitionToCollectionVod';
export const EVENT_EMBED_TRANSITION_TO_REC_VOD = 'transitionToRecommendedVod';
export const EVENT_EMBED_STATS_UPDATE = 'statsupdate';
export const EVENT_EMBED_TIME_UPDATE = 'timeUpdate';
export const EVENT_EMBED_SEEKED = 'seeked';

// First Party Analytic Events
export const EVENT_EMBED_MINUTE_WATCHED = 'minuteWatched';
export const EVENT_EMBED_VIDEO_PLAY = 'videoPlay';
export const EVENT_EMBED_BUFFER_EMPTY = 'bufferEmpty';
export const EVENT_EMBED_VIDEO_PLAYBACK_ERROR = 'videoError';

// Embed Error Event
export const EVENT_EMBED_ERROR = 'error';

// Available Bridge this._player Method Calls
export const METHOD_PLAY = 'play';
export const METHOD_PAUSE = 'pause';
export const METHOD_SET_CHANNEL = 'channel';
export const METHOD_SET_CHANNEL_ID = 'channelId';
export const METHOD_SET_VIDEO = 'video';
export const METHOD_SET_COLLECTION = 'collection';
export const METHOD_SEEK = 'seek';
export const METHOD_SET_QUALITY = 'quality';
export const METHOD_SET_MUTE = 'mute';
export const METHOD_SET_VOLUME = 'volume';
export const METHOD_DESTROY = 'destroy';
export const METHOD_SET_THEATRE = 'theatre';
export const METHOD_SET_FULLSCREEN = 'fullscreen';
export const METHOD_SET_MINI_PLAYER_MODE = 'setminiplayermode';
export const METHOD_SET_CONTENT = 'setcontent';
export const METHOD_SET_CLIP = 'setclip';
export const METHOD_SET_VIDEO_SOURCE = 'setvideosource';
export const METHOD_SET_TRACKING_PROPERTIES = 'setTrackingProperties';
export const METHOD_SET_PLAYER_TYPE = 'setPlayerType';
export const METHOD_ENABLE_CAPTIONS = 'enableCaptions';
export const METHOD_DISABLE_CAPTIONS = 'disableCaptions';
export const METHOD_SET_CAPTION_SIZE = 'setCaptionSize';

// Intra Bridge Communication
export const BRIDGE_REQ_SUBSCRIBE = 'subscribe';
export const BRIDGE_HOST_READY = 'ready';
export const BRIDGE_STATE_UPDATE = 'bridgestateupdate';
export const BRIDGE_STORE_STATE_UPDATE = 'bridgestorestateupdate';
export const BRIDGE_PLAYER_EVENT = 'bridgeplayerevent';
export const BRIDGE_PLAYER_EVENT_WITH_PAYLOAD = 'bridgeplayereventwithpayload';
export const BRIDGE_DOCUMENT_EVENT = 'bridgedocumentevent';
export const BRIDGE_HOST_NAMESPACE = 'player.embed.host';
export const BRIDGE_CLIENT_NAMESPACE = 'player.embed.client';
export const BRIDGE_DESTROY = 'bridgedestroy';

// Duplicate of playback states in state-tracker, as importing it in the client
// will cause the file size to balloon
export const PLAYBACK_PAUSED = 'paused';
export const PLAYBACK_PLAYING = 'playing';
export const PLAYBACK_ENDED = 'ended';

// used to call player state callbacks
const PLAYER_STATE_UPDATE = 'playerstateupdate';

export class EmbedClient {
    constructor(root, rawOptions = DEFAULT_OPTIONS) {
        this._eventEmitter = new EventEmitter();
        this._playerStateEmitter = new EventEmitter();
        this._playerState = DEFAULT_EMBED_STATE;
        this._storeState = {};
        this._onHostReady = this._getHostReady();
        this._iframe = this._createPlayerIframe(rawOptions);

        root.appendChild(this._iframe);
        this._host = this._iframe.contentWindow;

        this._send(BRIDGE_REQ_SUBSCRIBE);
    }

    destroy() {
        this.callPlayerMethod(METHOD_DESTROY);
    }

    _createPlayerIframe(rawOptions) {
        const options = this._normalizeOptions(rawOptions);
        options.origin = document.location.origin;

        const iframe = document.createElement('iframe');
        const paramString = paramsToString(omit(options, 'width', 'height'));
        const url = `${PLAYER_HOST}/?${paramString}`;

        iframe.setAttribute('src', url);
        iframe.setAttribute('width', options.width);
        iframe.setAttribute('height', options.height);
        iframe.setAttribute('frameBorder', '0');
        iframe.setAttribute('scrolling', 'no');

        if (options.allowfullscreen) {
            iframe.setAttribute('allowfullscreen', '');
        }

        return iframe;
    }

    _normalizeOptions(rawOptions) {
        const options = assign({}, DEFAULT_OPTIONS, rawOptions);

        if (rawOptions.allowfullscreen !== false) {
            options.allowfullscreen = true;
        }

        return options;
    }

    _getHostReady() {
        return new Promise((resolve, reject) => {
            const listenerFunc = hostListener.bind(this);
            window.addEventListener('message', listenerFunc);

            // 15 second timeout for bridge to be established
            setTimeout(reject, HOST_WAIT_TIMEOUT);

            function hostListener(message) {
                if (this._isClientMessage(message) && message.data.method === BRIDGE_HOST_READY) {
                    this._storeState = message.data.args[0];
                    window.removeEventListener('message', listenerFunc);
                    window.addEventListener('message', this);
                    this._eventEmitter.emit(EVENT_EMBED_READY);
                    resolve();
                }
            }
        });
    }

    _send(method, ...args) {
        // Set the namespace so only servers can read this message.
        // This prevents us from accidentally talking to ourselves.
        const payload = {
            namespace: BRIDGE_HOST_NAMESPACE,
            method: method,
            args: args,
        };

        this._host.postMessage(payload, '*');
    }

    _deferSend(...args) {
        return this._onHostReady.then(
            () => this._send.apply(this, args)
        );
    }

    _isClientMessage(message) {
        // Other components use this API so we must check if the message
        // is actually intended for the player.
        // It is possible to have a client and server listening on the same
        // window, so we must also avoid talking to ourself.

        if (!this._iframe) {
            // The iframe is destroyed, don't process any messages.
            return false;
        }

        return (
            Boolean(message.data) &&
            message.data.namespace === BRIDGE_CLIENT_NAMESPACE &&
            message.source === this._iframe.contentWindow
        );
    }

    handleEvent(event) {
        // This function is fired any time the window receives a message.

        if (!this._isClientMessage(event)) {
            return;
        }

        switch (event.data.method) {
        case BRIDGE_STATE_UPDATE:
            this._playerState = event.data.args[0];
            this._playerStateEmitter.emit(PLAYER_STATE_UPDATE, this._playerState);
            break;
        case BRIDGE_PLAYER_EVENT:
            this._eventEmitter.emit(event.data.args[0].event);
            break;
        case BRIDGE_PLAYER_EVENT_WITH_PAYLOAD:
            this._eventEmitter.emit(event.data.args[0].event, event.data.args[0].data);
            break;
        case BRIDGE_STORE_STATE_UPDATE:
            this._storeState = event.data.args[0];
            break;
        case BRIDGE_DESTROY:
            this._iframe.parentNode.removeChild(this._iframe);
            delete this._iframe;
            delete this._host;
            break;
        }
    }

    getPlayerState() {
        return this._playerState;
    }

    getStoreState() {
        return this._storeState;
    }

    addEventListener(name, callback) {
        this._eventEmitter.on(name, callback);
    }

    addPlayerStateListener(callback) {
        this._playerStateEmitter.on(PLAYER_STATE_UPDATE, callback);
        callback(this._playerState);
    }

    removePlayerStateListener(callback) {
        this._playerStateEmitter.off(PLAYER_STATE_UPDATE, callback);
    }

    removeEventListener(name, callback) {
        this._eventEmitter.off(name, callback);
    }

    callPlayerMethod(name, ...args) {
        return this._deferSend(name, ...args);
    }

    setWidth(width) {
        if (isFinite(width) && width >= 0) {
            this._iframe.setAttribute('width', width);
        }
    }

    setHeight(height) {
        if (isFinite(height) && height >= 0) {
            this._iframe.setAttribute('height', height);
        }
    }
}
