import isString from 'lodash/isString';
import * as Bridge from './client';

// Embed Version 0.0

export class PlayerEmbed {
    constructor(root, options) {
        this._bridge = new Bridge.EmbedClient(rootElement(root), options);
    }

    // Embed API Function Calls

    // Embed Playback Controls Functions
    play() {
        this._bridge.callPlayerMethod(Bridge.METHOD_PLAY);
    }

    pause() {
        this._bridge.callPlayerMethod(Bridge.METHOD_PAUSE);
    }

    seek(time) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SEEK, time);
    }

    setVolume(volume) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_VOLUME, volume);
    }

    setMuted(muted) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_MUTE, muted);
    }

    setChannel(channel) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_CHANNEL, channel);
    }

    setVideo(video) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_VIDEO, video);
    }

    setCollection(collection, video = '') {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_COLLECTION, collection, video);
    }

    setQuality(quality) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_QUALITY, quality);
    }

    // Embed Event Handling Functions

    addEventListener(name, callback) {
        this._bridge.addEventListener(name, callback);
    }

    removeEventListener(name, callback) {
        this._bridge.removeEventListener(name, callback);
    }

    // Embed Player Status Functions

    getChannel() {
        return this._bridge.getPlayerState().channelName;
    }

    getCurrentTime() {
        return this._bridge.getPlayerState().currentTime;
    }

    getDuration() {
        return this._bridge.getPlayerState().duration;
    }

    getEnded() {
        return this._bridge.getPlayerState().playback === Bridge.PLAYBACK_ENDED;
    }

    getMuted() {
        return this._bridge.getPlayerState().muted;
    }

    getPlaybackStats() {
        return this._bridge.getPlayerState().stats;
    }

    getPlaySessionId() {
        return this._bridge.getStoreState().playSessionId;
    }

    isPaused() {
        return this._bridge.getPlayerState().playback === Bridge.PLAYBACK_PAUSED;
    }

    getQuality() {
        return this._bridge.getPlayerState().quality;
    }

    getQualities() {
        return this._bridge.getPlayerState().qualitiesAvailable;
    }

    getViewers() {
        return this._bridge.getStoreState().viewercount;
    }

    getVideo() {
        return this._bridge.getPlayerState().videoID;
    }

    getVolume() {
        return this._bridge.getPlayerState().volume;
    }

    // Properties specifically for Dashboard

    get channel() {
        return this.getChannel();
    }

    get muted() {
        return this.getMuted();
    }

    get viewers() {
        return this.getViewers();
    }

    // Misc

    destroy() {
        this._bridge.destroy();
    }
}

/**
 * Gets the root HTMLElement from a given input. If a string, use it as an ID;
 * if already an element, return it as-is.
 *
 * @param {String|HTMLElement} root
 * @return {HTMLElement}
 */
function rootElement(root) {
    if (isString(root)) {
        return document.getElementById(root);
    }
    return root;
}

// Make the class global as 'Twitch.embed.Player' just for v0
window.Twitch = window.Twitch || {};
window.Twitch.embed = window.Twitch.embed || {};
window.Twitch.embed.Player = PlayerEmbed;
