import isString from 'lodash/isString';
import * as Bridge from './client';
import { isTwitchEmbed } from 'util/twitch-embed';
import ERROR_METADATA from 'error-metadata';

// Embed Version 1.0
// API Here: https://github.com/justintv/Twitch-API/blob/master/embed-video.md

const EMBED_ERRORS = Object.freeze({
    ABORTED: ERROR_METADATA.ABORTED.code,
    NETWORK: ERROR_METADATA.NETWORK.code,
    DECODE: ERROR_METADATA.DECODE.code,
    FORMAT_NOT_SUPPORTED: ERROR_METADATA.FORMAT_NOT_SUPPORTED.code,
    CONTENT_NOT_AVAILABLE: ERROR_METADATA.CONTENT_NOT_AVAILABLE.code,
    RENDERER_NOT_AVAILABLE: ERROR_METADATA.RENDERER_NOT_AVAILABLE.code,
});

const EMBED_ANALYTICS = Object.freeze({
    MINUTE_WATCHED: Bridge.EVENT_EMBED_MINUTE_WATCHED,
    VIDEO_PLAY: Bridge.EVENT_EMBED_VIDEO_PLAY,
    BUFFER_EMPTY: Bridge.EVENT_EMBED_BUFFER_EMPTY,
    VIDEO_ERROR: Bridge.EVENT_EMBED_VIDEO_PLAYBACK_ERROR,
});

const EMBED_EVENTS = Object.freeze({
    TRANSITION_TO_RECOMMENDED_VOD: Bridge.EVENT_EMBED_TRANSITION_TO_REC_VOD,
});

export class PlayerEmbed {
    constructor(root, options) {
        this._bridge = new Bridge.EmbedClient(rootElement(root), options);
    }

    // Embed Event Constants
    static get READY() {
        return Bridge.EVENT_EMBED_READY;
    }

    static get PLAY() {
        return Bridge.EVENT_EMBED_PLAY;
    }

    static get PAUSE() {
        return Bridge.EVENT_EMBED_PAUSE;
    }

    static get ENDED() {
        return Bridge.EVENT_EMBED_ENDED;
    }

    static get ONLINE() {
        return Bridge.EVENT_EMBED_ONLINE;
    }

    static get OFFLINE() {
        return Bridge.EVENT_EMBED_OFFLINE;
    }

    static get ERROR() {
        return Bridge.EVENT_EMBED_ERROR;
    }

    // Embed Playback Controls Functions
    play() {
        this._bridge.callPlayerMethod(Bridge.METHOD_PLAY);
    }

    pause() {
        this._bridge.callPlayerMethod(Bridge.METHOD_PAUSE);
    }

    seek(time) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SEEK, time);
    }

    setVolume(volume) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_VOLUME, volume);
    }

    setTheatre(value) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_THEATRE, value);
    }

    // Setting fullscreen to true only works if it is called inside of an input event.
    setFullscreen(value) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_FULLSCREEN, value);
    }

    setMuted(muted) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_MUTE, muted);
    }

    setChannel(channel) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_CHANNEL, channel);
    }

    setChannelId(channelId) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_CHANNEL_ID, channelId);
    }

    setCollection(collection, video = '', timestamp = '') {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_COLLECTION, collection, video, timestamp);
    }

    setVideo(video, timestamp = '') {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_VIDEO, video, timestamp);
    }

    setContent({ contentId, customerId }) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_CONTENT, contentId, customerId);
    }

    setClip(slug) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_CLIP, slug);
    }

    setVideoSource(url) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_VIDEO_SOURCE, url);
    }

    setQuality(quality) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_QUALITY, quality);
    }

    setWidth(width) {
        this._bridge.setWidth(width);
    }

    setHeight(height) {
        this._bridge.setHeight(height);
    }

    setMiniPlayerMode(value) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_MINI_PLAYER_MODE, value);
    }

    setTrackingProperties(props) {
        if (isTwitchEmbed()) {
            this._bridge.callPlayerMethod(Bridge.METHOD_SET_TRACKING_PROPERTIES, props);
        }
    }

    setPlayerType(playerType) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_PLAYER_TYPE, playerType);
    }

    // Embed Event Handling Functions

    addEventListener(name, callback) {
        this._bridge.addEventListener(name, callback);
    }

    removeEventListener(name, callback) {
        this._bridge.removeEventListener(name, callback);
    }

    enableCaptions() {
        this._bridge.callPlayerMethod(Bridge.METHOD_ENABLE_CAPTIONS);
    }

    disableCaptions() {
        this._bridge.callPlayerMethod(Bridge.METHOD_DISABLE_CAPTIONS);
    }

    setCaptionSize(sizeInPx) {
        this._bridge.callPlayerMethod(Bridge.METHOD_SET_CAPTION_SIZE, sizeInPx);
    }

    // Embed Player Status Functions
    getContentId() {
        return this._bridge.getStoreState().stream.contentId;
    }

    getChannel() {
        return this._bridge.getPlayerState().channelName;
    }

    getChannelId() {
        return this._bridge.getPlayerState().channelId;
    }

    getCurrentTime() {
        return this._bridge.getPlayerState().currentTime;
    }

    getCustomerId() {
        return this._bridge.getStoreState().stream.customerId;
    }

    getDuration() {
        return this._bridge.getPlayerState().duration;
    }

    getEnded() {
        return this._bridge.getPlayerState().playback === Bridge.PLAYBACK_ENDED;
    }

    getMuted() {
        return this._bridge.getPlayerState().muted;
    }

    getPlaybackStats() {
        return this._bridge.getStoreState().stats.videoStats;
    }

    getPlaySessionId() {
        return this._bridge.getStoreState().playSessionId;
    }

    isPaused() {
        return this._bridge.getPlayerState().playback === Bridge.PLAYBACK_PAUSED;
    }

    getQuality() {
        return this._bridge.getPlayerState().quality;
    }

    getQualities() {
        return this._bridge.getPlayerState().qualitiesAvailable;
    }

    getViewers() {
        return this._bridge.getStoreState().viewercount;
    }

    getVideo() {
        return this._bridge.getPlayerState().videoID;
    }

    getVolume() {
        return this._bridge.getPlayerState().volume;
    }

    getTheatre() {
        return this._bridge.getStoreState().screenMode.isTheatreMode;
    }

    getFullscreen() {
        return this._bridge.getStoreState().screenMode.isFullScreen;
    }

    getFullscreenEnabled() {
        return this._bridge.getStoreState().screenMode.canFullScreen;
    }

    getSessionInfo() {
        return {
            broadcastId: this._bridge.getStoreState().broadcastId,
            playSessionId: this._bridge.getStoreState().playSessionId,
        };
    }

    getCaptionsAvailable() {
        return this._bridge.getStoreState().captions.available;
    }

    // Events for twitch-everywhere bridge
    _addPlayerStateListener(callback) {
        this._bridge.addPlayerStateListener(callback);
    }

    _removePlayerStateListener(callback) {
        this._bridge.removePlayerStateListener(callback);
    }

    destroy() {
        this._bridge.destroy();
    }
}

/**
 * Gets the root HTMLElement from a given input. If a string, use it as an ID;
 * if already an element, return it as-is.
 *
 * @param {String|HTMLElement} root
 * @return {HTMLElement}
 */
function rootElement(root) {
    if (isString(root)) {
        return document.getElementById(root);
    }
    return root;
}

// Make the class global as 'Twitch.Player'
window.Twitch = window.Twitch || {};
window.Twitch.Player = PlayerEmbed;
window.Twitch.Analytics = EMBED_ANALYTICS;
window.Twitch.Errors = EMBED_ERRORS;
window.Twitch.EmbedEvents = EMBED_EVENTS;
