import isObject from 'lodash/isObject';
import * as StreamActions from '../actions/stream';
import { embedHostLoaded } from '../actions/embed-event-emitter';
import { pause, play, changeVolume, mutePlayer } from '../actions/video-api';
import { setTheatreMode, setFullScreen } from '../actions/screen-mode';
import { setMiniPlayerMode } from '../actions/ui';
import { ONLINE_STATUS } from 'state/online-status';
import { OPEN_STREAM } from '../backend/events/twitch-event';
import { clearCollection, requestCollection } from '../actions/collection';
import { selectQuality } from '../actions/quality';
import { subscribe } from '../util/subscribe';
import { PLAYER_FRONTPAGE, PLAYER_POPOUT, PLAYER_SITE, PLAYER_FEED } from '../util/player-type';
import { VODTwitchContentStream } from '../stream/twitch-vod';
import { LiveTwitchContentStream } from '../stream/twitch-live';
import * as EmbedClient from './client';
import { COMPANION_RENDERED } from '../ads/advertisement-event';
import { FLASH_AD_EVENTS } from '../backend/flash';
import { TIER_1_EVENTS, VIDEO_PLAYBACK_ERROR } from '../analytics/events';
import { CONTENT_MODE_PROVIDED } from '../stream/provided';
import includes from 'lodash/includes';
import { setTrackingProperties } from 'actions/tracking';
import { setStartTime } from 'actions/playback';
import { setPlayerType } from 'actions/env';
import { setCaptionsEnabled, setCaptionsPreset, CUSTOM } from 'actions/captions';
import { krakenRequestv5 } from 'api';
import { isTwitchEmbed, isDesktopApp } from 'util/twitch-embed';

const ANALYTICS_EMBED_EVENT_MAP = Object.freeze({
    [TIER_1_EVENTS.MINUTE_WATCHED]: EmbedClient.EVENT_EMBED_MINUTE_WATCHED,
    [TIER_1_EVENTS.VIDEO_PLAY]: EmbedClient.EVENT_EMBED_VIDEO_PLAY,
    [TIER_1_EVENTS.BUFFER_EMPTY]: EmbedClient.EVENT_EMBED_BUFFER_EMPTY,
    [VIDEO_PLAYBACK_ERROR]: EmbedClient.EVENT_EMBED_VIDEO_PLAYBACK_ERROR,
});

export const PUBLIC_EVENTS = Object.freeze({
    [EmbedClient.EVENT_EMBED_ENDED]: true,
    [EmbedClient.EVENT_EMBED_PAUSE]: true,
    [EmbedClient.EVENT_EMBED_PLAY]: true,
    [EmbedClient.EVENT_EMBED_OFFLINE]: true,
    [EmbedClient.EVENT_EMBED_ONLINE]: true,
    [EmbedClient.EVENT_EMBED_READY]: true,
    [EmbedClient.EVENT_EMBED_MINUTE_WATCHED]: true,
    [EmbedClient.EVENT_EMBED_VIDEO_PLAY]: true,
    [EmbedClient.EVENT_EMBED_BUFFER_EMPTY]: true,
    [EmbedClient.EVENT_EMBED_VIDEO_PLAYBACK_ERROR]: true,
});

const DEFAULT_OPTIONS = Object.freeze({
    origin: '',
});

export class EmbedHost {
    constructor(player, store, opts = DEFAULT_OPTIONS) {
        this._player = player;
        this._store = store;
        this._clients = [];
        this._unsubscribes = [];
        this._window = this._store.getState().window;
        this._targetOrigin = opts.origin;

        this._bridgeIsReady = () => {
            throw new Error('_bridgeIsReady should only be called in _sendReadyEvent');
        };
        this._onBridgeReady = new Promise(resolve => {
            this._bridgeIsReady = resolve;
        });

        // Start listening for any events.
        this._window.addEventListener('message', this);

        const { playerType } = this._store.getState().env;
        if (playerType === PLAYER_FRONTPAGE || playerType === PLAYER_FEED) {
            this._player.addEventListener(OPEN_STREAM, this._sendOpenStreamEvent.bind(this));
        }

        // If we were opened as an iframe or popup, inform the client.
        const frame = this._window.opener || this._window.parent;
        if (frame && frame !== this._window) {
            this._addClient(frame);
        }

        this._store.dispatch(embedHostLoaded(this));
        this._initSubscriptions();
    }

    _sendOpenStreamEvent() {
        this._sendPlayerEvent(EmbedClient.EVENT_EMBED_OPEN_STREAM);
    }

    _sendReadyEvent() {
        this._sendBridgeReady();
        this._bridgeIsReady();
    }

    _initSubscriptions() {
        this._unsubscribes.push(subscribe(this._store, [
            'analytics.playSessionId',
            'captions.available',
            'manifestInfo.broadcast_id',
            'stream',
        ], () => {
            this._sendStoreState();
        }));

        this._unsubscribes.push(subscribe(this._store, ['error'], this.onError.bind(this)));
        this.onError(this._store.getState());

        this._unsubscribes.push(subscribe(this._store, ['analytics.playSessionId', 'manifestInfo.broadcast_id'], () => {
        }));

        this._unsubscribes.push(subscribe(this._store, ['viewercount'], () => {
            this._sendStoreState();
            this._sendPlayerEvent(EmbedClient.EVENT_EMBED_VIEWERS_CHANGE);
        }));

        this._unsubscribes.push(subscribe(this._store, ['playback.contentShowing'], ({ playback }) => {
            if (playback.contentShowing) {
                this._sendPlayerEvent(EmbedClient.EVENT_EMBED_CONTENT_SHOWING);
            }
        }));

        // eslint-disable-next-line max-len
        this._unsubscribes.push(subscribe(this._store, ['playback.paused', 'playback.ended', 'playback.hasPlayed'], ({ playback }) => {
            const { paused, ended, hasPlayed } = playback;

            if (ended) {
                this._sendPlayerEvent(EmbedClient.EVENT_EMBED_ENDED);
            } else if (paused) {
                this._sendPlayerEvent(EmbedClient.EVENT_EMBED_PAUSE);
            } else if (hasPlayed) {
                this._sendPlayerEvent(EmbedClient.EVENT_EMBED_PLAY);
            }
        }));

        this._unsubscribes.push(subscribe(this._store, ['onlineStatus'], ({ onlineStatus }) => {
            if (onlineStatus === ONLINE_STATUS) {
                this._sendPlayerEvent(EmbedClient.EVENT_EMBED_ONLINE);
            } else {
                this._sendPlayerEvent(EmbedClient.EVENT_EMBED_OFFLINE);
            }
        }));

        // eslint-disable-next-line max-len
        this._unsubscribes.push(subscribe(this._store, ['screenMode.isTheatreMode'], ({ screenMode }) => {
            this._sendStoreState();
            if (screenMode.isTheatreMode) {
                this._sendPlayerEvent(EmbedClient.EVENT_THEATRE_ENTERED);
            } else {
                this._sendPlayerEvent(EmbedClient.EVENT_THEATRE_EXITED);
            }
        }));
        // eslint-disable-next-line max-len
        this._unsubscribes.push(subscribe(this._store, ['screenMode.isFullScreen'], ({ screenMode }) => {
            this._sendStoreState();
            if (screenMode.isFullScreen) {
                this._sendPlayerEvent(EmbedClient.EVENT_FULLSCREEN_ENTERED);
            } else {
                this._sendPlayerEvent(EmbedClient.EVENT_FULLSCREEN_EXITED);
            }
        }));

        this._unsubscribes.push(subscribe(this._store, ['playback.currentTime'], ({ playback }) => {
            const { currentTime } = playback;
            this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD, {
                event: EmbedClient.EVENT_EMBED_TIME_UPDATE,
                data: currentTime,
            });
        }));

        this._unsubscribes.push(subscribe(this._store, [
            'stream',
            'streamMetadata.channelName',
            'streamMetadata.channel.id',
            'streamMetadata.videoId',
            'quality.current',
            'quality.available',
            'playback.currentTime',
            'playback.duration',
            'playback.muted',
            'playback.volume',
            'playback.paused',
            'playback.ended',
        ], this._sendPlayerState.bind(this)));
    }

    onError({ error }) {
        if (!error.hasError) {
            return;
        }

        this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD, {
            event: EmbedClient.EVENT_EMBED_ERROR,
            data: {
                code: error.code,
            },
        });
    }

    _addClient(client) {
        this._clients.push(client);
    }

    _sendBridgeReady() {
        this._clients.forEach(client => {
            // Set the namespace so only clients can read this message.
            // This prevents us from accidentally talking to ourselves
            client.postMessage({
                namespace: EmbedClient.BRIDGE_CLIENT_NAMESPACE,
                method: EmbedClient.BRIDGE_HOST_READY,
                args: [this._getNormalizedStoreState()],
            }, this._targetOrigin);
        });
    }

    _postTo(method, args, origin) {
        this._clients.forEach(client => {
            // Set the namespace so only clients can read this message.
            // This prevents us from accidentally talking to ourselves
            client.postMessage({
                namespace: EmbedClient.BRIDGE_CLIENT_NAMESPACE,
                method,
                args,
            }, origin);
        });
    }

    /**
     * postMessage's to public/twitch origins depending on event name
     * NOTE: args can be either an object, array, or string.
     *
     * @param {String} method - used by client.js to determine which method to invoke
     * @param {Object} args - event and payload, or a state blob
     */
    _sendAll(method, args = {}) {
        const parsedArgs = Array.isArray(args) ? args : [args];

        return this._onBridgeReady.then(() => {
            const isTwitchOrigin = isTwitchEmbed(this._targetOrigin);
            const isDesktopOrigin = isDesktopApp(this._targetOrigin);
            const isEventMethod = this._isEvent(method);
            const isPublicEvent = this._isPublicEvent(method, parsedArgs);

            if (isTwitchOrigin || isDesktopOrigin || !isEventMethod || isPublicEvent) {
                return this._postTo(method, parsedArgs, this._targetOrigin);
            }
        });
    }

    _isEvent(method) {
        return (
            method === EmbedClient.BRIDGE_PLAYER_EVENT ||
            method === EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD
        );
    }

    _isPublicEvent(method, parsedArgs) {
        if (!Array.isArray(parsedArgs)) {
            return false;
        }

        const [args] = parsedArgs;
        if (!this._isEvent(method) || args.event === undefined) {
            return false;
        }

        return PUBLIC_EVENTS[args.event] === true;
    }

    // eslint-disable-next-line complexity
    handleEvent(event) {
        // This function is fired any time the window receives a message.
        // Other components use this API so we must check if the message
        // is actually intended for the player.
        // It is possible to have a client and server listening on the same
        // window, so we must also avoid talking to ourselves.

        // Make sure we can parse the message.
        if (!isObject(event.data)) {
            return;
        }

        // Only handle messages targeting the host.
        if (event.data.namespace !== EmbedClient.BRIDGE_HOST_NAMESPACE) {
            return;
        }

        // Ignore messages from non specified origins
        if (event.origin !== this._targetOrigin) {
            return;
        }

        switch (event.data.method) {
        case EmbedClient.BRIDGE_REQ_SUBSCRIBE:
            this._addClient(event.source);
            break;
        case EmbedClient.METHOD_PLAY:
            this._store.dispatch(play());
            break;
        case EmbedClient.METHOD_PAUSE:
            this._store.dispatch(pause());
            break;
        case EmbedClient.METHOD_SET_CONTENT: // eslint-disable-line no-case-declarations
            this._store.dispatch(StreamActions.setStream({
                contentType: StreamActions.TYPE_PROVIDED,
                contentId: event.data.args[0],
                customerId: event.data.args[1],
            }));
            this._store.dispatch(clearCollection());
            break;
        case EmbedClient.METHOD_SET_CLIP:
            this._store.dispatch(StreamActions.setStream({
                contentType: StreamActions.TYPE_CLIP,
                contentId: event.data.args[0],
            }));
            this._store.dispatch(clearCollection());
            break;
        case EmbedClient.METHOD_SET_VIDEO_SOURCE:
            this._store.dispatch(StreamActions.setStream({
                contentType: StreamActions.TYPE_VIDEO_SOURCE,
                contentId: event.data.args[0],
            }));
            this._store.dispatch(clearCollection());
            break;
        case EmbedClient.METHOD_SET_CHANNEL: // eslint-disable-line no-case-declarations
            this._store.dispatch(StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: event.data.args[0],
            }));
            this._store.dispatch(clearCollection());
            break;
        case EmbedClient.METHOD_SET_CHANNEL_ID: // eslint-disable-line no-case-declarations
            this._handleChannelId(event.data.args[0]);
            break;
        case EmbedClient.METHOD_SET_VIDEO: // eslint-disable-line no-case-declarations
            this._handleSetVideo(event.data.args);
            break;
        case EmbedClient.METHOD_SET_COLLECTION:
            this._handleSetCollection(event.data.args);
            break;
        case EmbedClient.METHOD_SEEK:
            this._player.setCurrentTime(parseFloat(event.data.args[0]));
            break;
        case EmbedClient.METHOD_SET_QUALITY:
            this._store.dispatch(selectQuality(event.data.args[0]));
            break;
        case EmbedClient.METHOD_SET_MUTE:
            this._store.dispatch(mutePlayer(!!event.data.args[0]));
            break;
        case EmbedClient.METHOD_SET_VOLUME:
            this._store.dispatch(changeVolume(event.data.args[0]));
            break;
        case EmbedClient.METHOD_SET_THEATRE:
            this._store.dispatch(setTheatreMode(event.data.args[0]));
            break;
        case EmbedClient.METHOD_SET_FULLSCREEN:
            this._store.dispatch(setFullScreen(event.data.args[0]));
            break;
        case EmbedClient.METHOD_DESTROY:
            this._player.destroy();
            this.destroy();
            // Let the client know the destroy has started so it can remove the iframe.
            this._sendAll(EmbedClient.BRIDGE_DESTROY);
            break;
        case EmbedClient.METHOD_SET_MINI_PLAYER_MODE:
            this._store.dispatch(setMiniPlayerMode(event.data.args[0]));
            break;
        case EmbedClient.METHOD_SET_TRACKING_PROPERTIES:
            this._setTrackingProperties(event.data.args[0]);
            break;
        case EmbedClient.METHOD_SET_PLAYER_TYPE:
            this._store.dispatch(setPlayerType(event.data.args[0]));
            break;
        case EmbedClient.METHOD_ENABLE_CAPTIONS:
            this._store.dispatch(setCaptionsEnabled(true));
            break;
        case EmbedClient.METHOD_DISABLE_CAPTIONS:
            this._store.dispatch(setCaptionsEnabled(false));
            break;
        case EmbedClient.METHOD_SET_CAPTION_SIZE:
            this._store.dispatch(setCaptionsPreset(CUSTOM, {
                fontSize: event.data.args[0],
            }));
            break;
        }
    }

    _handleChannelId(channelId) {
        const sanitizedChannelId = String(channelId).replace(/[^A-Za-z0-9_]/g, '');
        krakenRequestv5(`channels/${sanitizedChannelId}`).then(data => {
            this._store.dispatch(StreamActions.setStream({
                contentType: StreamActions.TYPE_CHANNEL,
                contentId: data.name,
            }));
            this._store.dispatch(clearCollection());
        });
    }

    _handleSetVideo([videoId, timestamp]) {
        this._store.dispatch(StreamActions.setStream({
            contentType: StreamActions.TYPE_VIDEO,
            contentId: videoId,
        }));
        this._store.dispatch(clearCollection());
        this._store.dispatch(setStartTime(timestamp));
    }

    _handleSetCollection([collectionId, videoId, timestamp]) {
        this._store.dispatch(requestCollection(collectionId, videoId, timestamp));
    }

    // TODO: Consolidate this method's functionality into _sendStoreState
    _sendPlayerState({ streamMetadata, stream, quality, playback, viewercount }) {
        const playbackState = this._configurePlaybackState();
        const currentStream = stream;

        const state = {
            channelName: currentStream instanceof LiveTwitchContentStream ? currentStream.channel : '',
            channelId: streamMetadata.channel.id,
            currentTime: playback.currentTime,
            duration: playback.duration,
            muted: playback.muted,
            playback: playbackState,
            quality: quality.current,
            qualitiesAvailable: quality.available,
            videoID: currentStream instanceof VODTwitchContentStream ? currentStream.videoId : '',
            viewers: viewercount,
            volume: playback.volume,
        };

        this._sendAll(EmbedClient.BRIDGE_STATE_UPDATE, state);
    }

    _configurePlaybackState() {
        if (this._store.getState().playback.ended) {
            return EmbedClient.PLAYBACK_ENDED;
        } else if (this._store.getState().playback.paused) {
            return EmbedClient.PLAYBACK_PAUSED;
        }

        return EmbedClient.PLAYBACK_PLAYING;
    }

    _sendPlayerEvent(event) {
        this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT, { event });
    }

    _sendAdCompanionEvent(event, data) {
        switch (event) {
        case COMPANION_RENDERED:
            this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD, {
                event: EmbedClient.EVENT_EMBED_AD_COMPANION_RENDERED,
                data,
            });
            break;
        case FLASH_AD_EVENTS.AD_COMPANION_RENDERED:
            this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD, {
                event: EmbedClient.EVENT_EMBED_FLASH_AD_COMPANION_RENDERED,
                data,
            });
            break;
        }
    }

    _emitWheelEvent(data) {
        this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD, {
            event: EmbedClient.EVENT_EMBED_WHEEL,
            data,
        });
    }

    _sendFirstPartyAnalyticEvent(eventName, properties) {
        this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD, {
            event: ANALYTICS_EMBED_EVENT_MAP[eventName],
            data: properties,
        });
    }

    _getNormalizedStoreState() {
        const {
            analytics,
            captions,
            manifestInfo,
            screenMode,
            stats,
            stream,
            viewercount,
        } = this._store.getState();

        return {
            captions: {
                available: captions.available,
            },
            viewercount,
            stats: {
                videoStats: stats.videoStats,
            },
            playSessionId: analytics.playSessionId,
            broadcastId: manifestInfo.broadcast_id,
            screenMode,
            stream: {
                contentId: stream.contentType === CONTENT_MODE_PROVIDED ? stream.contentId : '',
                customerId: stream.contentType === CONTENT_MODE_PROVIDED ? stream.customerId : '',
            },
        };
    }

    _sendStoreState() {
        this._sendAll(EmbedClient.BRIDGE_STORE_STATE_UPDATE, this._getNormalizedStoreState());
    }

    _sendTransitionToCollectionEvent(data) {
        this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD, {
            event: EmbedClient.EVENT_EMBED_TRANSITION_TO_COLLECTION_VOD,
            data,
        });
    }

    _sendTransitionToRecommendedVodEvent(data) {
        this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD, {
            event: EmbedClient.EVENT_EMBED_TRANSITION_TO_REC_VOD,
            data,
        });
    }

    _sendStatsUpdateEvent() {
        this._sendStoreState();
        this._sendPlayerEvent(EmbedClient.EVENT_EMBED_STATS_UPDATE);
    }

    _sendSeekedEvent(currentTime) {
        this._sendAll(EmbedClient.BRIDGE_PLAYER_EVENT_WITH_PAYLOAD, {
            event: EmbedClient.EVENT_EMBED_SEEKED,
            data: currentTime,
        });
    }

    _setTrackingProperties(props) {
        const allowedTypes = [
            PLAYER_SITE,
            PLAYER_FRONTPAGE,
            PLAYER_POPOUT, // for mobile web player
            PLAYER_FEED,
        ];

        if (!includes(allowedTypes, this._store.getState().env.playerType)) {
            return;
        }

        this._store.dispatch(setTrackingProperties(props));
    }

    destroy() {
        this._window.removeEventListener('message', this);
        if (this._store.getState().env.playerType === PLAYER_FRONTPAGE) {
            this._player.removeEventListener(OPEN_STREAM, this._sendOpenStreamEvent.bind(this));
        }
        this._unsubscribes.forEach(unsub => unsub());
    }
}
