import { cookie } from 'cookie_js';
import assign from 'lodash/assign';
import Client from 'minixperiment-client';
import ServiceProvider from 'minixperiment-client/lib/providers/service';
import { krakenUserInfo } from './api';
import bowser from 'bowser';

// Opts users in to network profile collection; some set of users will report
// segment-granularity bandwidth information (ttfb, bytes, download duration)
// for each segment downloaded during HLS playback.
export const NETWORK_PROFILE_COLLECTION = '5fbb67a0-b4ff-4775-b836-e9a348a87481';
// Experiment for video content analysis and vod markers.
export const MARKERS = '9e5fffd0-9693-412b-913d-7da332fdeea4';
// Experiment using similar VODs for in-player-recs
export const PINEAPPLE = '1a42d36b-3f3f-4b80-8252-bdd6cc675ec3';
// Experiment for removing the HTML5 Toggle
export const HTML5_TOGGLE_REMOVAL = '354aaebd-7643-434f-be37-98fe12c01220';
// Experiment for Prime Advertising conversion experiemnt
export const PADTIME = 'b8d1e7be-59f7-4ee3-a0c6-54ce94366d14';
// Experiment for the viewer experience of watch party
export const VIEW_WATCH_TOGETHER = 'd0518b75-fbba-418a-9445-ff7c517beb32';
// Experiment for Skippable ads
export const CHOMPERZ = 'd48fe249-fd73-4c8f-bd17-415b70dc3708';
// Experiment for auto suggest notification
export const AUTO_SUGGEST_NOTIFICATION = '74d6faab-1794-458a-a5f6-b92602254e9e';
// Experiment using real-time qos metrics to select playback cluster/CDN
export const REALTIME_QOS = '6a2630a4-edc7-4f2a-8cfe-e4de75eb0d98';
// Minixperiment to control version distribution of mediaplayer for users
export const MEDIAPLAYER_VERSION_CONTROL = 'ec70bd08-dbc8-4f48-9f60-d035fec3eb20';
// Experiment for communicating premiere/rerun status
export const TWILIGHT_PREMIERE_UPLOAD_FLOW = '4d3d19a7-38c5-4328-9730-7d99565d5edd';
// Experiment for declining prerolls on Raids
export const LEVIATHAN = '3886e2be-85fd-43a4-84b9-02024290026a';
// Experiment for low-latency streaming
export const FAST_BREAD = 'fd5223cb-3ae7-4a9f-9af8-ce0d44d4b9f1';
// Experiment for low-latency clips embed
// CLIPS_GOT_EM tests differing latency across clips-upload and player-ui
// We always fire variant1 on player-ui.
export const CLIPS_GOT_EM = '1b959783-5196-4c52-863d-06949b814e43';

function staffOnlyOverride(experimentValue) {
    return krakenUserInfo().then(user => user.type === 'staff' ? experimentValue : Promise.reject());
}

// Disable stream stitching browser for Firefox 45 or below
export function bakingBreadSafeBrowser(userAgent) {
    const uaFingerprint = bowser._detect(userAgent);
    // Non Firefox browsers are safe
    if (!uaFingerprint.firefox) {
        return true;
    }
    // If malformed UA, return false.
    return uaFingerprint.version ? parseInt(uaFingerprint.version.slice(0,2), 10) > 45 : false;
}

/**
 * Create an Minixperiment client with the provided options.
 *
 * @param {Object} options
 *        `deviceID`: The user's device ID
 *        `login`:    The user's login name (`null` if not logged in)
 *        `userAgent`: The user's userAgent
 * @return {Minixperiment.Client}
 */
export function createClient(options) {
    return new Client({
        defaults: {
            [NETWORK_PROFILE_COLLECTION]: 'no',
            [MARKERS]: 'no',
            [PINEAPPLE]: 'no',
            [HTML5_TOGGLE_REMOVAL]: 'keep',
            [PADTIME]: 'pad_control',
            [VIEW_WATCH_TOGETHER]: 'no',
            [CHOMPERZ]: 'control',
            [AUTO_SUGGEST_NOTIFICATION]: 'control',
            [REALTIME_QOS]: 'control',
            [MEDIAPLAYER_VERSION_CONTROL]: '2.1.0',
            [TWILIGHT_PREMIERE_UPLOAD_FLOW]: 'no',
            [LEVIATHAN]: 'no',
            [FAST_BREAD]: 'control',
        },
        deviceID: options.deviceID,
        overrides: assign({
            [MARKERS]: staffOnlyOverride('vca'),
            [PINEAPPLE]: staffOnlyOverride('yes'),
            [VIEW_WATCH_TOGETHER]: staffOnlyOverride('yes'),
            [AUTO_SUGGEST_NOTIFICATION]: staffOnlyOverride('show'),
            [MEDIAPLAYER_VERSION_CONTROL]: staffOnlyOverride('2.0.0'),
            [TWILIGHT_PREMIERE_UPLOAD_FLOW]: staffOnlyOverride('yes'),
            [CLIPS_GOT_EM]: 'variant1',
        }, JSON.parse(cookie.get('experiment_overrides', '{}'))),
        platform: 'web',
        // TODO do this correctly
        login: options.login,
        provider: new ServiceProvider(ServiceProvider.SERVICE_URL),
        Promise: Promise,
    });
}
