import { subscribe } from './util/subscribe';
import { fetchFollowInfo, followInfoFetched } from './actions/follow';
import { PLATFORM_MOBILE_WEB } from 'state/env';
import { OFFLINE_STATUS } from './state/online-status';
import { LOGGED_IN } from './state/user';
import includes from 'lodash/includes';

const CHANNEL_BLACKLIST = [
    'amazon_livestream',
];

function normalizeChannelName(channelName) {
    if (typeof channelName !== 'string') {
        return '';
    }
    return channelName.trim().toLowerCase();
}

export class FollowManager {
    constructor(store) {
        this.store = store;
        this.unsubs = [];
        this._initSubscribes();
    }

    _initSubscribes() {
        this.unsubs.push(subscribe(this.store, ['onlineStatus'], this.onOnlineStatusChange.bind(this)));
        // eslint-disable-next-line max-len
        this.unsubs.push(subscribe(this.store, ['playback.ended', 'playback.hasPlayed'], this.onPlaybackStateChange.bind(this)));
    }

    onOnlineStatusChange({ onlineStatus }) {
        if (onlineStatus !== OFFLINE_STATUS) {
            return;
        }

        const { playback } = this.store.getState();

        if (!playback.hasPlayed) {
            this._fetchFollowInfo();
        }
    }

    onPlaybackStateChange({ playback }) {
        if (!playback.ended) {
            return;
        }

        const { onlineStatus } = this.store.getState();

        if (onlineStatus === OFFLINE_STATUS && playback.hasPlayed) {
            this._fetchFollowInfo();
        }
    }

    _getIDs() {
        const { streamMetadata, user } = this.store.getState();

        return {
            userId: user.id,
            channelId: streamMetadata.channel.id,
            loggedIn: (user.loggedInStatus === LOGGED_IN),
        };
    }

    _checkChannelBlackList() {
        const { streamMetadata } = this.store.getState();
        if (!streamMetadata.channelName) {
            return false;
        }
        return includes(CHANNEL_BLACKLIST, normalizeChannelName(streamMetadata.channelName));
    }

    _fetchFollowInfo() {
        const { env } = this.store.getState();
        const { userId, channelId, loggedIn } = this._getIDs();
        const userOnOwnChannel = (userId === channelId);
        const blacklistedChannel = this._checkChannelBlackList();
        const isMobile = (env.platform === PLATFORM_MOBILE_WEB);

        if (
            userOnOwnChannel ||
            !channelId ||
            blacklistedChannel ||
            isMobile
        ) {
            return;
        }

        if (!loggedIn && channelId) {
            this.store.dispatch(followInfoFetched({
                following: false,
                notificationsEnabled: false,
            }));
            return;
        }

        this.store.dispatch(fetchFollowInfo(userId, channelId));
    }

    destroy() {
        this.unsubs.forEach(unsub => unsub());
    }
}
