import { subscribe } from './util/subscribe';

import isFinite from 'lodash/isFinite';
import findIndex from 'lodash/findIndex';
import * as PlayerType from './util/player-type';
import * as Settings from './settings';
import { closeCollectionSidebar } from './actions/collection';
import { toggleFullScreen, setFullScreen, setTheatreMode } from './actions/screen-mode';
import { changePlaybackRate, changeVolume, play, pause } from './actions/video-api';
import { SIDEBAR_VIEW } from './state/collection';
import { PLAYBACK_SPEEDS } from './ui/components/settings/playback-speed-menu';
import includes from 'lodash/includes';

const KEY_CTRL = 'Control';
const KEY_SPACE = ' ';
const KEY_PGUP = 'PageUp';
const KEY_PGDOWN = 'PageDown';
const KEY_ESC = 'Escape';
const KEY_UP = 'ArrowUp';
const KEY_DOWN = 'ArrowDown';
const KEY_LEFT = 'ArrowLeft';
const KEY_RIGHT = 'ArrowRight';
const KEY_F = 'f';
const KEY_F_CAP = 'F';
const MINIMUM_VOD_TIME = 0.01;
const KEY_SPACEBAR = 'Spacebar';
const KEY_COMMA = ',';
const KEY_PERIOD = '.';

// IE9 and FF36 emit alternative events
const KEY_ESC_COMPAT = 'Esc';
const KEY_UP_COMPAT = 'Up';
const KEY_DOWN_COMPAT = 'Down';
const KEY_LEFT_COMPAT = 'Left';
const KEY_RIGHT_COMPAT = 'Right';

// Safari 10 doesn't support `key`--convert key code to key value
const KEY_CODE_TO_KEY = Object.freeze({
    17: KEY_CTRL,
    32: KEY_SPACE,
    33: KEY_PGUP,
    34: KEY_PGDOWN,
    27: KEY_ESC,
    38: KEY_UP,
    40: KEY_DOWN,
    37: KEY_LEFT,
    39: KEY_RIGHT,
    70: KEY_F,
    188: KEY_COMMA,
    190: KEY_PERIOD,
});

function _getKeyFromEvent(e) {
    const deprecatedKeyCode = e.which || e.keyCode || e.charCode;
    return e.key ? e.key : KEY_CODE_TO_KEY[deprecatedKeyCode];
}

export class PlayerHotkeys {
    constructor(player, root, store, options) {
        this.ctrl = false;
        this.store = store;
        this.player = player;
        this.root = root;
        this.unsubs = [];
        this._initSubscribes();

        // eslint-disable-next-line max-statements
        this.rootKeydownListener = e => {
            const target = e.target || e.srcElement;
            const isTextInput = (target.tagName === 'INPUT' && target.getAttribute('type') === 'text');
            const isTextArea = (target.tagName === 'TEXTAREA');
            const isContentEditable = (target.contentEditable === 'true');
            const isMiniPlayer = store.getState().ui.isMini;
            const disableKeyboardShortcuts = (isTextInput || isTextArea || isContentEditable || isMiniPlayer);

            if (disableKeyboardShortcuts) {
                return; // in an input or textarea, should not use keyboard shortcuts
            }

            this.keydownHandler(e);
        };

        this.rootKeyupListener = e => {
            const key = _getKeyFromEvent(e);

            if (key === KEY_CTRL) {
                this.ctrl = false;
            }
        };

        root.setAttribute('tabindex', options.tabindex || -1);
        root.addEventListener('keydown', this.rootKeydownListener);
        root.addEventListener('keyup', this.rootKeyupListener);

        // Initialize handlers
        this.updateHandlers(store.getState());
    }

    _initSubscribes() {
        this.unsubs.push(subscribe(this.store, ['env.playerType'], this.updateHandlers.bind(this)));
    }

    updateHandlers({ env }) {
        const { playerType } = env;

        if (includes(PlayerType.CLIPS_PLAYER_TYPES, playerType)) {
            this.keydownHandler = this.handleClipsViewingHotkeys;
        } else {
            this.keydownHandler = this.handlePlayerHotkeys;
        }
    }

    seekSeconds(seconds) {
        const { store, player } = this;

        const channel = player.getChannel();
        const currentTime = player.getCurrentTime();
        const { duration } = store.getState().playback;

        // Ignore live streams, non-seekable streams, and paused streams.
        if (channel || !isFinite(duration) || player.getPaused()) {
            return;
        }

        var targetTime = this.calculateTargetTime(
            currentTime,
            seconds,
            MINIMUM_VOD_TIME,
            duration,
            player.getEnded()
        );

        if (targetTime) player.setCurrentTime(targetTime);
    }

    calculateTargetTime(currentTime, seconds, min, max, ended) {
        // Ended VODs have currentTime equal to the last seeked time
        const value = ended ? max + seconds : currentTime + seconds;
        if (value < min) {
            return min;
        } else if (value > max) {
            return null;
        }
        return value;
    }

    togglePlayback() {
        const { store } = this;

        if (store.getState().playback.paused) {
            store.dispatch(play());
        } else {
            store.dispatch(pause());
        }
    }

    changeVolumeByStep(increment = true) {
        const { store } = this;

        const currentVolume = store.getState().playback.volume;
        const stepAmount = Settings.volumeStepAmount;

        let newVolume;
        if (increment) {
            newVolume = Math.round(Math.min(currentVolume + stepAmount, 1.0) * 100) / 100;
        } else {
            newVolume = Math.round(Math.max(currentVolume - stepAmount, 0.0) * 100) / 100;
        }

        store.dispatch(changeVolume(newVolume));
    }

    updatePlaybackRate(increment = true) {
        const { store } = this;

        const currentPlaybackRate = store.getState().playback.playbackRate;
        const currentIndex = findIndex(PLAYBACK_SPEEDS, playbackRate => playbackRate === currentPlaybackRate);

        let newPlaybackRate = currentPlaybackRate;
        if (increment && currentIndex < PLAYBACK_SPEEDS.length - 1) {
            newPlaybackRate = PLAYBACK_SPEEDS[currentIndex + 1];
        } else if (!increment && currentIndex > 0) {
            newPlaybackRate = PLAYBACK_SPEEDS[currentIndex - 1];
        }

        if (currentPlaybackRate !== newPlaybackRate) {
            store.dispatch(changePlaybackRate(newPlaybackRate));
        }
    }

    // eslint-disable-next-line complexity, max-statements
    handlePlayerHotkeys(e) {
        const { store, player } = this;

        const key = _getKeyFromEvent(e);

        switch (key) {
        case KEY_CTRL:
            this.ctrl = true;
            break;
        case KEY_SPACE:
        case KEY_SPACEBAR:
            this.togglePlayback();
            break;
        case KEY_PGUP:
            // Page up/down toggle muted.
            player.setMuted(false);
            break;
        case KEY_PGDOWN:
            player.setMuted(true);
            break;
        case KEY_UP:
        case KEY_UP_COMPAT:
            this.changeVolumeByStep(true);
            break;
        case KEY_DOWN:
        case KEY_DOWN_COMPAT:
            this.changeVolumeByStep(false);
            break;
        case KEY_ESC:
        case KEY_ESC_COMPAT:
            if (store.getState().collection.currentView === SIDEBAR_VIEW) {
                store.dispatch(closeCollectionSidebar());
            } else if (PlayerType.isEmbed() && store.getState().screenMode.isTheatreMode) {
                // Only exit theatre mode for embedded players.
                // Web-client handles this for the main site.
                // Remove embed check when Twilight goes live [VP-2499].
                store.dispatch(setTheatreMode(false));
            }
            break;
        case KEY_F:
        case KEY_F_CAP:
            if (!this.ctrl) {
                return;
            }
            // ctrl+f toggles fullscreen.
            store.dispatch(toggleFullScreen());
            break;
        case KEY_RIGHT:
        case KEY_RIGHT_COMPAT:
            this.seekSeconds(Settings.hotkeySeekAmount);
            break;
        case KEY_LEFT:
        case KEY_LEFT_COMPAT:
            this.seekSeconds(-Settings.hotkeySeekAmount);
            break;
        default:
            // Avoid calling preventDefault if it wasn't a hotkey.
            return;
        }

        e.preventDefault();
    }

    // eslint-disable-next-line complexity, max-statements
    handleClipsViewingHotkeys(e) {
        const { store } = this;
        const key = _getKeyFromEvent(e);

        switch (key) {
        case KEY_CTRL:
            this.ctrl = true;
            break;
        case KEY_SPACE:
        case KEY_SPACEBAR:
            this.togglePlayback();
            break;
        case KEY_UP:
        case KEY_UP_COMPAT:
            this.changeVolumeByStep(true);
            break;
        case KEY_DOWN:
        case KEY_DOWN_COMPAT:
            this.changeVolumeByStep(false);
            break;
        case KEY_ESC:
        case KEY_ESC_COMPAT:
            setFullScreen(false);
            break;
        case KEY_F:
        case KEY_F_CAP:
            store.dispatch(toggleFullScreen());
            break;
        case KEY_RIGHT:
        case KEY_RIGHT_COMPAT:
            this.seekSeconds(Settings.clipsHotkeySeekAmount);
            break;
        case KEY_LEFT:
        case KEY_LEFT_COMPAT:
            this.seekSeconds(-Settings.clipsHotkeySeekAmount);
            break;

        case KEY_COMMA:
            this.updatePlaybackRate(false);
            break;

        case KEY_PERIOD:
            this.updatePlaybackRate(true);
            break;

        default:
            // Avoid calling preventDefault if it wasn't a hotkey.
            return;
        }

        e.preventDefault();
    }

    destroy() {
        this.unsubs.forEach(unsub => unsub());
        this.root.removeEventListener('keyup', this.rootKeyupListener);
        this.root.removeEventListener('keydown', this.rootKeydownListener);
    }
}
