/* global I18N_LOCALES */
import i18next from 'i18next';
import XHR from 'i18next-xhr-backend';
import { playerHost as PLAYER_HOST } from '../settings';
import reduce from 'lodash/reduce';
import assign from 'lodash/assign';
import includes from 'lodash/includes';

const LANG_TO_COUNTRY_CODE = assign(reduce(I18N_LOCALES, (langMap, locale) => {
    const shortCode = locale.replace(/-[a-zA-Z]{2}$/, '');
    // eslint-disable-next-line no-param-reassign
    langMap[shortCode] = locale;
    return langMap;
}, {}), {
    // Default overrides for some languages that have multiple country codes
    es: 'es-ES',
    pt: 'pt-PT',
    zh: 'zh-CN',
});

export function missingKeyHandler(lang, ns, key) {
    // eslint-disable-next-line no-undef
    if (process.env.NODE_ENV === 'development') {
        // eslint-disable-next-line no-console
        console.error(`Tried to translate ${key} and failed: no such key`);
    }
}

i18next.use(XHR).init({
    fallbackLng: 'en-US',
    load: 'currentOnly',
    backend: {
        crossDomain: true,
        loadPath: `${PLAYER_HOST}/localization/{{lng}}/strings.json`,
    },
    nsSeparator: false,
    keySeparator: false,
    missingKeyHandler,
    saveMissing: true,
});

i18next.addResourceBundle(
    'en-US',
    i18next.options.ns[0],
    require('../../../localization/en-US/strings.json')
);

class I18N {
    constructor(shortCode, langCode, translate) {
        this.shortCode = shortCode;
        this.langCode = langCode;
        this._translate = translate;
    }

    translate(msgid, options = {}) {
        let translation = this._translate(msgid, options);
        if (options.replacements) {
            Object.keys(options.replacements).map(function(key) {
                translation = translation.replace(key, options.replacements[key]);
            });
        }

        return translation;
    }
}

/**
 * Retrieves an i18next translation function that represents the given language.
 *
 * @param {String} lang
 * @return {Promise<I18n>}
 */
export function getI18N(lang) {
    let language = lang.replace(/-[a-zA-Z]{2}$/, function(countryCode) {
        return countryCode.toUpperCase();
    });
    const shortCode = lang.replace(/-[a-zA-Z]{2}$/, '');

    if (!includes(I18N_LOCALES, language)) {
        language = LANG_TO_COUNTRY_CODE[shortCode] || shortCode;
    }

    return new Promise(function(resolve, reject) {
        i18next.changeLanguage(language, (err, t) => {
            if (err) {
                reject(err);
            } else {
                resolve(new I18N(shortCode, language, t));
            }
        });
    });
}

export const I18N_INSTANCE = i18next;
export const DEFAULT_LANGUAGE = new I18N('en', 'en-US', i18next.getFixedT('en-US'));
