import * as EventEmitterActions from 'actions/event-emitter';
import * as TwitchEvents from 'backend/events/twitch-event';
import * as MediaEvents from 'backend/events/media-event';
import { NullEventEmitter } from 'defaults/null-event-emitter';
import { FULLSCREEN_CHANGE } from 'util/fullscreen';
import { ACTION_SET_PERSISTENCE } from 'actions/settings';
import { ACTION_SET_THEATRE_MODE, ACTION_TOGGLE_THEATRE_MODE } from 'actions/screen-mode';
import { ACTION_PROMPT_LOGIN_MODAL } from 'actions/follow';

const NULL_EVENT_EMITTER = new NullEventEmitter();

/**
 * Creates Event Emitter middleware with a supplied null emitter object
 *
 * @param {Object} [eventEmitter=NULL_EVENT_EMITTER] - the event emitter to use
 */
export function createEventEmitterMiddleware(emitter = NULL_EVENT_EMITTER) {
    // eslint-disable-next-line no-unused-vars
    return store => next => {
        let _eventEmitter = emitter;

        // eslint-disable-next-line complexity
        return action => {
            const { screenMode } = store.getState();

            switch (action.type) {
            case EventEmitterActions.ACTION_USE_EVENT_EMITTER:
                _eventEmitter = action.emitter;
                break;
            case EventEmitterActions.ACTION_EMIT_OPEN_STREAM:
                _eventEmitter.emit(TwitchEvents.OPEN_STREAM);
                break;
            case EventEmitterActions.ACTION_TRANSITION_TO_COLLECTION:
                _eventEmitter.emit(TwitchEvents.TRANSITION_TO_COLLECTION_VOD, {
                    vodId: action.vodId,
                    collectionId: action.collectionId,
                });
                break;
            case EventEmitterActions.ACTION_EMIT_FULLSCREEN_CHANGE:
                _eventEmitter.emit(FULLSCREEN_CHANGE);
                break;
            case EventEmitterActions.ACTION_TRANSITION_TO_REC_VOD:
                _eventEmitter.emit(TwitchEvents.TRANSITION_TO_RECOMMENDED_VOD, {
                    vodId: action.vodId,
                });
                break;
            case ACTION_SET_PERSISTENCE:
                _eventEmitter.emit(TwitchEvents.PERSISTENT_PLAYER_TOGGLE);
                break;
            case ACTION_SET_THEATRE_MODE:
                if (screenMode.isTheatreMode === action.isTheatreMode) {
                    break;
                }

                _eventEmitter.emit(
                    TwitchEvents.THEATRE_CHANGE,
                    (screenMode.isFullScreen ? false : action.isTheatreMode)
                );
                break;
            case ACTION_TOGGLE_THEATRE_MODE:
                _eventEmitter.emit(
                    TwitchEvents.THEATRE_CHANGE,
                    (screenMode.isFullScreen ? false : !(screenMode.isTheatreMode))
                );
                break;
            case ACTION_PROMPT_LOGIN_MODAL:
                _eventEmitter.emit(TwitchEvents.PROMPT_LOGIN_MODAL, {
                    channelDisplayName: action.channelName,
                });
                break;
            case EventEmitterActions.ACTION_AD_PLAY:
                _eventEmitter.emit(MediaEvents.PLAY);
                break;
            case EventEmitterActions.ACTION_AD_PLAYING:
                _eventEmitter.emit(MediaEvents.PLAYING);
                break;
            case EventEmitterActions.ACTION_AD_PAUSE:
                _eventEmitter.emit(MediaEvents.PAUSE);
                break;
            case EventEmitterActions.ACTION_EMIT_PLAYER_READY:
                _eventEmitter.emit(TwitchEvents.PLAYER_READY);
                break;
            }

            next(action);
        };
    };
}
