import { setUIDisplay, showUIWithAutoHide, hoveringOffUIElement,
         ACTION_SHOW_UI_WITH_AUTO_HIDE, ACTION_HIDE_UI,
         ACTION_SET_MINIPLAYER_MODE } from 'actions/ui';
import { ACTION_PAUSE, ACTION_PLAYING, ACTION_PLAYER_SEEKING } from 'actions/playback';
import { ACTION_SET_THEATRE_MODE, ACTION_TOGGLE_THEATRE_MODE } from 'actions/screen-mode';
import { ACTION_SET_CASTING_STATE } from 'actions/chromecast';
import { SIDEBAR_VIEW } from 'state/collection';
import { OFFLINE_STATUS } from 'state/online-status';
import { ACTION_SET_CURRENT_QUALITY } from 'actions/quality';
import { initialControlsDelay, hoverControlsDelay } from 'settings';
import { PLAYER_HIGHLIGHTER } from 'util/player-type';
import includes from 'lodash/includes';

export function createUIMiddleware() {
    return store => next => {
        let hideTimer;
        const { window } = store.getState();

        const hideUICallback = () => {
            const { env, onlineStatus, playback, ui } = store.getState();
            const inHighlightPlayer = env.playerType === PLAYER_HIGHLIGHTER;
            const hoveringOnUIElement = ui.onUIElement;
            const paused = playback.paused;
            const offline = onlineStatus === OFFLINE_STATUS;
            if (
                !inHighlightPlayer &&
                !hoveringOnUIElement &&
                !paused &&
                !offline
            ) {
                store.dispatch(setUIDisplay(false));
            }
        };

        const clearTimer = () => {
            window.clearTimeout(hideTimer);
        };

        const resetTimer = delay => {
            clearTimer();
            hideTimer = window.setTimeout(hideUICallback, delay);
        };

        const forceHideUI = () => {
            // forcibly reset hovering state to off
            store.dispatch(hoveringOffUIElement());
            // hide display
            store.dispatch(setUIDisplay(false));
        };

        const showUIIfApplicable = () => {
            const { ui, collection } = store.getState();
            const inCollectionSidebarView = collection.currentView === SIDEBAR_VIEW;
            if (!ui.isMini && !inCollectionSidebarView) {
                store.dispatch(setUIDisplay(true));
            }
        };

        store.dispatch(showUIWithAutoHide());
        resetTimer(initialControlsDelay);
        // eslint-disable-next-line complexity
        return action => {
            switch (action.type) {
            case ACTION_SET_MINIPLAYER_MODE:
                if (action.value) {
                    forceHideUI();
                    clearTimer();
                }
                break;
            case ACTION_PAUSE:
                showUIIfApplicable();
                break;
            case ACTION_HIDE_UI: {
                const { env } = store.getState();
                if (env.playerType !== PLAYER_HIGHLIGHTER) {
                    forceHideUI();
                }
                clearTimer();
                break;
            }
            case ACTION_SET_CASTING_STATE:
                if (includes(['connected', 'connecting'], action.castingState)) {
                    store.dispatch(setUIDisplay(true));
                } else {
                    resetTimer(hoverControlsDelay);
                }
                break;
            case ACTION_SET_CURRENT_QUALITY:
                if (!action.isAuto) {
                    showUIIfApplicable();
                    resetTimer(hoverControlsDelay);
                }
                break;
            case ACTION_PLAYER_SEEKING:
            case ACTION_PLAYING:
            case ACTION_SET_THEATRE_MODE:
            case ACTION_SHOW_UI_WITH_AUTO_HIDE:
            case ACTION_TOGGLE_THEATRE_MODE:
                showUIIfApplicable();
                resetTimer(hoverControlsDelay);
                break;
            }

            next(action);
        };
    };
}
