import { pushScreen, popScreen, VOD_RECOMMENDATION_SCREEN } from './actions/screen';
import { OFFLINE_STATUS } from './state/online-status';
import { PLAYER_DASHBOARD, PLAYER_FRONTPAGE, PLAYER_CREATIVE,
         PLAYER_HIGHLIGHTER, PLAYER_AMAZON_VSE, PLAYER_AMAZON_LIVE, PLAYER_FEED } from './util/player-type';
import { CONTENT_MODE_LIVE } from './stream/twitch-live';
import { TRANSITION_TYPE_RECOMMENDATIONS } from './state/playback';
import { subscribe } from './util/subscribe';
import includes from 'lodash/includes';
import { CHANNEL_VODS, setRecommendedVODs, OFFLINE_RECOMMENDATIONS_TYPE,
         isWatched } from './actions/recommendations';
import { featuredCollectionFetched } from './actions/collection';
import { krakenRequestv5, getFeaturedCollection } from './api';
import { getResumeTimes } from './actions/resume-watch';
import { AdContentTypes } from './actions/ads';

// Do not show offline recommendations on these player types
const INVALID_PLAYER_TYPES = [
    PLAYER_DASHBOARD,
    PLAYER_FRONTPAGE,
    PLAYER_CREATIVE,
    PLAYER_HIGHLIGHTER,
    PLAYER_AMAZON_VSE,
    PLAYER_AMAZON_LIVE,
    PLAYER_FEED,
];

// Minimum player size in pixels to show offline recs
const MIN_PLAYER_WIDTH = 560;
const MIN_PLAYER_HEIGHT = 300;

// Do not show offline recommendations on these channels
const CHANNEL_ID_BLACKLIST = [
    152325025, // amazon_livestream
];

const NO_CHANNEL_VIDEOS_MESSAGE = 'Offline Recommendations: No channel videos';
const ALL_VIDEOS_WATCHED_MESSAGE = 'Offline Recommendations: All channel videos watched';
const STREAM_CHANGED_WHILE_FETCH_MESSAGE = 'Offline Recommendations: Stream changed while fetching';

const MAX_JITTER_TIMEOUT = 5000; // ms

export class OfflineRecommendationsManager {
    constructor(store) {
        this._store = store;
        this._unsubs = [];

        this._unsubs.push(subscribe(this._store, [
            'onlineStatus',
            'streamMetadata',
        ], this.tryFetchOfflineRecs.bind(this)));

        this._unsubs.push(subscribe(this._store, [
            'ads',
            'recommendations',
            'playback.ended',
        ], this.tryShowOfflineRecs.bind(this)));

        this._unsubs.push(subscribe(this._store, ['playerDimensions'], this.onDimensionsChange.bind(this)));
    }

    tryFetchOfflineRecs() {
        if (!this._shouldFetchOfflineRecs()) {
            return Promise.resolve();
        }

        const {
            stream: streamOnFetch,
            window: windowObject,
            playback,
        } = this._store.getState();

        const jitter = playback.hasPlayed ? Math.floor(Math.random() * MAX_JITTER_TIMEOUT) + 1 : 0;
        const delay = period => new Promise(resolve => windowObject.setTimeout(resolve, period));

        return delay(jitter).
            then(() => Promise.all([this._getChannelVideos(), this._getFeaturedCollection()])).
            then(recs => (recs[0].length !== 0) ? recs : Promise.reject(NO_CHANNEL_VIDEOS_MESSAGE)).
            then(([channelVideos, collections]) => {
                const { stream } = this._store.getState();
                if (stream !== streamOnFetch) {
                    return Promise.reject(STREAM_CHANGED_WHILE_FETCH_MESSAGE);
                }

                if (channelVideos.length > 0) {
                    this._store.dispatch(setRecommendedVODs(channelVideos, OFFLINE_RECOMMENDATIONS_TYPE));
                }

                if (collections.length > 0) {
                    this._store.dispatch(featuredCollectionFetched(collections[0]));
                }
            }).
            // eslint-disable-next-line no-console
            catch(error => console.warn(error));
    }

    tryShowOfflineRecs() {
        if (this._shouldShowOfflineRecs()) {
            this._store.dispatch(pushScreen(VOD_RECOMMENDATION_SCREEN));
        }
    }

    onDimensionsChange() {
        const { screen, playerDimensions } = this._store.getState();
        const { height, width } = playerDimensions;
        const isShowingRecs = screen[0] === VOD_RECOMMENDATION_SCREEN;
        const playerSizeTooSmall = height < MIN_PLAYER_HEIGHT || width < MIN_PLAYER_WIDTH;

        if (playerSizeTooSmall && isShowingRecs) {
            this._store.dispatch(popScreen());
            return;
        }

        if (this._shouldShowOfflineRecs()) {
            this._store.dispatch(pushScreen(VOD_RECOMMENDATION_SCREEN));
        }
    }

    _shouldShowOfflineRecs() {
        const {
            screen: screenStack,
            recommendations,
            playerDimensions,
            stream,
            playback,
            onlineStatus,
            ads,
        } = this._store.getState();

        const { height, width } = playerDimensions;
        const isValidPlayerSize = height >= MIN_PLAYER_HEIGHT && width >= MIN_PLAYER_WIDTH;
        const recScreenNotInStack = !includes(screenStack, VOD_RECOMMENDATION_SCREEN);
        const notShowingAds = ads.currentMetadata.contentType === AdContentTypes.NONE;
        const offlineRecommendationsFetched = recommendations.type === OFFLINE_RECOMMENDATIONS_TYPE;
        const isOfflineLiveStream = stream.contentType === CONTENT_MODE_LIVE && onlineStatus === OFFLINE_STATUS;
        const hasNotPlayed = !playback.hasPlayed;
        const hasEnded =  playback.ended;

        return (
            isValidPlayerSize &&
            recScreenNotInStack &&
            notShowingAds &&
            offlineRecommendationsFetched &&
            isOfflineLiveStream &&
            (hasNotPlayed || hasEnded)
        );
    }

    _shouldFetchOfflineRecs() {
        const {
            onlineStatus,
            playback,
            stream,
            streamMetadata,
            env,
        } = this._store.getState();

        const isLiveStream = stream.contentType === CONTENT_MODE_LIVE;
        const hasChannelID = streamMetadata.channel.id !== 0;
        const channelNotBlacklisted = !includes(CHANNEL_ID_BLACKLIST, streamMetadata.channel.id);
        const isOffline = onlineStatus === OFFLINE_STATUS;
        const isRecommendationTransitionScheme = playback.transitionScheme === TRANSITION_TYPE_RECOMMENDATIONS;
        const isValidPlayerType = !includes(INVALID_PLAYER_TYPES, env.playerType);

        return (
            isLiveStream &&
            hasChannelID &&
            channelNotBlacklisted &&
            isOffline &&
            isRecommendationTransitionScheme &&
            isValidPlayerType
        );
    }

    _getFeaturedCollection() {
        const { streamMetadata } = this._store.getState();

        return getFeaturedCollection(streamMetadata.channel.id).
            then(({ collections }) => collections).
            catch(error => {
                // eslint-disable-next-line no-console
                console.warn(error);
                return [];
            });
    }

    _getChannelVideos() {
        return this._fetchChannelVideos().
            then(videos => this._filterOutWatchedVideos(videos)).
            then(videos => this._normalizeChannelVideos(videos)).
            catch(error => {
                // eslint-disable-next-line no-console
                console.warn(error);
                return [];
            });
    }

    _fetchChannelVideos() {
        const { streamMetadata } = this._store.getState();

        return krakenRequestv5(`channels/${streamMetadata.channel.id}/videos?limit=30`).
            then(({ videos }) => (videos.length > 0) ? videos : Promise.reject(NO_CHANNEL_VIDEOS_MESSAGE));
    }

    _filterOutWatchedVideos(videos) {
        const { user } = this._store.getState();

        return getResumeTimes(user.id).
            catch(() => ({ videos: [] })). // if resume time retrieval fails, return empty resume times
            then(resumeTimes => {
                const unwatchedVideos = videos.filter(v => !isWatched(resumeTimes,v));
                return (unwatchedVideos.length > 0) ? unwatchedVideos : Promise.reject(ALL_VIDEOS_WATCHED_MESSAGE);
            });
    }

    _normalizeChannelVideos(videos) {
        return videos.map(video => {
            let thumbnailURL = '';
            if (video.thumbnails.medium.length > 0) {
                thumbnailURL = video.thumbnails.medium[0].url;
            }

            return {
                recommendationType: CHANNEL_VODS,
                creationDate: video.created_at,
                thumbnailURL: thumbnailURL,
                title: video.title,
                channelName: video.channel.display_name,
                duration: video.length,
                id: video._id,
                game: video.game,
            };
        });
    }

    destroy() {
        this._unsubs.forEach(unsub => unsub());
    }
}
