import ResizeObserver from 'resize-observer-polyfill';
import { updatePlayerDimensions } from './actions/player-dimensions';
import isElement from 'lodash/isElement';

export class PlayerResizeManager {
    constructor(store) {
        this._store = store;
        this.resizeObserver = this._createResizeObserver();
    }

    /**
     * Create the ResizeObserver to track the following elements:
     * { left, top, width, height }
     *
     * Player dimensions are then dispatched to state store.
     */
    _createResizeObserver() {
        return new ResizeObserver(entries => {
            const rootElement = entries[0];
            const { left, top, width, height } = rootElement.contentRect;
            this._store.dispatch(updatePlayerDimensions({
                left,
                top,
                width,
                height,
            }));
        });
    }

    /**
     * Observe the given element's dimensions.
     *
     * @param {String} root is the element to observe in player
     */
    observe(root) {
        this.observeElement = root;

        this._store.dispatch(updatePlayerDimensions({
            left: this.observeElement.getBoundingClientRect().left,
            top: this.observeElement.getBoundingClientRect().top,
            width: this.observeElement.getBoundingClientRect().width,
            height: this.observeElement.getBoundingClientRect().height,
        }));

        this.resizeObserver.observe(this.observeElement);
    }

    /**
     * Unobserve the given element's dimensions.
     *
     * @param {String} root is the element to unobserve in player
     */
    destroy() {
        if (isElement(this.observeElement)) {
            this.resizeObserver.unobserve(this.observeElement);
            this.resizeObserver.disconnect();
        }
    }
}
