require('es6-promise').polyfill();
import 'whatwg-fetch';

import defaults from 'lodash/defaults';
import includes from 'lodash/includes';
import isString from 'lodash/isString';
import pick from 'lodash/pick';
import { Dobbin } from 'dobbin';

import { AD_REQUEST_ERROR } from './analytics/spade-events';
import { Analytics } from './analytics/analytics';
import { AnalyticsTracker } from './analytics/tracker';
import * as sentinel from './analytics/sentinel';
import { AutoSuggestNotificationManager } from './auto-suggest-notification-manager';
import * as params from 'util/params';
import { isTwitchEmbed } from 'util/twitch-embed';
import * as Settings from './settings';
import { localStore } from 'util/storage';
import * as PlayerType from 'util/player-type';
import { cookie } from 'cookie_js';
import * as DeviceId from 'util/device-id';
import { State } from './state-tracker';
import * as Timestamp from 'util/timestamp';
import { Video } from './video';
import { forwardProperties } from 'util/forward-properties';
import { EmbedHost } from './embed/host';
import { PlayerHotkeys } from './hotkeys';
import { uiLoader } from './ui/ui-loader';
import { BackendPlayerCore, BACKEND_PLAYER_CORE } from './backend/player-core';
import { BackendHls, BACKEND_HLS } from './backend/hls';
import { BACKEND_FLASH } from './backend/flash';
import { BackendMediaPlayer, BACKEND_MEDIA_PLAYER } from './backend/mediaplayer';
import { PubSub } from './pubsub';
import { FullScreen } from 'util/fullscreen';
import { NoFullScreen } from 'util/no-fullscreen';
import { init as initStore } from './state';
import { PLATFORM_MOBILE_WEB } from 'state/env';
import { setPlayerOptions } from './actions/player-options';
import { setAdblockDetected, imaScriptLoaded } from './actions/ads';
import * as StreamActions from './actions/stream';
import { requestCollection, clearCollection } from './actions/collection';
import * as ExperimentActions from './actions/experiments';
import * as LanguageActions from './actions/lang';
import { initializeQuality } from './actions/quality';
import { setEnvironment, setPlayerType, setTwitchEverywhereParams } from './actions/env';
import { setWindow } from './actions/window';
import { setCanFullScreen } from './actions/screen-mode';
import { initVodResume } from './actions/resume-watch';
import { setAutoplay, setStartTime } from './actions/playback';
import { setAnalyticsTracker } from './actions/analytics-tracker';
import * as MediaEvents from './backend/events/media-event';
import { krakenRequestv5, setOAuthToken } from './api';
import { ClipGenerator } from './clip-generator';
import { clipGeneratorLoaded } from 'actions/clips';
import { userCanClipSelector } from './state/clips';
import { HAVE_CURRENT_DATA } from './backend/state/ready-state';
import { CONTENT_MODE_VOD } from './stream/twitch-vod';
import { setTrackingProperties } from './actions/tracking';
import { setMiniPlayerMode } from './actions/ui';
import { ONLINE_STATUS } from './state/online-status';
import { UNKNOWN as USER_STATUS_UNKNOWN, LOGGED_IN } from './state/user';
import { CollectionManager } from './collection-manager';
import { PlayerResizeManager } from './player-resize-manager';
import { FollowManager } from './follow-manager';
import { VodMidrollManager } from './vod-midroll-manager';
import { TwitchEverywhereManager } from './twitch-everywhere-manager';
import { TimelineMetadataManager } from './timeline-metadata/manager';
import { OfflineRecommendationsManager } from './offline-recommendations-manager';
import { UserManager } from './user-manager';
import { parseUri } from 'util/parseuri';
import { PROMPT_LOGIN_MODAL } from './backend/events/twitch-event';
import { fetchFollowInfo } from './actions/follow';
import { fetchDeadLtvStreamMetadata } from './actions/stream-metadata';
import { subscribe } from 'util/subscribe';
import { SubscribeOverlayManager } from 'subscribe-overlay-manager';
import { emitPlayerReady } from './actions/event-emitter';
import { fetchIMAScript } from 'util/ima';
import { PlayerUIResume } from './resume-vod';
import { setUsherHostOverride } from 'actions/usher';
import bowser from 'bowser';

Dobbin.configure({ Promise });

// eslint-disable-next-line no-undef, camelcase
__webpack_public_path__ = `${Settings.playerHost}/`;

require('../sass/player.sass');

let imaScriptPromise = Promise.resolve();
if (!window.google) {
    imaScriptPromise = fetchIMAScript();
}

const EMBED_WHITELIST = Object.freeze([
    PlayerType.PLAYER_IMDB,
    PlayerType.PLAYER_CURSE,
    PlayerType.PLAYER_SITE,
    PlayerType.PLAYER_AMAZON_LIVE,
    PlayerType.PLAYER_AMAZON_VSE,
    ...PlayerType.CLIPS_PLAYER_TYPES,
]);

const BOWSER_SAFARI_NAME = 'Safari';

export function Player(rootId, opts) {
    const self = this;

    /* eslint-disable prefer-const */
    let tracker;
    /* eslint-enable prefer-const */

    let fullscreenHelper;
    let playerUI;
    let state;
    let store;
    let video;
    let analytics;
    let embedHost;
    let clipGenerator;
    let timelineMetadata;
    let collectionManager;
    let playerResizeManager;
    let followManager;
    let resumeVod;

    const destructibles = [];

    // eslint-disable-next-line complexity, max-statements
    function init() {
        // Root DOM element.
        let root;

        if (typeof rootId === 'string' || rootId instanceof String) {
            // Support passing in an ID of the root.
            root = document.getElementById(rootId);
        } else {
            // We also support passing the root element
            root = rootId;
        }

        store = initStore();
        store.dispatch(setWindow(window));

        // Initialize the options to their defaults.
        const options = initDefaults(opts);
        store.dispatch(setPlayerOptions(options));
        setOAuthToken(options);

        if (options.debug && options.verbose) {
            store.subscribe(function() {
                // eslint-disable-next-line no-console
                console.debug('state change: %o', store.getState());
            });
        }

        if (options.force_manifest_node) {
            store.dispatch(setUsherHostOverride(options.force_manifest_node));
        }

        // Initialize the experiments service client
        store.dispatch(ExperimentActions.loadExperiments({
            login: cookie.get('login') || null,
            deviceID: DeviceId.get(false),
            userAgent: store.getState().window.navigator.userAgent,
        }));

        store.dispatch(LanguageActions.loadDefaultLang(
            store.getState().window.navigator,
            options.lang
        ));
        store.dispatch(setEnvironment({
            deviceId: DeviceId.get(false),
        }));
        store.dispatch(setPlayerType(options.player));
        store.dispatch(setTrackingProperties(options.tracking));

        destructibles.push(new PubSub(store, options));
        destructibles.push(new OfflineRecommendationsManager(store));
        destructibles.push(new UserManager(store));
        destructibles.push(new AutoSuggestNotificationManager(store));
        destructibles.push(new VodMidrollManager(store));
        destructibles.push(new SubscribeOverlayManager(store));

        // Create the analytics tracker for sending events to spade/mixpanel
        tracker = new AnalyticsTracker(store, options);
        store.dispatch(setAnalyticsTracker(tracker));
        store.dispatch(initializeQuality());

        _checkIMAScriptLoaded();

        // Create fullscreen utility
        fullscreenHelper = options.allowfullscreen ? new FullScreen(root, store) : new NoFullScreen(root, store);

        // Create timeline metadata service
        timelineMetadata = new TimelineMetadataManager(store);

        // Create collection service
        collectionManager = new CollectionManager(store);

        // Create the video object.
        video = new Video(root, store, options);

        // Create the state tracker, pass in backend.
        state = new State(video, store, tracker, options);
        if (options.debug) {
            window.state = state;
        }

        // Observe and store current player dimensions.
        playerResizeManager = new PlayerResizeManager(store);
        playerResizeManager.observe(root);

        // Manages in player follow/notification prompts
        followManager = new FollowManager(store);

        if (options.player === PlayerType.PLAYER_TWITCH_EVERYWHERE) {
            destructibles.push(new TwitchEverywhereManager(store));

            if (!options.targetOrigin) {
                // eslint-disable-next-line no-console
                console.error('Missing targetOrigin option');
            }

            store.dispatch(setTwitchEverywhereParams({
                targetOrigin: options.targetOrigin,
            }));
        }

        // Forward any calls on this Player to the Video object.
        //   ex. player.setVolume -> video.setVolume
        // TODO Explicitly define the interface instead of forwarding.
        forwardProperties(self, video);

        // Create the embed API that uses postMessage.
        embedHost = new EmbedHost(video, store, {
            origin: options.origin,
        });

        // Use a seperate module to track video events.
        analytics = new Analytics(video, tracker, state, store, options);

        store.dispatch(setAutoplay(options.autoplay));

        // Once the analytics module is ready, start playing the video.
        // We do this before the UI has loaded because 1) the UI is more
        // likely to contain a syntax error and 2) the UI is async anyway.
        initVideo(options);

        // Listens for key presses and sends commands to the player.
        destructibles.push(new PlayerHotkeys(video, root, store, options));

        if (options.controls) {
            clipGenerator = new ClipGenerator(state, store);
            store.dispatch(clipGeneratorLoaded(clipGenerator));

            // Create the HTML UI last.
            playerUI = uiLoader(video, root, store, clipGenerator, options);
        }

        resumeVod = new PlayerUIResume(video, state, store, options);

        setInitialStream(options).then(() => {
            store.dispatch(setTrackingProperties({
                // eslint-disable-next-line camelcase
                content: options.tt_content,
                // eslint-disable-next-line camelcase
                medium: options.tt_medium,
            }));
            // player is now ready for commands via API
            store.dispatch(emitPlayerReady());
        });

        store.dispatch(initVodResume());

        video.addEventListener(MediaEvents.LOADED_METADATA, () => {
            // iOS/safari requires the video tag to have `loadedmetadata` before knowing if fullscreen is
            // available. Calling setCanFullScreen here sets canFullScreen in store
            store.dispatch(setCanFullScreen(fullscreenHelper.canFullScreen()));
        });
    }

    function _checkIMAScriptLoaded() {
        const { window: windowObj } = store.getState();
        const { analyticsTracker } = store.getState();

        if (!windowObj.google) {
            imaScriptPromise.then(() => {
                store.dispatch(imaScriptLoaded(true));
            }).catch(() => {
                store.dispatch(imaScriptLoaded(false));
                analyticsTracker.trackEvent(AD_REQUEST_ERROR, { reason: 'ima load failed' });
            });
        } else if (windowObj.google) {
            store.dispatch(imaScriptLoaded(true));
        }
    }

    // eslint-disable-next-line complexity
    function setInitialStream(options) {
        if (PlayerType.CLIPS_PLAYER_TYPES.includes(options.player)) {
            return self.setClip(options.clip);
        } else if (options.customerId && options.contentId) {
            return self._setContent(options.customerId, options.contentId);
        } else if (options.stream && options.channelId) {
            return self._setLiveToVod(options.stream, options.channelId);
        } else if (options.channelId) {
            return krakenRequestv5(`channels/${options.channelId}`).then(data => {
                return self.setChannel(data.name);
            });
        } else if (options.channel) {
            return self.setChannel(options.channel);
        } else if (options.collection) {
            return setAdblockStatus().then(() => {
                store.dispatch(requestCollection(
                    options.collection,
                    options.video,
                    options.time,
                    true
                ));
            });
        } else if (options.video && options.time) {
            return self.setVideo(options.video, Timestamp.parse(options.time));
        } else if (options.video) {
            return self.setVideo(options.video);
        }
        return setAdblockStatus();
    }
    // Initialize the options to their defaults.
    // eslint-disable-next-line complexity, max-statements
    function initDefaults(opts = {}) {
        let options = isString(opts) ? params.parse(opts) : opts;

        setForceAdOptions(options);

        if (!isTwitchEmbed()) {
            const UNBRANDED = [
                PlayerType.PLAYER_CURSE,
                PlayerType.PLAYER_SITE,
                PlayerType.PLAYER_FRONTPAGE,
                PlayerType.PLAYER_AMAZON_VSE,
                PlayerType.PLAYER_FEED,
            ];

            if (!includes(EMBED_WHITELIST, options.player)) {
                // for non-Twitch embeds, accept only a whitelist of properties
                options = pick(options, Settings.embedParameters);
            }

            // Force branding unless explicitly listed as an unbranded player type
            options.branding = !includes(UNBRANDED, options.player);
        }

        options.backend = getBackendTypeFromOptions(options);

        // Don't allow full screen if allowfullscreen in query parameters is set to 'false'
        // or player is a highlighter
        // eslint-disable-next-line max-len
        options.allowfullscreen = (options.allowfullscreen === false || options.player === PlayerType.PLAYER_HIGHLIGHTER) ? false : true;

        // Rename some parameters for backwards compatibility.
        options = defaults(options, {
            time: options.t,
        });

        options.time = params.convertUrlQueryParamTime(options.time);

        // Reroute people on player-core to mediaplayer
        // TODO: Remove in a few months after most users' local stores
        // have been transitioned. JISA Issue: VP-3010
        if (localStore.get('backend') === BACKEND_PLAYER_CORE) {
            localStore.set('backend', BACKEND_MEDIA_PLAYER);
        }

        // Start by using localstorage values if available.
        options = defaults(options, {
            volume: localStore.get('volume'),
            muted: localStore.get('muted'),
            backend: localStore.get('backend'),
            player: (options.player === PlayerType.PLAYER_IMDB ?
                PlayerType.PLAYER_IMDB : PlayerType.getPlayerType()),
        });

        // Some default values.
        options = defaults(options, {
            // Backwards compatibility until we can change web.
            showInfo: options.channelInfo,
        },{
            showInfo: true,
            branding: true,
            volume: 0.5,
            muted: false,
            controls: true,
            autoplay: true,
            playsinline: false,
            quality: 'medium',
            time: null,

            leaveDialogEnabled: Settings.leaveDialog.enabled,
            leaveDialogViewerThreshold: Settings.leaveDialog.viewerThreshold,
            refreshWarningEnabled: Settings.leaveDialog.enabled,

            lang: Settings.defaultLanguage,

            showtheatre: false,

            abs: false,

            origin: '',

            // eslint-disable-next-line camelcase
            tt_content: '',
            // eslint-disable-next-line camelcase
            tt_medium: '',
        });

        // Tracked here in VP-2836. bowser.name === BOWSER_SAFARI_NAME
        // temporarily mutes autoplay on desktop safari until
        // VP-2704's autoplay fix is deployed
        const isMobileWeb = store.getState().env.platform === PLATFORM_MOBILE_WEB;
        const isSafariBrowser = bowser.name === BOWSER_SAFARI_NAME;
        if (
            (isMobileWeb || isSafariBrowser) &&
            options.autoplay
        ) {
            options.muted = true;
        }

        // Add the player to the tracking events too.
        options.tracking = defaults({}, options.tracking, options.trackingProperties, {
            player: options.player,
        });

        return options;
    }

    function getBackendTypeFromOptions(options) {
        if (options.flash) {
            return BACKEND_FLASH;
        } else if (options.hls) {
            return BACKEND_HLS;
        }

        // The HTML5 option uses either mediaplayer, playercore, or HLS if supported.
        const mediaPlayerAvailable = BackendMediaPlayer.canPlay();
        const playerCoreAvailable = BackendPlayerCore.canPlay();
        const hlsAvailable = BackendHls.canPlay();
        if (options.html5 && mediaPlayerAvailable) {
            return BACKEND_MEDIA_PLAYER;
        } else if (options.html5 && playerCoreAvailable) {
            return BACKEND_PLAYER_CORE;
        } else if (options.html5 && hlsAvailable) {
            return BACKEND_HLS;
        }
    }

    // Set up the video given parameters.
    function initVideo(options) {
        video.setVolume(options.volume);
        video.setMuted(options.muted);

        if (options.debug) {
            Settings.allEvents.forEach(function(name) {
                if (!includes(Settings.debugIgnoreEvents, name)) {
                    video.addEventListener(name, function() {
                        // eslint-disable-next-line no-console
                        console.log('video event: ', name);
                    });
                }
            });
        }

        video.addEventListener(MediaEvents.ERROR, function(err) {
            // eslint-disable-next-line no-console
            console.error('video error:', video.getError() || err);
        });
    }

    /**
     * Gets the status of adblock, using the Sentinel service. Defaults to the
     * existing detected value, if one can be found on the page, or requests
     * Sentinel directly.
     *
     * @return {Promise<Boolean>}
     */
    function getAdblockStatus() {
        const { window } = store.getState();
        if (window.Twitch.sentinel) {
            // TODO: VP-2455 - hack where window.Twitch.sentinel.detect
            // hangs indefinitely on IE11.
            if (bowser.msie) {
                return Promise.resolve(true);
            }

            return Promise.resolve(window.Twitch.sentinel.detect);
        }
        return sentinel.getSentinel(window);
    }

    function setAdblockStatus() {
        return getAdblockStatus().
            then(status => store.dispatch(setAdblockDetected(status)),
                // if getAdblockStatus fails, we'll assume they have adblock
                () => store.dispatch(setAdblockDetected(true)));
    }

    /**
     * Copy force ads URL parameters into options
     * @param {Object} options
     * @returns {Object}
     */
    function setForceAdOptions(options) {
        const url = parseUri(store.getState().window.location.href);

        /* eslint-disable camelcase, no-param-reassign */
        if (url.queryKey.hasOwnProperty('force_preroll')) {
            options.usher_force_preroll = true;
            options.force_preroll = url.queryKey.force_preroll;
        }

        if (url.queryKey.hasOwnProperty('force_midroll')) {
            options.usher_force_midroll = true;
            options.force_midroll = url.queryKey.force_midroll;
        }

        if (url.queryKey.hasOwnProperty('force_preroll_id')) {
            options.force_preroll_id = url.queryKey.force_preroll_id;
        }

        if (url.queryKey.hasOwnProperty('force_midroll_id')) {
            options.force_midroll_id = url.queryKey.force_midroll_id;
        }
        /* eslint-enable camelcase, no-param-reassign */
    }

    /*
     * Given a stream (broadcastId from master manifest) and channel id,
     * show the live stream if it is still live. If the stream is over,
     * show the related vod if it was archived. If there is no vod,
     * fallback to showing a notifications/follow screen for the channel.
     *
     * @param {string} broadcastId
     * @param {string} channelId
     */
    self._setLiveToVod = function(broadcastId, channelId) {
        const notificationScreenFallback = channelName => {
            const { user } = store.getState();
            const fetchFollows = () => {
                const { user } = store.getState();
                store.dispatch(fetchFollowInfo(user.id, channelId));
            };

            store.dispatch(StreamActions.clearStream());
            store.dispatch(fetchDeadLtvStreamMetadata(channelName));

            if (user.loggedInStatus !== USER_STATUS_UNKNOWN) {
                fetchFollows();
            } else {
                const unsubscribe = subscribe(store, ['user.loggedInStatus'], function({ user }) {
                    unsubscribe();
                    if (user.loggedInStatus === LOGGED_IN) {
                        fetchFollows();
                    }
                });
            }
        };

        return krakenRequestv5(`content?stream_id=${broadcastId}&channel_id=${channelId}`).then(resp => {
            if (resp.stream && resp.stream._id.toString() === broadcastId) {
                // stream is still live
                return self.setChannel(resp.channel.name);
            } else if (resp.videos.length > 0) {
                const archivedVideo = resp.videos.find(function(vod) {
                    return vod.broadcast_type === 'archive';
                });

                const highlightVideo = resp.videos.find(function(vod) {
                    return vod.broadcast_type === 'highlight';
                });

                if (archivedVideo) {
                    // stream is over and there's an archived vod
                    return self.setVideo(archivedVideo._id);
                } else if (highlightVideo) {
                    // stream is over and there's a highlight vod
                    return self.setVideo(highlightVideo._id);
                }
            }
            // stream is no longer live and vod is not found
            // so we display notifications/follow pane
            notificationScreenFallback(resp.channel.name);
        }).catch(() => {
            // if live to vod api fails for some reason, fall back to follow pane
            return krakenRequestv5(`channels/${channelId}`).then(resp => {
                notificationScreenFallback(resp.name);
            });
        });
    };

    self._setContent = function(customerId, contentId) {
        const action = StreamActions.setStream({
            contentType: StreamActions.TYPE_PROVIDED,
            customerId,
            contentId,
        });
        return Promise.resolve(store.dispatch(action));
    };

    self.setChannel = function(value) {
        return setAdblockStatus().
            then(() => {
                const action = StreamActions.setStream({
                    contentType: StreamActions.TYPE_CHANNEL,
                    contentId: value,
                });
                store.dispatch(action);
                store.dispatch(clearCollection());
            });
    };

    self.setClip = function(value) {
        // not setting adblock status since we are not showing ads and it reduces latency
        const action = StreamActions.setStream({
            contentType: StreamActions.TYPE_CLIP,
            contentId: value,
        });
        store.dispatch(action);
        store.dispatch(clearCollection());

        return Promise.resolve(true);
    };

    self.setVideo = function(videoId, timestamp) {
        if (typeof videoId !== 'string') {
            return;
        }

        const normalizedVideoId = videoId.charAt(0) !== 'v' ? `v${videoId}` : videoId;

        return setAdblockStatus().
            then(() => {
                const action = StreamActions.setStream({
                    contentType: StreamActions.TYPE_VIDEO,
                    contentId: normalizedVideoId,
                });
                store.dispatch(action);
                store.dispatch(clearCollection());
                store.dispatch(setStartTime(timestamp));
            });
    };

    /**
     * If allowed, alert the collection-manager to attempt loading a collection with an optional starting vod
     *
     * @param {string} collection id
     * @param {string} video id
     * @param {number} timestamp
     * @param {boolean} preferVideo - loads first video of collection if specified vod is not in collection
     */
    self.setCollection = function(
        collectionId = '',
        videoId = '',
        timestamp
    ) {
        return setAdblockStatus().then(() => {
            store.dispatch(requestCollection(
                collectionId,
                videoId,
                timestamp
            ));
        });
    };

    self.setLanguage = function(language) {
        store.dispatch(LanguageActions.loadLanguage(language));
    };

    self.setMiniPlayerMode = function(value) {
        store.dispatch(setMiniPlayerMode(value));
    };

    self.recordClip = function() {
        if (clipGenerator && userCanClipSelector(store.getState())) {
            clipGenerator.recordClip();
        }
    };

    // TODO: this method is a mess. Revisit when the playing status becomes available in store
    self.isPlaying = function() {
        const { stream, onlineStatus } = store.getState();
        const isVODStream = stream.contentType === CONTENT_MODE_VOD;
        const isOnlineLiveStream = onlineStatus === ONLINE_STATUS;

        return (
            !video.getPaused() &&
            !video.getEnded() &&
            (isOnlineLiveStream || isVODStream) &&
            video.getReadyState() > HAVE_CURRENT_DATA
        );
    };

    self.setTrackingProperties = function(properties) {
        store.dispatch(setTrackingProperties(properties));
    };

    init();

    self.destroy = function() {
        resumeVod.destroy();
        if (playerUI) {
            playerUI.destroy();
        }
        video.destroy();
        state.destroy();
        analytics.destroy();
        embedHost.destroy();
        destructibles.forEach(obj => obj.destroy());
        timelineMetadata.destroy();
        collectionManager.destroy();
        playerResizeManager.destroy();
        followManager.destroy();
    };
}

window.Twitch = window.Twitch || {};
window.Twitch.video = window.Twitch.video || {};
window.Twitch.video.Player = Player;
window.Twitch.Player = Player;
window.Twitch.Player.PROMPT_LOGIN = PROMPT_LOGIN_MODAL;
