import pull from 'lodash/pull';
import PubSubClient from 'pubsub-js-client';
import { subscribe } from './util/subscribe';
import { updateViewerCount } from './actions/viewercount';
import { setOnline } from './actions/online';
import { setStream, TYPE_CHANNEL } from './actions/stream';
import { updateWatchPartyVod } from './actions/watch-party';
import { CONTENT_MODE_LIVE } from './stream/twitch-live';

const PUBSUB_ENVIRONMENT = 'production';

export const EVENT_ONLINE = 'stream-up';
export const EVENT_OFFLINE = 'stream-down';

/**
 * PubSub adapter to mediate state changes in the application and messages from
 * the PubSub service.
 */
export class PubSub {
    constructor(store, options = {}) {
        this._store = store;
        this._debugMode = options.debug;
        this._pubSub = PubSubClient.getInstance(PUBSUB_ENVIRONMENT);
        this._topics = [];
        this._onMessage = this._onPubSubMessage.bind(this);

        this._unsubscribe = this._subscribe();
    }

    /**
     * Destroy the PubSub client.
     */
    destroy() {
        this._unsubscribe();

        this._topics.forEach(topic => this._unlisten(topic));
    }

    /**
     * Called on new message from Twitch PubSub connection.
     * @private
     *
     * @param {Object} msg
     */
    _onPubSubMessage(msgString) {
        let msg;
        try {
            msg = JSON.parse(msgString);
        } catch (e) {
            // eslint-disable-next-line no-console
            console.error('Pubsub: %o', e);
            return;
        }

        if (this._debugMode) {
            // eslint-disable-next-line no-console
            console.info('PubSub: %o', msg);
        }

        switch (msg.type) {
        case 'stream-up':
            this._store.dispatch(setStream({
                contentType: TYPE_CHANNEL,
                contentId: this._store.getState().stream.channel,
            }));
            break;

        case 'stream-down':
            this._store.dispatch(setOnline(false));
            break;

        case 'viewcount':
            this._store.dispatch(updateViewerCount(msg.viewers));
            break;

        case 'tos-strike':
            // When a channel is flagged for Terms of Service violation, refresh the page
            this._store.getState().window.document.location.reload();
            break;

        case 'watchparty-vod':
            // When a watchparty is playing, we will need to update the vod.
            this._store.dispatch(updateWatchPartyVod(msg.vod));
            break;

        default:
            if (this._debugMode) {
                // eslint-disable-next-line no-console
                console.warn('PubSub: Unrecognized message %o', msg);
            }
        }
    }

    /**
     * Subscribes to meaningful changes in the state store, to keep the adapter
     * in sync with the application.
     * @private
     *
     * @return {Function} unsubscribe function for this store subscription
     */
    _subscribe() {
        return subscribe(this._store, ['streamMetadata.channel.id'], (newProps, oldState) => {
            const oldChannelId = oldState.streamMetadata.channel.id;
            const oldChannelName = oldState.streamMetadata.channelName;
            const newChannelId = newProps.streamMetadata.channel.id;
            const newStream = this._store.getState().stream;

            if (oldChannelId && oldChannelName) {
                // if we're moving away from a valid channel, call unlisten
                this._unlisten(channelTopic(oldChannelId));
            }
            if (newStream.contentType === CONTENT_MODE_LIVE && newChannelId) {
                // if we're a livestream and the id property has changed a non-empty value, then listen
                this._listen(channelTopic(newChannelId));
            }
        });
    }

    /**
     * Registers a listener with the PubSub client for the given topic.
     * @private
     *
     * @param {String} topic
     */
    _listen(topic) {
        this._topics.push(topic);

        // eslint-disable-next-line new-cap
        this._pubSub.Listen({
            topic,
            success: () => {
                if (this._debugMode) {
                    // eslint-disable-next-line no-console
                    console.info('PubSub: successfully listened to %s', topic);
                }
            },
            failure: err => {
                this._topics = pull(this._topics, topic);
                // eslint-disable-next-line no-console
                console.error('PubSub: error listening: %o', err);
            },
            message: this._onMessage,
        });
    }

    /**
     * Unregisters a listener on a topic with the PubSub client.
     * @private
     *
     * @param {String} topic
     */
    _unlisten(topic) {
        this._topics = pull(this._topics, topic);

        // eslint-disable-next-line new-cap
        this._pubSub.Unlisten({
            topic,
            success: () => {
                if (this._debugMode) {
                    // eslint-disable-next-line no-console
                    console.info('PubSub: successfully unlistened to %s', topic);
                }
            },
            failure: err => {
                this._topics.push(topic);
                // eslint-disable-next-line no-console
                console.error('PubSub: error unlistening: %o', err);
            },
            message: this._onMessage,
        });
    }
}

/**
 * Convert a channel name into the PubSub topic on which updates are broadcast.
 *
 * @param {String} channelName
 * @return {String}
 */
function channelTopic(channelId) {
    return `video-playback-by-id.${channelId}`;
}
