import isNumber from 'lodash/isNumber';
import { cancelResumeAmount as CANCEL_RESUME_AMOUNT,
         livestreamResumePushback as LIVESTREAM_RESUME_PUSHBACK } from './settings';
import { EVENT_PLAYER_UPDATE } from './state-tracker';
import * as MediaEvents from './backend/events/media-event';
import { setVodResumeTime, cancelVodResumeTime, setResumeTimes,
         setLivestreamResumeTime, cancelLivestreamResumeTime,
         getResumeTimes, setIsSeeked } from './actions/resume-watch';
import { ONLINE_STATUS } from './state/online-status';
import { videoInfo }  from './api';
import { CONTENT_MODE_LIVE } from './stream/twitch-live';
import { CONTENT_MODE_VOD } from './stream/twitch-vod';
import { UIStateSubscriber } from './ui/state-subscriber';

export class PlayerUIResume extends UIStateSubscriber {
    constructor(player, state, store, options) {
        super();
        this._state = state;
        this._store = store;
        this._player = player;
        this._options = options;

        // HOTFIX: master manifest data from getVideoInfo should live on store.
        this._streamTimeOffset = 0;

        this._state.addEventListener(EVENT_PLAYER_UPDATE, this.handleEvent.bind(this));
        this.subscribe(this._store, ['playback.currentTime'], this._onTimeUpdate.bind(this));
    }

    handleEvent(event) {
        switch (event) {
        case MediaEvents.LOADED_METADATA:
            this._onLoadedMetadata();
            break;
        }
    }

    _seekToResumeTime(videoInfo, watchedVODs) {
        const { stream, playback } = this._store.getState();
        const watchedTimes = watchedVODs.filter(
            video => video.video_id === stream.videoId || video.video_id === `${videoInfo.broadcast_id}`
        );

        this._store.dispatch(setIsSeeked(true));

        if (watchedTimes.length === 0) {
            return;
        }
        const { position, type } = watchedTimes[0];
        const duration = playback.duration;
        if (position < duration - CANCEL_RESUME_AMOUNT) {
            if (type === CONTENT_MODE_LIVE) {
                const resumeTime = Math.max(0, position - LIVESTREAM_RESUME_PUSHBACK);
                this._player.setCurrentTime(resumeTime);
            } else {
                this._player.setCurrentTime(position);
            }
        }

        if (watchedTimes.filter(video => video.type === CONTENT_MODE_LIVE).length > 0) {
            this._store.dispatch(cancelLivestreamResumeTime(videoInfo.broadcast_id));
        }
    }

    _seekToResumeTimeLocal(videoInfo) {
        const { resumeWatch, stream, playback } = this._store.getState();
        const VODResumeTime = resumeWatch.times[stream.videoId];
        const streamResumeTime = resumeWatch.streamTimes[videoInfo.broadcast_id];
        const duration = playback.duration;

        this._store.dispatch(setIsSeeked(true));

        if (isNumber(VODResumeTime) && VODResumeTime < duration - CANCEL_RESUME_AMOUNT) {
            this._player.setCurrentTime(VODResumeTime);
        } else if (isNumber(streamResumeTime) && streamResumeTime < duration - CANCEL_RESUME_AMOUNT) {
            const resumeTime = Math.max(0, streamResumeTime - LIVESTREAM_RESUME_PUSHBACK);
            this._player.setCurrentTime(resumeTime);
        }
        this._store.dispatch(cancelLivestreamResumeTime(videoInfo.broadcast_id));
    }

    _onLoadedMetadata() {
        const { stream, resumeWatch, collection, playback } = this._store.getState();
        if (stream.contentType === CONTENT_MODE_LIVE) {
            this._streamTimeOffset = this._player.getVideoInfo().stream_time_offset;
            return;
        }
        if (
            collection.id ||
            !(stream.contentType === CONTENT_MODE_VOD) ||
            resumeWatch.isSeeked ||
            playback.startTimeSet
        ) {
            return;
        }

        const videoInfoRequest = videoInfo(stream.videoId);
        if (!resumeWatch.userId) {
            return videoInfoRequest.then(vi => {
                this._seekToResumeTimeLocal(vi);
            });
        }

        const watchedVODsRequest = getResumeTimes(resumeWatch.userId);
        return Promise.all([videoInfoRequest, watchedVODsRequest]).then(([vi, watchedVODsResp]) => {
            const watchedVODs = watchedVODsResp.videos;
            this._store.dispatch(setResumeTimes(watchedVODs));
            this._seekToResumeTime(vi, watchedVODs);
        });
    }

    _onTimeUpdate() {
        let action;
        const { stream, streamMetadata, onlineStatus, playback } = this._store.getState();
        if (playback.isLoading) {
            return;
        }
        const channelID = streamMetadata.channel.id;
        if (stream.contentType === CONTENT_MODE_LIVE) {
            const broadcastId = streamMetadata.broadcastID;

            if (onlineStatus === ONLINE_STATUS) {
                action = setLivestreamResumeTime(
                    broadcastId,
                    channelID,
                    this._streamTimeOffset + playback.currentTime
                );
            } else {
                action = cancelLivestreamResumeTime(broadcastId, channelID);
            }
        } else if (stream.contentType === CONTENT_MODE_VOD) {
            if (playback.duration === 0) {
                return;
            }
            if ((playback.duration - playback.currentTime) > CANCEL_RESUME_AMOUNT) {
                action = setVodResumeTime(stream.videoId, channelID, playback.currentTime);
            } else {
                action = cancelVodResumeTime(stream.videoId, channelID);
            }
        }

        if (action) {
            this._store.dispatch(action);
        }
    }
}
