import includes from 'lodash/includes';
import EventEmitter from 'event-emitter';
import { TimelineMetadata } from './timeline';
import * as Settings from './settings';
import * as PlayerApi from './api';
import * as MediaEvents from './backend/events/media-event';
import * as TwitchEvents from './backend/events/twitch-event';
import { VODTwitchContentStream } from './stream/twitch-vod';
import { LiveTwitchContentStream } from './stream/twitch-live';
import { nullContentStream } from './stream/null';

// Types of events outputted by state tracker
export const EVENT_STATE_UPDATE = 'stateupdate';
export const EVENT_PLAYER_UPDATE = 'player';

// States of player playback
export const PLAYBACK_PAUSED = 'paused';
export const PLAYBACK_PLAYING = 'playing';
export const PLAYBACK_ENDED = 'ended';

// States of Chromecast
export const CHROMECAST_UNAVAILABLE = 'unavailable';
export const CHROMECAST_AVAILABLE = 'available';
export const CHROMECAST_CONNECTING = 'connecting';
export const CHROMECAST_CONNECTED = 'connected';

// State tracker events available to be listened to
const AVAILABLE_EVENTS = [
    EVENT_STATE_UPDATE,
    EVENT_PLAYER_UPDATE,
];

// Internal logging events
const EVENT_MARKERS_CHANGE = 'markerschange';

/**
 * State
 * Keeps track of the state of the player and passes through backend event.
 * One stop shop for all your state needs!
 * @constructor
 */
export class State {
    constructor(backend, store, analytics, options) {
        this._options = options;
        this._backend = backend;
        this._store = store;
        this._timelineMetadata = new TimelineMetadata(analytics);

        this._eventEmitter = new EventEmitter();

        // Properties not proxied, derived, or retrieved from kraken API
        this._isVODRestricted = false;
        this._hasCurrentStreamPlayed = false;

        this._markers = [];

        // Listen to all events from backend
        Settings.allEvents.forEach(function(event) {
            backend.addEventListener(event, this.handleEvent.bind(this, event));
        }, this);
    }

    // Getter & Setter Functions
    get currentTime() {
        return this._backend.getCurrentTime();
    }

    get error() {
        return this._backend.error;
    }

    getMarkers() {
        return this._markers;
    }

    get playback() {
        if (this._backend.getEnded()) {
            return PLAYBACK_ENDED;
        } else if (this._backend.getPaused()) {
            return PLAYBACK_PAUSED;
        }

        return PLAYBACK_PLAYING;
    }

    get videoID() {
        return this._backend.getVideo();
    }

    isVODRestricted() {
        return this._isVODRestricted;
    }

    // Kraken API Calls - Returns Promise
    // TODO: All of this will be encompassed in our polymorphic object approach to
    // storing video/stream info, instead of separate repeated calls to Player API (kraken calls)

    getVideoInfo(videoID) {
        return PlayerApi.videoInfo(videoID);
    }

    /**
     * Update state of player based on events from backend
     * @private
     */
    handleEvent(event) {
        // TODO: Currently only a small handful states have been implemented. Need MOAR states.

        this._logStateChange(event);

        switch (event) {
        case MediaEvents.LOADED_METADATA:
            this._updateIsVODRestricted();
            break;
        case MediaEvents.LOADSTART:
            this._hasCurrentStreamPlayed = false;
            this._resetTimelineMetadata();
            // Paywalled VODs do not emit EVENT_LOADED_METADATA from backend
            this._updateIsVODRestricted();
            break;
        case MediaEvents.PLAYING:
            if (!this._hasCurrentStreamPlayed && this.videoID) {
                this._retrieveTimelineMetadata();
            }
            this._hasCurrentStreamPlayed = true;
            break;
        case TwitchEvents.USHER_FAIL_ERROR:
            this._updateIsVODRestricted();
        }

        // Pass through backend event
        this._eventEmitter.emit(EVENT_PLAYER_UPDATE, event);

        // Update all listeners with new state
        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
    }

    _resetTimelineMetadata() {
        this._markers = [];
    }

    _retrieveTimelineMetadata() {
    }

    /**
     * Add event listener
     */
    addEventListener(event, callback) {
        if (includes(AVAILABLE_EVENTS, event)) {
            this._eventEmitter.addListener(event, callback);
        }
    }

    /**
     * Remove event listener
     */
    removeEventListener(event, callback) {
        this._eventEmitter.removeListener(event, callback);
    }

    /**
     * ONLY FOR EMBED API
     * Post messaging only accepts regular JS objects.
     * Creates JS object for embed API with the public state properties
     * @return {Object} A javascript object with the public state properties
     */
    toJSON() {
        const currentState = this._store.getState();
        // eslint-disable-next-line new-cap
        const currentStream = currentState.stream || nullContentStream();
        return {
            channelName: currentStream instanceof LiveTwitchContentStream ? currentStream._channelName : '',
            currentTime: this.currentTime,
            duration: currentState.playback.duration,
            muted: currentState.playback.muted,
            playback: this.playback,
            quality: currentState.quality.current,
            qualitiesAvailable: currentState.quality.available,
            videoID: currentStream instanceof VODTwitchContentStream ? currentStream._vodId : '',
            viewers: currentState.viewercount,
            volume: currentState.playback.volume,
        };
    }

    /**
     * Logs state changes.
     */
    _logStateChange(event = null) {
        if (!(this._options.debug && this._options.verbose) ||
            includes(Settings.debugIgnoreEvents, event) ||
            event === null) {
            return;
        }
        /* eslint-disable no-console */
        console.groupCollapsed('state change: %s', event);
        console.log('partial state: %o', this.toJSON());
        console.groupEnd();
        /* eslint-enable no-console */
    }

    _updateMarkers(markers) {
        this._markers = markers;
        this._logStateChange(EVENT_MARKERS_CHANGE);
        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
    }

    _updateIsVODRestricted() {
        this._isVODRestricted = false;
        this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
        if (!(this._store.getState().stream instanceof VODTwitchContentStream)) {
            return;
        }
        const videoInfo  = PlayerApi.videoInfo(this._store.getState().stream._vodId);
        const viewerInfo = videoInfo.then(info => PlayerApi.channelViewerInfo(info.channel.name));
        Promise.all([videoInfo, viewerInfo]).then(([videoInfo, viewerInfo]) => {
            this._isVODRestricted = PlayerApi.isVODRestricted(viewerInfo, videoInfo);
            this._eventEmitter.emit(EVENT_STATE_UPDATE, this);
        });
    }

    destroy() {
        AVAILABLE_EVENTS.map(event => {
            this._eventEmitter.removeEvent(event);
        });
    }
}
